// LinkLable.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _LINK_LABEL_H_
#define _LINK_LABEL_H_

#include "Window.h"
#include "ToolTipCtrl.h"
#include <stdexcept>

/* available window style */
#define LLS_LEFT		0x0000L
#define LLS_CENTER		0x0001L
#define LLS_RIGHT		0x0002L
#define LLS_TOP			0x0000L
#define LLS_VCENTER		0x0010L
#define LLS_BOTTOM		0x0020L
#define LLS_SINGLELINE	0x0040L
#define LLS_WORDBREAK	0x0080L


// CLinkLabel class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace Controls {

class CLinkLabel : public CWindow {
	// RXgN^
public:
	CLinkLabel();
	CLinkLabel(const TCHAR* lpszCaption, const TCHAR* lpszAlternation);
	virtual ~CLinkLabel();
private:
	CLinkLabel(const CLinkLabel& rhs);

	// \bh
public:
	bool			Create(HWND hwndParent, HINSTANCE hInstance, UINT nIDCtrl);
	const TCHAR*	GetAlternationText() const;
	void			SetAlternationText(const TCHAR* lpszText);
protected:
	LRESULT			DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam);
	static LRESULT CALLBACK	WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);

	// bZ[Wnh
protected:
	virtual void	OnDestroy();								// WM_DESTROY
	virtual void	OnKillFocus(HWND hwndNew);					// WM_KILLFOCUS
	virtual void	OnLButtonDown(UINT nFlags, POINT pt);		// WM_LBUTTONDOWN
	virtual void	OnLButtonUp(UINT nFlags, POINT pt);			// WM_LBUTTONUP
	virtual void	OnMouseMove(UINT nFlags, POINT pt);			// WM_MOUSEMOVE
	virtual void	OnPaint();									// WM_PAINT
	virtual bool	OnSetCursor(UINT nHitTest, UINT message);	// WM_SETCURSOR
	virtual void	OnSetFocus(HWND hwndOld);					// WM_SETFOCUS
	virtual void	OnSetText(const TCHAR* lpszText);			// WM_SETTEXT
	virtual void	OnSysColorChange();							// WM_SYSCOLORCHANGE

	// f[^o
public:
	HCURSOR		m_hCursor;			// cursor as overt label
	COLORREF	m_clrNormal;		// normal text color
	COLORREF	m_clrHover;			// hover text color
	COLORREF	m_clrBackground;	// background color(if first bit is set, use TRANSPARENT mode)
	bool		m_bUnderlineAlways;	// draw with underline when the cursor is not over a label
	LONG		m_lStyle;			// drawing style
private:
	TCHAR*		m_pszAlternation;	// alt text(tooltip text)
	bool		m_bHover;			// now hover to true
	HFONT		m_hNormalFont;		// no underline font
	HFONT		m_hUnderlineFont;	// underline font
};

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */


// CLinkLabel class implementation
/////////////////////////////////////////////////////////////////////////////

using Manah::Windows::Controls::CLinkLabel;

inline CLinkLabel::CLinkLabel() : m_pszAlternation(0), m_bHover(false),
		m_hNormalFont(0), m_hUnderlineFont(0), m_lStyle(DT_LEFT) {
	m_hCursor = 0;
	m_clrNormal = ::GetSysColor(26);	// COLOR_HOTLIGHT
	m_clrHover = ::GetSysColor(26);
	m_clrBackground = 0x80000000;
	m_bUnderlineAlways = true;
}

inline CLinkLabel::CLinkLabel(const TCHAR* lpszCaption, const TCHAR* lpszAlternation /* = 0 */) {
	CLinkLabel();
	SetWindowText(lpszCaption);
	if(lpszAlternation != 0)
		SetAlternationText(lpszAlternation);
}

inline CLinkLabel::~CLinkLabel() {
	delete[] m_pszAlternation;
}

inline bool CLinkLabel::Create(HWND hwndParent, HINSTANCE hInstance, UINT nIDCtrl) {
	AssertValid();
	assert(hwndParent == 0 || ::IsWindow(hwndParent));

	static bool	bRegistered = false;
	HFONT		hParentFont;
	LOGFONT		lf;
	
	if(!bRegistered) {
		WNDCLASSEXW	wc;

		wc.cbSize			= sizeof(WNDCLASSEX);
		wc.style			= CS_BYTEALIGNCLIENT | CS_BYTEALIGNWINDOW | CS_DBLCLKS;
		wc.lpfnWndProc		= reinterpret_cast<WNDPROC>(CLinkLabel::WndProc);
		wc.cbClsExtra		= 0;
		wc.cbWndExtra		= sizeof(long);
		wc.hInstance		= ::GetModuleHandle(0);
		wc.hIcon			= 0;
		wc.hIconSm			= 0;
		wc.hCursor			= ::LoadCursor(0, MAKEINTRESOURCE(32649));
		wc.hbrBackground	= 0;
		wc.lpszClassName	= L"LinkLabel";
		wc.lpszMenuName		= 0;

		if(!::RegisterClassExW(&wc))
			throw std::runtime_error("Failed to register window class!");
		bRegistered = true;
	}

	m_hWnd = ::CreateWindow(_T("LinkLabel"), _T(""), WS_CHILD | WS_TABSTOP | WS_VISIBLE,
		CW_USEDEFAULT ,CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
		hwndParent, reinterpret_cast<HMENU>(nIDCtrl), hInstance, reinterpret_cast<void*>(this));
	if(m_hWnd == 0)
		return false;

	hParentFont = reinterpret_cast<HFONT>(::SendMessage(hwndParent, WM_GETFONT, 0, 0L));
	::GetObject(hParentFont, sizeof(LOGFONT), &lf);
	lf.lfUnderline = true;
	m_hUnderlineFont = ::CreateFontIndirect(&lf);
	lf.lfUnderline = CLinkLabel::m_bUnderlineAlways;
	m_hNormalFont = ::CreateFontIndirect(&lf);

	return true;
}

inline LRESULT CLinkLabel::DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam) {
	AssertValid();

	if(PreTranslateMessage(message, wParam, lParam))
		return false;

	static POINT	pt = {-1, -1};

	switch(message) {
/*	case BM_SETSTATE:
		if(wParam == FALSE) {
			OnLButtonDown(0, pt);
			OnLButtonUp(0, pt);
		}
		break;
	case WM_GETDLGCODE:
		return DLGC_BUTTON | DLGC_UNDEFPUSHBUTTON;*/
	case WM_KEYDOWN:
		if(wParam == VK_RETURN)
			GetParent()->SendMessage(WM_COMMAND, GetWindowLong(GWL_ID), reinterpret_cast<LPARAM>(m_hWnd));
		break;
	case WM_SETTEXT:
		OnSetText(reinterpret_cast<TCHAR*>(lParam));
		break;
	}

	return CWindow::DispatchEvent(message, wParam, lParam);
}

inline const TCHAR* CLinkLabel::GetAlternationText() const {
	AssertValid();
	return m_pszAlternation;
}

inline void CLinkLabel::SetAlternationText(const TCHAR* lpszText) {
	AssertValid();
	assert(lpszText != 0);

	if(m_pszAlternation != 0)
		delete[] m_pszAlternation;
	m_pszAlternation = new TCHAR[_tcslen(lpszText) + 1];
	_tcscpy(m_pszAlternation, lpszText);
}

inline LRESULT CALLBACK CLinkLabel::WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam) {
	CLinkLabel*	pLinkLabel = 0;
	if(message == WM_CREATE) {
		pLinkLabel = reinterpret_cast<CLinkLabel*>(reinterpret_cast<CREATESTRUCTW*>(lParam)->lpCreateParams);
		pLinkLabel->m_hWnd = hWnd;
		pLinkLabel->SetWindowLong(
			GWL_USERDATA, reinterpret_cast<long>(reinterpret_cast<CREATESTRUCTW*>(lParam)->lpCreateParams));
		return pLinkLabel->DispatchEvent(message, wParam, lParam);
	} else {
		pLinkLabel = reinterpret_cast<CLinkLabel*>(::GetWindowLong(hWnd, GWL_USERDATA));
		return pLinkLabel ? pLinkLabel->DispatchEvent(message, wParam, lParam) : ::DefWindowProc(hWnd, message, wParam, lParam);
	}
}

inline void CLinkLabel::OnDestroy() {
	::DeleteObject(m_hNormalFont);
	::DeleteObject(m_hUnderlineFont);
}

inline void CLinkLabel::OnKillFocus(HWND hwndNew) {
	HDC		hDC;
	RECT	rect;

	GetClientRect(&rect);
	::OffsetRect(&rect, -rect.left, -rect.top);
	hDC = GetDC();
	::DrawFocusRect(hDC, &rect);
	ReleaseDC(hDC);
}

inline void CLinkLabel::OnLButtonDown(UINT nFlags, POINT pt) {
	SetFocus();
}

inline void CLinkLabel::OnLButtonUp(UINT nFlags, POINT pt) {
	GetParent()->SendMessage(WM_COMMAND, GetWindowLong(GWL_ID), reinterpret_cast<LPARAM>(m_hWnd));
}

inline void CLinkLabel::OnMouseMove(UINT nFlags, POINT pt) {
	RECT	rect;
	bool	bHoverd = m_bHover;

	GetWindowRect(&rect);
	::ClientToScreen(m_hWnd, &pt);
	if(!::PtInRect(&rect,pt)) {
		m_bHover = false;
		ReleaseCapture();
	} else
		m_bHover = true;

	if(m_bHover != bHoverd)
		InvalidateRect(0, false);
}

inline void CLinkLabel::OnPaint() {
	HDC			hDC;
	PAINTSTRUCT	paint;
	HFONT		hOldFont;
	RECT		rect;
	size_t		cch = GetWindowTextLength();
	TCHAR*		psz = new TCHAR[cch + 1];

	GetWindowText(psz, cch + 1);
	hDC = ::BeginPaint(m_hWnd, &paint);

	if(m_bHover) {	// hover
		hOldFont = reinterpret_cast<HFONT>(::SelectObject(hDC, m_hUnderlineFont));
		::SetTextColor(hDC, CLinkLabel::m_clrHover);
	} else {	// normal
		hOldFont = reinterpret_cast<HFONT>(::SelectObject(hDC, m_hNormalFont));
		::SetTextColor(hDC, CLinkLabel::m_clrNormal);
	}
	::SetBkColor(hDC, CLinkLabel::m_clrBackground);
	::SetBkMode(hDC, (CLinkLabel::m_clrBackground & 0x80000000) ? TRANSPARENT : OPAQUE);
	GetWindowRect(&rect);
	::OffsetRect(&rect, -rect.left, -rect.top);
	::InflateRect(&rect, -1, -1);
	::DrawText(
		hDC, psz, cch, &rect,
		(m_lStyle & LLS_LEFT) ? DT_LEFT : 0
		| (m_lStyle & LLS_CENTER) ? DT_CENTER : 0
		| (m_lStyle & LLS_RIGHT) ? DT_RIGHT : 0
		| (m_lStyle & LLS_TOP) ? DT_TOP : 0
		| (m_lStyle & LLS_VCENTER) ? DT_VCENTER : 0
		| (m_lStyle & LLS_BOTTOM) ? DT_BOTTOM : 0
		| (m_lStyle & LLS_SINGLELINE) ? DT_SINGLELINE : 0
		| (m_lStyle & LLS_WORDBREAK) ? DT_WORDBREAK : 0);
	delete[] psz;

	::SelectObject(hDC, hOldFont);
	::EndPaint(m_hWnd, &paint);
}

inline bool CLinkLabel::OnSetCursor(UINT nHitTest, UINT message) {
	if(m_hCursor != 0)
		SetCursor(m_hCursor);
	if(!m_bHover)
		SetCapture();

	return true;
}

inline void CLinkLabel::OnSetFocus(HWND hwndOld) {
	HDC		hDC;
	RECT	rect;

	GetClientRect(&rect);
	::OffsetRect(&rect, -rect.left, -rect.top);
	hDC = GetDC();
	::DrawFocusRect(hDC, &rect);
	ReleaseDC(hDC);
}

inline void CLinkLabel::OnSetText(const TCHAR* lpszText) {
	HDC		hDC = GetDC();
	RECT	rect = {0, 0, 0, 0};
	HFONT	hOldFont;

	hOldFont = reinterpret_cast<HFONT>(::SelectObject(hDC, m_hNormalFont));
	::DrawText(hDC, lpszText, -1, &rect, DT_CALCRECT);
	::SelectObject(hDC, hOldFont);
	ReleaseDC(hDC);
	rect.right += 2;
	rect.bottom += 2;
	SetWindowPos(0, &rect, SWP_NOMOVE | SWP_NOZORDER);
}

inline void CLinkLabel::OnSysColorChange() {
	CLinkLabel::m_clrNormal = ::GetSysColor(26);	// COLOR_HOTLIGHT
	CLinkLabel::m_clrHover = ::GetSysColor(26);
}

#endif /* _LINK_LABEL_H_ */

/* [EOF] */