// MenuOperator.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _MENU_OPERATOR_H_
#define _MENU_OPERATOR_H_

#pragma warning(disable : 4297)

#include "..\Manah\Object.h"
#include <commctrl.h>
#include <stack>
#include <vector>
#include <map>
using namespace std;


namespace Manah {
namespace Windows {
namespace Controls {

class CMenuItem;
class CPopupMenu;
class CMenuOperator;


// CMenuItem class definition
/////////////////////////////////////////////////////////////////////////////

class CMenuItem : public CObject {
	friend class CPopupMenu;
	friend class CMenuOperator;

	// RXgN^
private:
	CMenuItem() : m_pPopupMenu(0)	{}
	virtual ~CMenuItem();

	// \bh
public:
	TCHAR			GetAccelerator() const;
	UINT			GetAttribute() const;
	const TCHAR*	GetCaption() const;
	UINT			GetIconIndex() const;
	UINT			GetID() const;
	CPopupMenu*		GetSubMenu() const;

	// f[^o
private:
	UINT		m_nID;				// item's id (0 if item is popup)
	UINT		m_nAttributes;		// item's attributes
	TCHAR		m_szCaption[128];	// item's caption (includes first and seconds accelerators)
	UINT		m_iImage;			// index of item's icon
	CPopupMenu*	m_pPopupMenu;		// available pointer to CPopup when item is popup
};


// CPopupMenu class definition
/////////////////////////////////////////////////////////////////////////////

typedef map<HMENU, CPopupMenu*>	PopupMap;
static PopupMap	g_mapPopups;	// all existing popups

class CPopupMenu : public Manah::CObject {
	friend class CMenuItem;
	friend class CMenuOperator;

	// RXgN^
public:
	CPopupMenu();
	virtual ~CPopupMenu();

	// \bh
public:
	bool		AppendChildPopup(const CPopupMenu* pPopup, UINT iItem);
	CPopupMenu*	AppendMenuItem(UINT nID,
					UINT nAttributes, const TCHAR* lpszCaption, UINT iImage = -1) throw();
	DWORD		CheckMenuItem(UINT nID, bool bCheck = true);
	bool		CheckMenuRadioItem(UINT nFirstID, UINT nLastID, UINT nItemID);
	bool		DeleteMenuItem(UINT iItem, bool bByCommand = true);
	bool		EnableMenuItem(UINT nID, bool bEnable = true);
	UINT		GetDefaultMenuItem(UINT gmdiFlags) const;
	UINT		GetItemCount() const;
	CMenuItem*	GetMenuItem(UINT iItem, bool bByCommand = true) const;
	UINT		GetMenuItemState(UINT nID, UINT nFlags) const;
	HMENU		GetSafeHmenu() const;
	CPopupMenu*	GetSubMenu(UINT iItem) const;
	CPopupMenu*	GetSubMenuFromHandle(HMENU);
	CPopupMenu*	InsertMenuItem(UINT nID, UINT nPrevID, UINT nAttributes,
					const TCHAR* lpszCaption, UINT iImage = -1) throw();
	bool		ModifyMenuItem(UINT nID, UINT nFlags, const TCHAR* lpszPrompt);
	bool		RemoveMenuItem(UINT iItem, CPopupMenu* pRemoved);
	void		ResetContent();
	bool		SetDefaultMenuItem(UINT nID);
	bool		TrackPopupMenu(UINT nFlags, int x, int y, HWND hWnd, const RECT* prcRect = 0);

	// f[^o
private:
	HMENU				m_hMenu;		// handle for popup menu
	vector<CMenuItem*>	m_vecItems;		// children
};


// CMenuOperator class definition
/////////////////////////////////////////////////////////////////////////////

class CMenuOperator : public CPopupMenu {
	// RXgN^
private:
	CMenuOperator();
public:
	CMenuOperator(HWND hWnd);
	~CMenuOperator()	{}

	// \bh
public:
	LRESULT				ExecuteMenuChar(TCHAR chCode, UINT nFlag, HMENU hMenu);	// operate WM_MENUCHAR
	const CPopupMenu*	FindPopupMenu(HMENU hMenu);								// find popup menu in all
	const CMenuItem*	FindMenuItem(UINT nID);									// find menu item in all

	// f[^o
private:
	HMENU				m_hTopMenu;		// application menu
	vector<CPopupMenu*>	m_vecPopups;	// children
	HIMAGELIST			m_hImageList;	// imagelist for drawing icons
};

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */


// CMenuItem class implementation
/////////////////////////////////////////////////////////////////////////////

using Manah::Windows::Controls::CMenuItem;
using Manah::Windows::Controls::CPopupMenu;
using Manah::Windows::Controls::CMenuOperator;

inline CMenuItem::~CMenuItem() {
	if(m_pPopupMenu != 0)
		delete m_pPopupMenu;
}

inline TCHAR CMenuItem::GetAccelerator() const {
	AssertValid();

	TCHAR*	pszAccel;

	pszAccel = const_cast<TCHAR*>(_tcschr(m_szCaption, _T('&')));

	return (pszAccel != 0) ? *(pszAccel + 1) : 0;
}

inline UINT CMenuItem::GetAttribute() const {
	AssertValid();
	return m_nAttributes;
}

inline const TCHAR* CMenuItem::GetCaption() const {
	AssertValid();
	return m_szCaption;
}

inline UINT CMenuItem::GetIconIndex() const
{
	AssertValid();

	return m_iImage;
}

inline UINT CMenuItem::GetID() const {
	AssertValid();
	return m_nID;
}

inline CPopupMenu* CMenuItem::GetSubMenu() const {
	AssertValid();
	return m_pPopupMenu;
}


// CPopupMenu class implementation
/////////////////////////////////////////////////////////////////////////////

inline CPopupMenu::CPopupMenu() {
	m_hMenu = ::CreatePopupMenu();
	g_mapPopups[m_hMenu] = this;
}

inline CPopupMenu::~CPopupMenu() {
	g_mapPopups.erase(m_hMenu);
	::DestroyMenu(m_hMenu);
	for(vector<CMenuItem*>::iterator i = m_vecItems.begin(); i != m_vecItems.end(); ++i)
		delete *i;
}

inline bool CPopupMenu::AppendChildPopup(const CPopupMenu* pPopup, UINT iItem) {
	AssertValid();

	MENUITEMINFO	mii;

	ZeroMemory(&mii, sizeof(MENUITEMINFO));
	mii.cbSize = sizeof(MENUITEMINFO);
	mii.fMask = MIIM_DATA | MIIM_SUBMENU;
	if(!::GetMenuItemInfo(m_hMenu, iItem, TRUE, &mii))
		return false;
	mii.fMask |= MIIM_SUBMENU;
	mii.hSubMenu = pPopup->m_hMenu;

	return toBoolean(::SetMenuItemInfo(m_hMenu, iItem, true, &mii));
}

inline CPopupMenu* CPopupMenu::AppendMenuItem(
	UINT nID, UINT nAttributes, const TCHAR* lpszCaption,
	UINT iImage /* = -1 */) throw(invalid_argument) {
	AssertValid();

	CMenuItem*	pNewItem;		// new menu item be added
	CPopupMenu*	pNewPopup = 0;	// creates instance if pNewItem is popup
	MENUITEMINFO	mii;

	pNewItem = new CMenuItem();
	if(lpszCaption != 0)
		_tcscpy(pNewItem->m_szCaption, lpszCaption);
	else
		*pNewItem->m_szCaption = 0;

	if(!::AppendMenu(m_hMenu, nAttributes, (nAttributes & MF_POPUP) ? 0 : nID, pNewItem->m_szCaption)) {
		DWORD dw = ::GetLastError();
		delete pNewItem;
		throw invalid_argument("First or second argument is invalid!");
	}

	if(nAttributes & MF_POPUP) {
		pNewPopup = new CPopupMenu();
		ZeroMemory(&mii, sizeof(MENUITEMINFO));
		mii.cbSize = sizeof(MENUITEMINFO);
		mii.fMask = MIIM_SUBMENU;
		mii.hSubMenu = pNewPopup->m_hMenu;
		::SetMenuItemInfo(m_hMenu, nID, true, &mii);
	}

	pNewItem->m_nID = (nAttributes & MF_POPUP) ? 0 : nID;
	pNewItem->m_nAttributes = nAttributes;
	pNewItem->m_iImage = iImage;
	pNewItem->m_pPopupMenu = pNewPopup;
	m_vecItems.push_back(pNewItem);

	return pNewItem->m_pPopupMenu;
}

inline DWORD CPopupMenu::CheckMenuItem(UINT nID, bool bCheck /* = true */) {
	AssertValid();
	return ::CheckMenuItem(m_hMenu, nID, bCheck ? MFS_CHECKED : MFS_UNCHECKED);
}

inline bool CPopupMenu::CheckMenuRadioItem(UINT nFirstID, UINT nLastID, UINT nItemID) {
	AssertValid();
	return toBoolean(::CheckMenuRadioItem(m_hMenu, nFirstID, nLastID, nItemID, MF_BYCOMMAND));
}

inline bool CPopupMenu::DeleteMenuItem(UINT iItem, bool bByCommand /* = true */) {
	AssertValid();

	vector<CMenuItem*>::iterator	i;
	UINT							n;

	if(bByCommand) {
		for(i = m_vecItems.begin(); i != m_vecItems.end(); i++) {
			if(reinterpret_cast<CMenuItem*>(*i)->m_nID == iItem) {
				m_vecItems.erase(i);
				return toBoolean(::DeleteMenu(m_hMenu, iItem, MF_BYCOMMAND));
			}
		}
	} else {
		for(i = m_vecItems.begin(), n = 0; i != m_vecItems.end(); i++, n++) {
			if(n == iItem) {
				m_vecItems.erase(i);
				return toBoolean(::DeleteMenu(m_hMenu, iItem, MF_BYPOSITION));
			}
		}
	}

	return false;
}

inline bool CPopupMenu::EnableMenuItem(UINT nID, bool bEnable /* = true */) {
	AssertValid();
	return toBoolean(::EnableMenuItem(m_hMenu, nID, MF_BYCOMMAND | (bEnable ? MF_ENABLED : MF_GRAYED)));
}

inline UINT CPopupMenu::GetDefaultMenuItem(UINT gmdiFlags) const {
	AssertValid();
	return ::GetMenuDefaultItem(m_hMenu, false, gmdiFlags);
}

inline UINT CPopupMenu::GetItemCount() const {
	AssertValid();
	return m_vecItems.size();
}

inline CMenuItem* CPopupMenu::GetMenuItem(UINT iItem, bool bByCommand /* = true */) const {
	AssertValid();

	UINT nSize;

	nSize = m_vecItems.size();
	if(bByCommand) {	// use command id
		for(UINT i = 0; i < nSize; i++) {
			if(iItem == m_vecItems[i]->m_nID)
				return m_vecItems[i];
		}
	} else {	// use index
		if(iItem < nSize)
			return m_vecItems[iItem];
	}

	return 0;
}

inline UINT CPopupMenu::GetMenuItemState(UINT nID, UINT nFlags) const {
	AssertValid();
	return ::GetMenuState(m_hMenu, nID, nFlags);
}

inline HMENU CPopupMenu::GetSafeHmenu() const {
	return (this != 0) ? m_hMenu : 0;
}

inline CPopupMenu* CPopupMenu::GetSubMenu(UINT iItem) const {
	AssertValid();

	try {
		return m_vecItems.at(iItem)->m_pPopupMenu;
	} catch(...) {
		return 0;
	}
}

inline CPopupMenu* CPopupMenu::GetSubMenuFromHandle(HMENU hMenu) {
	AssertValid();

	CPopupMenu*		pTarget;	// popup check currently
	unsigned int	cItems;		// count of pTarget children

	// search all items
	pTarget = this;
	cItems = pTarget->m_vecItems.size();
	for(unsigned int iItem = 0; iItem < cItems; ++iItem) {
		if(pTarget->m_vecItems[iItem]->m_nAttributes & MF_POPUP) {
			if(pTarget->m_vecItems[iItem]->m_pPopupMenu->m_hMenu == hMenu)
				return pTarget->m_vecItems[iItem]->m_pPopupMenu;
			CPopupMenu*	pFound = pTarget->m_vecItems[iItem]->m_pPopupMenu->GetSubMenuFromHandle(hMenu);
			if(pFound != 0)
				return pFound;
		}
	}

	return 0;
}

inline CPopupMenu* CPopupMenu::InsertMenuItem(UINT nID, UINT nPrevID,
		UINT nAttributes, const TCHAR* lpszCaption, UINT iImage /* = -1 */) throw(invalid_argument) {
	AssertValid();

	CMenuItem*	pNewItem;		// new menu item be added
	CPopupMenu*	pNewPopup = 0;	// creates instance if pNewItem is popup
	MENUITEMINFO	mii;

	pNewItem = new CMenuItem();
	if(lpszCaption != 0)
		_tcscpy(pNewItem->m_szCaption, lpszCaption);
	else
		*pNewItem->m_szCaption = 0;

	ZeroMemory(&mii, sizeof(MENUITEMINFO));
	mii.cbSize = sizeof(MENUITEMINFO);
	mii.fMask = MIIM_ID | MIIM_TYPE;
	mii.fType = nAttributes;
	mii.wID = nID;
	mii.dwItemData = reinterpret_cast<LPARAM>(pNewItem->m_szCaption);
	if(nAttributes & MF_POPUP) {
		pNewPopup = new CPopupMenu();
		mii.fMask |= MIIM_SUBMENU;
		mii.hSubMenu = pNewPopup->m_hMenu;
	}
	if(!::InsertMenuItem(m_hMenu, nPrevID, false, &mii)) {
		delete pNewItem;
		throw invalid_argument("First or third argument is invalid!");
	}

	pNewItem->m_nID = (nAttributes & MF_POPUP) ? 0 : nID;
	pNewItem->m_nAttributes = nAttributes;
	pNewItem->m_iImage = iImage;
	pNewItem->m_pPopupMenu = pNewPopup;
	m_vecItems.push_back(pNewItem);

	return pNewItem->m_pPopupMenu;
}

inline bool CPopupMenu::ModifyMenuItem(UINT nID, UINT nFlags, const TCHAR* lpszPrompt) {
	AssertValid();
	assert(!(nFlags & MF_POPUP));

	if(!::ModifyMenu(m_hMenu, nID, MF_BYCOMMAND | nFlags, nID, lpszPrompt))
		return false;
	_tcscpy(GetMenuItem(nID)->m_szCaption, lpszPrompt);
	return true;
}

inline bool CPopupMenu::RemoveMenuItem(UINT iItem, CPopupMenu* pRemoved) {
	AssertValid();
	assert(pRemoved != 0);

	if(!::RemoveMenu(m_hMenu, iItem, MF_BYPOSITION))
		return false;
	pRemoved = GetSubMenu(iItem);
	m_vecItems.erase(m_vecItems.begin() + iItem);
	return true;
}

inline void CPopupMenu::ResetContent() {
	AssertValid();

	for(size_t iItem = m_vecItems.size(); iItem != 0; --iItem)
		DeleteMenuItem(iItem - 1, false);
	m_vecItems.clear();
}

inline bool CPopupMenu::TrackPopupMenu(
	UINT nFlags, int x, int y, HWND hWnd, const RECT* prcRect /* = 0 */) {
	AssertValid();
	return toBoolean(::TrackPopupMenu(m_hMenu, nFlags, x, y, 0, hWnd, prcRect));
}

inline bool CPopupMenu::SetDefaultMenuItem(UINT nID) {
	AssertValid();
	return toBoolean(::SetMenuDefaultItem(m_hMenu, nID, false));
}


// CMenuOperator class implementation
/////////////////////////////////////////////////////////////////////////////

inline CMenuOperator::CMenuOperator(HWND hWnd) {
	m_hMenu = ::GetMenu(hWnd);
	::DeleteMenu(m_hMenu, 0, MF_BYPOSITION);
	m_vecItems.clear();
}

inline LRESULT CMenuOperator::ExecuteMenuChar(TCHAR chCode, UINT nFlag, HMENU hMenu) {
	AssertValid();

	const CPopupMenu*	pActiveMenu;
	UINT				cMenuItems;

	if(chCode >= _T('a') && chCode <= _T('z'))	// make upper
		chCode -= 0x20;

	pActiveMenu = GetSubMenuFromHandle(hMenu);
	if(pActiveMenu == 0) {
		pActiveMenu = CMenuOperator::FindPopupMenu(hMenu);
		if(pActiveMenu == 0)
			return MNC_IGNORE;
	}

	cMenuItems = pActiveMenu->GetItemCount();
	for(unsigned int iItem = 0; iItem < cMenuItems; ++iItem){
		if(pActiveMenu->m_vecItems[iItem]->GetAccelerator() == chCode)
			return (iItem | 0x00020000);
	}
	
	return MNC_IGNORE;
}

inline const CMenuItem* CMenuOperator::FindMenuItem(UINT nID) {
	AssertValid();

	CPopupMenu*			pTarget;
	UINT				nCount;
	stack<CPopupMenu*>	objPopups;

	// search all popups
	pTarget = this;
	while(true) {
		nCount = pTarget->m_vecItems.size();
		for(UINT i = 0; i < nCount; i++){
			if(nID == pTarget->m_vecItems[i]->m_nID)
				return pTarget->m_vecItems[i];
			else if(pTarget->m_vecItems[i]->m_nAttributes & MF_POPUP)
				objPopups.push(pTarget->m_vecItems[i]->m_pPopupMenu);
		}
		if(!objPopups.empty()) {
			pTarget = objPopups.top();
			objPopups.pop();
		} else
			return 0;
	}
}

inline const CPopupMenu* CMenuOperator::FindPopupMenu(HMENU hMenu) {
	AssertValid();

	PopupMap::const_iterator	it = g_mapPopups.find(hMenu);
	return (it != g_mapPopups.end()) ? it->second : 0;
}

#pragma warning(default : 4297)

#endif /* _MENU_OPERATOR_H_ */

/* [EOF] */