package tk.eclipse.plugin.visualjsf;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jdt.core.IField;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.JavaUI;

import tk.eclipse.plugin.htmleditor.HTMLUtil;

/**
 * Provides utility methods for VisualJSF.
 * 
 * @author Naoki Takezoe
 */
public class VisualJSFUtil {
	
	/**
	 * Removes the field and accessor methods.
	 * 
	 * @param clazz IType instance of the target class
	 * @param name the property name
	 */
	public static void removeProperty(IType clazz, String name){
		if(clazz==null){
			return;
		}
		try {
			IField field = clazz.getField(name);
			if(field!=null && field.exists()){
				IMethod[] methods = clazz.getMethods();
				String setter = HTMLUtil.getSetterName(name);
				String getter = HTMLUtil.getGetterName(name, false);
				for(int i=0;i<methods.length;i++){
					if(methods[i].getElementName().equals(setter)){
						methods[i].delete(true, new NullProgressMonitor());
					} else if(methods[i].getElementName().equals(getter)){
						methods[i].delete(true, new NullProgressMonitor());
					}
				}
				field.delete(true, new NullProgressMonitor());
			}
		} catch(Exception ex){
			VisualJSFPlugin.logException(ex);
		}
	}
	
	/**
	 * Renames the field and accessor methods.
	 * 
	 * @param type IType instance of the target class
	 * @param from the old mame
	 * @param to the new name
	 * @param type the field class name
	 * @throws Exception
	 */
	public static void renameProperty(IType clazz, String from, String to, String type){
		if(type==null){
			return;
		}
		removeProperty(clazz, from);
		createProperty(clazz, to, type);

//		try {
//			IField field = type.getField(from);
//			if(field!=null && field.exists()){
//				RenameSupport support = RenameSupport.create(field, to, 
//						RenameSupport.UPDATE_GETTER_METHOD|
//						RenameSupport.UPDATE_SETTER_METHOD|
//						RenameSupport.UPDATE_REFERENCES);
//				
//				IWorkbench workbench = PlatformUI.getWorkbench(); 
//				IWorkbenchWindow window = workbench.getActiveWorkbenchWindow(); 
//				
//				support.perform(window.getShell(), window);
//			}
//		} catch(Exception ex){
//			VisualJSFPlugin.logException(ex);
//		}
	}
	
	/**
	 * Returns the managed-bean name.
	 * 
	 * @param pageName the page path
	 * @return the managed-bean name
	 */
	public static String getBackingBeanName(String pageName){
		String beanName = pageName.substring(0, pageName.lastIndexOf('.'));
		if(beanName.startsWith("/")){
			beanName = beanName.substring(1);
		}
		beanName = beanName.replaceAll("/|\\\\|\\.", "_");
		return beanName;
	}
	
	/**
	 * Returns the managed-bean class name.
	 * 
	 * @param pageName the page path
	 * @return the managed-bean class name
	 */
	public static String getBackingBeanClassName(String pageName){
		String className = pageName.substring(0, pageName.lastIndexOf('.'));
		if(className.startsWith("/")){
			className = className.substring(1);
		}
		className = className.replaceAll("/|\\\\|\\.", "_");
		className = className.substring(0, 1).toUpperCase() + 
		            className.substring(1);
		
		return className;
	}
	
	/**
	 * Generates the property and accessor methods.
	 * 
	 * @param clazz IType instance of the target class
	 * @param name the property name
	 * @param type the property type
	 * @throws JavaModelException
	 */
	public static void createProperty(IType clazz, String name, String type){
		if(clazz==null){
			return;
		}
		try {
			// add field
			clazz.createField("private " + type + " " + name + ";",
					null, false, new NullProgressMonitor());
			// add setter method
			clazz.createMethod(
					"public void " + HTMLUtil.getSetterName(name) + "(" + type + " " + name + "){\n" +
					"\tthis." + name + " = " + name + ";\n"+
					"}\n", null, false, new NullProgressMonitor());
			// add getter method
			clazz.createMethod(
					"public " + type + " " + HTMLUtil.getGetterName(name, false) + "(){\n" +
					"\treturn this." + name + ";\n"+
					"}\n", null, false, new NullProgressMonitor());
			
		} catch(Exception ex){
			VisualJSFPlugin.logException(ex);
		}
	}
	
	/**
	 * Generates the action method.
	 * 
	 * @param clazz IType instance of the target class
	 * @param name the method name
	 * @param value the result value
	 * @throws JavaModelException
	 */
	public static void createActionMethod(IType clazz, String name, String value){
		if(clazz==null){
			return;
		}
		try {
			// add action method
			if(value==null){
				clazz.createMethod(
						"public String " + name + "(){\n" +
						"\treturn null;\n" + 
						"}\n", null, false, new NullProgressMonitor());
			} else {
				clazz.createMethod(
						"public String " + name + "(){\n" +
						"\treturn \"" + value + "\";\n" + 
						"}\n", null, false, new NullProgressMonitor());
			}
			
		} catch(Exception ex){
			VisualJSFPlugin.logException(ex);
		}
	}
	
	/**
	 * Generates the actionListener method.
	 * 
	 * @param clazz IType instance of the target class
	 * @param name the method name
	 * @throws JavaModelException
	 */
	public static void createActionListenerMethod(IType clazz, String name){
		if(clazz==null){
			return;
		}
		try {
			// add actionListener method
			clazz.createMethod(
					"public void " + name + "(javax.faces.event.ActionEvent evt){\n" +
					"}\n", null, false, new NullProgressMonitor());
			
		} catch(Exception ex){
			VisualJSFPlugin.logException(ex);
		}
	}
	
	/**
	 * Generates the valueChangeListener method.
	 * 
	 * @param clazz IType instance of the target class
	 * @param name the method name
	 * @throws JavaModelException
	 */
	public static void createValueChangeMethod(IType clazz, String name){
		if(clazz==null){
			return;
		}
		try {
			// add valueChangeListener method
			clazz.createMethod(
					"public void " + name + "(javax.faces.event.ValueChangeEvent evt){\n" +
					"}\n", null, false, new NullProgressMonitor());
			
		} catch(Exception ex){
			VisualJSFPlugin.logException(ex);
		}
	}
	
	public static void openJavaElement(IJavaElement element){
		try {
			JavaUI.revealInEditor(JavaUI.openInEditor(element), (IJavaElement)element);
		} catch(Exception ex){
			VisualJSFPlugin.logException(ex);
		}
	}
	
}
