package net.sf.amateras.sastruts;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathFactory;

import jp.aonir.fuzzyxml.FuzzyXMLElement;
import net.sf.amateras.sastruts.bean.PropertyInfo;
import net.sf.amateras.sastruts.util.LogUtil;
import net.sf.amateras.sastruts.util.PreferencesUtil;
import net.sf.amateras.sastruts.util.StringUtil;
import net.sf.amateras.sastruts.util.WorkbenchUtil;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.Flags;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IField;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.Signature;
import org.eclipse.jdt.ui.ISharedImages;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.ui.IEditorPart;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import tk.eclipse.plugin.htmleditor.HTMLUtil;
import tk.eclipse.plugin.htmleditor.JavaUtil;
import tk.eclipse.plugin.htmleditor.assist.AssistInfo;
import tk.eclipse.plugin.jspeditor.editors.JSPEditor;

/**
 * SAStrutsプラグインで使用するユーティリティメソッドを提供します。
 * 
 * @author Naoki Takezoe
 */
public class SAStrutsUtil {
	
	public static String[] splitProperty(String el){
		String[] dim = el.trim().split("\\.|\\[");
		if(!el.endsWith(".")){
			return dim;
		}
		ArrayList<String> list = new ArrayList<String>();
		for(int i=0;i<dim.length;i++){
			list.add(dim[i]);
		}
		list.add("");
		return list.toArray(new String[list.size()]);
	}	
	
	public static PropertyInfo evalProperty(String[] dim, IType parent){
		PropertyInfo result = null;
		
		for(PropertyInfo property: getDisplayProperties(parent)){
			if(property.name.equals(dim[0])){
				result = property;
				break;
			}
		}
		
		if(result==null){
			return null;
		}
		
		try {
			for(int i=1;i<dim.length-1;i++){
				PropertyInfo[] props = result.getProperties();
				boolean flag = false;
				for(int j=0;j<props.length;j++){
					if(props[j].name.equals(dim[i])){
						result = props[j];
						flag = true;
						break;
					}
				}
				if(flag==false){
					return null;
				}
			}
		} catch(Exception ex){
		}
		return result;
	}	
	
	/**
	 * 配列またはジェネリクスを使用したリスト要素の型名を切り出して取得します。
	 * 取得できない場合は引数で渡された型名をそのまま返却します。
	 * 
	 * @param typeName 配列またはジェネリクスを使用したリストの型名
	 * @return 要素の型名
	 */
	public static String getElementType(String typeName){
		if(typeName.endsWith("[]")){
			typeName = typeName.substring(0, typeName.length() - 2);
		} else if(typeName.endsWith(">")){
			typeName = typeName.substring(typeName.indexOf('<') + 1, typeName.length() - 1);
		}
		return typeName;
	}
	
	/**
	 * 表示用のプロパティを取得します。
	 * <p>
	 * 引数に渡した<code>IType</code>のプロパティ、
	 * およびアクションフォームのプロパティのリストを返却します。
	 * 
	 * @param action アクションのITypeオブジェクト
	 * @return 表示用のプロパティのリスト
	 */
	public static PropertyInfo[] getDisplayProperties(IType action){
		
		if(action.getFullyQualifiedName().startsWith("java.lang.")){
			return new PropertyInfo[0];
		}
		
		List<PropertyInfo> result = new ArrayList<PropertyInfo>();
		
		try {
			// アクションのpublicフィールド
			IType actionForm = null;
			while(true){
				for(IMethod method: action.getMethods()){
					int flag = method.getFlags();
					if(Flags.isPublic(flag) && method.getElementName().startsWith("get")){
						String propertyName = method.getElementName().substring(3, 4).toLowerCase()
							+ method.getElementName().substring(4);
						
						result.add(new PropertyInfo(action, propertyName, 
								Signature.toString(method.getReturnType()), method));
					}
				}
				for(IField field: action.getFields()){
					int flag = field.getFlags();
					if((Flags.isPublic(flag) || Flags.isProtected(flag)) && !Flags.isStatic(flag)){
						if(field.getElementName().endsWith("Form")){
							String clazz = JavaUtil.getFullQName(action, Signature.toString(field.getTypeSignature()));
							actionForm = field.getJavaProject().findType(clazz);
						} else {
							if(Flags.isPublic(flag)){
								result.add(new PropertyInfo(action, field.getElementName(), 
										Signature.toString(field.getTypeSignature()), field));
							}
						}
					}
				}
				
				
				// スーパークラス
				if(action.getSuperclassTypeSignature()==null){
					break;
				}
				String superClass = JavaUtil.getFullQName(action, 
						Signature.toString(action.getSuperclassTypeSignature()));
				
				if(superClass.startsWith("java.lang.")){
					break;
				}
				
				action = action.getJavaProject().findType(superClass);
			}
			
			// アクションフォーム
			if(actionForm != null){
				while(true){
					for(IMethod method: actionForm.getMethods()){
						int flag = method.getFlags();
						if(Flags.isPublic(flag) && method.getElementName().startsWith("get")){
							String propertyName = method.getElementName().substring(3, 4).toLowerCase()
							+ method.getElementName().substring(4);
							
							result.add(new PropertyInfo(actionForm, propertyName, 
									Signature.toString(method.getReturnType()), method));
						}
					}
					for(IField field: actionForm.getFields()){
						int flag = field.getFlags();
						if(Flags.isPublic(flag) && !Flags.isStatic(flag)){
							result.add(new PropertyInfo(actionForm, field.getElementName(),
									Signature.toString(field.getTypeSignature()), field));
						}
					}
					
					// スーパークラス
					if(actionForm.getSuperclassTypeSignature() == null){
						break;
					}
					String superClass = JavaUtil.getFullQName(actionForm, 
							Signature.toString(actionForm.getSuperclassTypeSignature()));
					
					if(superClass.startsWith("java.lang.")){
						break;
					}
					
					actionForm = actionForm.getJavaProject().findType(superClass);
				}
			}
		} catch(Exception ex){
			LogUtil.log(Activator.getDefault(), ex);
		}
		
		return result.toArray(new PropertyInfo[result.size()]);
	}
	
	public static FuzzyXMLElement getFormElement(FuzzyXMLElement element){
		FuzzyXMLElement parent = element;
		while(parent != null){
			if(parent.getName().endsWith(":form")){
				return parent;
			}
			parent = (FuzzyXMLElement) parent.getParentNode();
		}
		return null;
	}
	
	/**
	 * JSPからアクションの<code>IType</code>オブジェクトを取得します。
	 * 
	 * @param jspFile JSPファイル
	 * @param <code>s:form</code>タグの<code>action</code>属性の値
	 * @return アクションの<code>IType</code>オブジェクト。
	 *   取得できない場合は<code>null</code>を返します。
	 */
	public static IType getAction(IFile jspFile, String action){
		try {
			IFile javaFile = SAStrutsUtil.getJavaFileFromJSP(jspFile, action);
			if(javaFile != null){
				ICompilationUnit cu = (ICompilationUnit) JavaCore.create(javaFile);
				IType type = cu.getAllTypes()[0];
				return type;
			}
		} catch(JavaModelException ex){
		}
		return null;
	}
	
	/**
	 * JSPファイルから対応するアクションのファイルを取得します。
	 * 
	 * @param jspFile JSPファイル
	 * @param <code>s:form</code>タグの<code>action</code>属性の値
	 * @return アクションのファイル
	 */
	public static IFile getJavaFileFromJSP(IFile jspFile, String action){
		IEditorPart editor = WorkbenchUtil.getActiveEditor();
		if(!(editor instanceof JSPEditor)){
			return null;
		}
		
		IProject project = jspFile.getProject();
		String rootPackageName = getRootPackageName(project);
		if (StringUtil.isEmpty(rootPackageName)) {
			return null;
		}
		String[] splitSubApplications = getSplitSubApplications(jspFile, action);
		if (splitSubApplications == null) {
			return null;
		}
		IFile javaFile = getJavaFile(rootPackageName, jspFile, splitSubApplications);
		return javaFile;
	}
	
	private static IFile getJavaFile(String rootPackageName, IFile jspFile,
			String[] splitSubApplications) {
		StringBuffer sb = new StringBuffer();
		for (int i = 0; i < splitSubApplications.length - 1; i++) {
			sb.append(splitSubApplications[i]).append(File.separator);
		}
		String firstCandidateAction = sb.toString()
				+ StringUtil
						.capitalize(splitSubApplications[splitSubApplications.length - 1])
				+ SAStrutsConstants.ACTION + SAStrutsConstants.JAVA_SUFFIX;
		String secondCandidateAction = sb.toString()
				+ splitSubApplications[splitSubApplications.length - 1]
				+ File.separator + SAStrutsConstants.CAPITALIZE_INDEX_ACTION
				+ SAStrutsConstants.JAVA_SUFFIX;
		String mainJavaPath = PreferencesUtil.getPreferenceStoreOfProject(
				jspFile.getProject()).getString(
				SAStrutsConstants.PREF_MAIN_JAVA_PATH);
		IFile firstCandidateJavaFile = jspFile.getProject().getFile(
				mainJavaPath + File.separator
						+ rootPackageName.replace('.', '/') + File.separator
						+ SAStrutsConstants.LOWER_CASE_ACTION + File.separator
						+ firstCandidateAction);
		IFile secondCandidateJavaFile = jspFile.getProject().getFile(
				mainJavaPath + File.separator
						+ rootPackageName.replace('.', '/') + File.separator
						+ SAStrutsConstants.LOWER_CASE_ACTION + File.separator
						+ secondCandidateAction);
		if (firstCandidateJavaFile.exists()) {
			return firstCandidateJavaFile;
		} else if (secondCandidateJavaFile.exists()) {
			return secondCandidateJavaFile;
		} else {
			return firstCandidateJavaFile;
		}
	}

	/**
	 * アプリケーションのルートパッケージを取得します。
	 * 
	 * @param project プロジェクト
	 * @return ルートパッケージ名。取得できない場合は<code>null</code>を返します。
	 */
	public static String getRootPackageName(IProject project) {
		String conventionDiconPath = PreferencesUtil
				.getPreferenceStoreOfProject(project).getString(
						SAStrutsConstants.PREF_CONVENTION_DICON_PATH);
		File conventionDicon = ((Path) project.getFile(conventionDiconPath)
				.getLocation()).toFile();
		try {
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory
					.newInstance();
			dbfactory.setNamespaceAware(true);
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			builder.setEntityResolver(new EntityResolver() {
				public InputSource resolveEntity(String publicId,
						String systemId) throws SAXException, IOException {
					if (publicId.equals(SAStrutsConstants.PUBLIC_ID_DICON_24)
							&& systemId
									.equals(SAStrutsConstants.SYSTEM_ID_DICON_24)) {
						try {
							InputSource source = new InputSource(Activator
									.getDefault().getBundle().getEntry(
											SAStrutsConstants.DTD_DICON_24)
									.openStream());
							return source;
						} catch (IOException e) {
							LogUtil.log(Activator.getDefault(), e);
						}
					}
					return null;
				}
			});
			Document doc = builder.parse(conventionDicon);
			XPathFactory factory = XPathFactory.newInstance();
			XPath xpath = factory.newXPath();
			XPathExpression expr = xpath
					.compile(SAStrutsConstants.ROOTPACKAGE_XPATH);
			Object result = expr.evaluate(doc, XPathConstants.NODESET);
			NodeList nodes = (NodeList) result;
			if (nodes.getLength() == 0) {
			} else if (nodes.getLength() == 1) {
				return StringUtil.decodeString(nodes.item(0).getNodeValue());
			} else {
				for (int i = 0; i < nodes.getLength(); i++) {
					String rootPackageName = StringUtil.decodeString(nodes
							.item(i).getNodeValue());
					IJavaProject javaProject = JavaCore.create(project);
					IPackageFragmentRoot[] roots = javaProject
							.getPackageFragmentRoots();
					for (int j = 0; j < roots.length; j++) {
						if (roots[j].getKind() == IPackageFragmentRoot.K_SOURCE) {
							IPackageFragment packageFragment = roots[j]
									.getPackageFragment(rootPackageName
											+ "."
											+ SAStrutsConstants.LOWER_CASE_ACTION);
							if (packageFragment.exists()) {
								return rootPackageName;
							}
						}
					}
				}
			}
		} catch (Exception e) {
		}
		return null;
	}
	
	private static String[] getSplitSubApplications(IFile jspFile,
			String actionAttribute) {
		String[] names = null;
		if (!StringUtil.isEmpty(actionAttribute)
				&& actionAttribute.startsWith("/")) {
			names = StringUtil.split(actionAttribute, "/");
		} else {
			IProject project = jspFile.getProject();
			String jspFilePath = jspFile.getFullPath().toOSString();
			String projectPath = project.getFullPath().toOSString();
			String webRootViewPrefix = getWebRootViewPrefix(project);
			if (StringUtil.isEmpty(webRootViewPrefix)) {
				return null;
			}
			jspFilePath = jspFilePath.substring(projectPath.length()
					+ webRootViewPrefix.length() + 1, jspFilePath.length());
			String componentName = jspFilePath.substring(0, jspFilePath
					.lastIndexOf(File.separator));
			names = StringUtil.split(componentName, File.separator);
		}
		return names;
	}
	
	/**
	 * Webアプリケーションのルート+ビュープレフィックスを取得します。
	 * 
	 * @param project プロジェクト
	 * @return Webアプリケーションのルート+ビュープレフィックス。
	 *   取得できない場合は<code>null</code>を返します。
	 */
	public static String getWebRootViewPrefix(IProject project) {
		String webRoot = PreferencesUtil.getPreferenceStoreOfProject(project)
				.getString(SAStrutsConstants.PREF_WEBCONTENTS_ROOT);
		if (webRoot.endsWith(File.separator)) {
			webRoot = webRoot.substring(0, webRoot.length() - 1);
		}
		File webXmlFile = ((Path) project.getFile(
				webRoot + SAStrutsConstants.WEB_INF_WEB_XML).getLocation())
				.toFile();
		if (webXmlFile.exists()) {
			String viewPrefix = null;
			try {
				viewPrefix = getViewPrefix(webXmlFile);
			} catch (Exception e) {
				return null;
			}
			if (!StringUtil.isEmpty(viewPrefix)) {
				webRoot += viewPrefix;
			}
			return webRoot;
			
		} else {
			return null;
		}
	}
	
	/**
	 * ビュープレフィックスを取得します。
	 * 
	 * @param project プロジェクト
	 * @return ビュープレフィックス。取得できない場合は<code>null</code>を返します。
	 */
	public static String getViewPrefix(File webXmlFile) throws SAXException,
			IOException, ParserConfigurationException {
		DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
		dbfactory.setNamespaceAware(true);
		DocumentBuilder builder = dbfactory.newDocumentBuilder();
		Document doc = builder.parse(webXmlFile);
		Element element = doc.getDocumentElement();
		NodeList contextParamNodeList = element
				.getElementsByTagName(SAStrutsConstants.CONTEXT_PARAM);
		if (contextParamNodeList.getLength() == 1
				&& contextParamNodeList.item(0) instanceof Element) {
			Element contextParamElement = (Element) contextParamNodeList
					.item(0);
			NodeList paramNameNodeList = contextParamElement
					.getElementsByTagName(SAStrutsConstants.PARAM_NAME);
			if (paramNameNodeList.getLength() == 1) {
				if (((Node) paramNameNodeList.item(0)).getTextContent().equals(
						SAStrutsConstants.SASTRUTS_VIEW_PREFIX)) {
					NodeList paramValueNodeList = contextParamElement
							.getElementsByTagName(SAStrutsConstants.PARAM_VALUE);
					if (paramValueNodeList.getLength() == 1) {
						return ((Node) paramValueNodeList.item(0))
								.getTextContent();
					}
				}
			}
		}
		return null;
	}
	
	/**
	 * プロパティの補完情報を作成します。
	 */
	public static AssistInfo createAssistInfo(PropertyInfo property){
		try {
			AssistInfo assistInfo = new AssistInfo(
					property.getNameWithPrefix(),
					property.name + " - " + property.type,
					JavaUI.getSharedImages().getImage(ISharedImages.IMG_FIELD_PUBLIC),
					HTMLUtil.extractJavadoc(property.element, null));
			
			return assistInfo;
		} catch(Exception ex){
			LogUtil.log(Activator.getDefault(), ex);
		}
		return null;
	}
}
