package net.sf.amateras.air.debug.thread;

import net.sf.amateras.air.debug.AirDebugElement;

import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IRegisterGroup;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;
import org.eclipse.debug.core.model.IVariable;

/**
 * AIR stack frame.
 * @author hideko ogawa
 */
public class AirStackFrame extends AirDebugElement implements IStackFrame {

	/** parent thread */
	private AirThread thread;

	/** target source name */
	private String sourceName = "";

	/** target source line no */
	private int lineNumber;

	/** FDB Stack No */
	private String fdbNo;

	/** Stack ObjectString */
	private String objectString;

	/** Stack Method */
	private String method;

	private int identifer;
	private static int finalIdentifer = 0;

	/**
	 * Constructs a stack frame in the given thread with the given
	 * frame data.
	 * 
	 * @param thread
	 * @param sourceName
	 * @param lineNo
	 */
	public AirStackFrame(AirThread thread, String sourceName, int lineNo) {
		this(thread);
		this.thread = thread;
		this.sourceName = sourceName;
		this.lineNumber = lineNo;
	}

	public AirStackFrame(AirThread thread, String sourceName, int lineNo, String no, String objectString, String method) {
		this(thread, sourceName, lineNo);
		this.fdbNo = no;
		this.objectString = objectString;
		this.method = method;
	}

	private AirStackFrame(AirThread thread) {
		super(thread.getAirDebugTarget());
		this.thread = thread;
		finalIdentifer++;
		identifer = finalIdentifer;
	}

	public IThread getThread() {
		return thread;
	}

	public IVariable[] getVariables() {
		return thread.getVariables();
	}

	public boolean hasVariables() {
		return getVariables().length > 0;
	}

	public int getCharStart() {
		return -1;
	}

	public int getCharEnd() {
		return -1;
	}

	public String getName() {
		StringBuilder sb = new StringBuilder();
		sb.append(sourceName);
		//		if (objectString != null) {
		//			sb.append(" ").append(objectString);
		//		}
		if (method != null) {
			sb.append("#").append(method);
		}
		sb.append(" [line: " + lineNumber + "]");
		return sb.toString();
	}

	public IRegisterGroup[] getRegisterGroups() {
		return null;
	}

	public boolean hasRegisterGroups() {
		return false;
	}

	public String getSourceName() {
		return sourceName;
	}

	@Override
	public boolean equals(Object obj) {
		if (obj instanceof AirStackFrame) {
			AirStackFrame sf = (AirStackFrame) obj;
			return sf.getSourceName().equals(getSourceName());
		} else {
			return false;
		}
	}

	@Override
	public int hashCode() {
		return getSourceName().hashCode();
	}

	protected int getIdentifier() {
		return identifer;
	}

	// *****************************************************
	// Command
	// see AirThread Command.
	// *****************************************************

	public boolean canStepInto() {
		return getThread().canStepInto();
	}

	public boolean canStepOver() {
		return getThread().canStepOver();
	}

	public boolean canStepReturn() {
		return getThread().canStepReturn();
	}

	public boolean isStepping() {
		return getThread().isStepping();
	}

	public void stepInto() throws DebugException {
		getThread().stepInto();
	}

	public void stepOver() throws DebugException {
		getThread().stepOver();
	}

	public void stepReturn() throws DebugException {
		getThread().stepReturn();
	}

	public boolean canResume() {
		return getThread().canResume();
	}

	public boolean canSuspend() {
		return getThread().canSuspend();
	}

	public boolean isSuspended() {
		return getThread().isSuspended();
	}

	public void resume() throws DebugException {
		getThread().resume();
	}

	public void suspend() throws DebugException {
		getThread().suspend();
	}

	public boolean canTerminate() {
		return getThread().canTerminate();
	}

	public boolean isTerminated() {
		return getThread().isTerminated();
	}

	public void terminate() throws DebugException {
		getThread().terminate();
	}

	public int getLineNumber() {
		return lineNumber;
	}

	public void setLineNumber(int lineNo) {
		this.lineNumber = lineNo;
	}

	public String getFdbNo() {
		return fdbNo;
	}

	public void setFdbNo(String no) {
		this.fdbNo = no;
	}

	public String getObjectString() {
		return objectString;
	}

	public void setObjectString(String objectString) {
		this.objectString = objectString;
	}

	public String getMethod() {
		return method;
	}

	public void setMethod(String method) {
		this.method = method;
	}

	public void setSourceName(String sourceName) {
		this.sourceName = sourceName;
	}

}
