package net.sf.amateras.air.wizards;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.util.Map;
import java.util.Map.Entry;

import net.sf.amateras.air.AIRPlugin;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.internal.ide.IDEWorkbenchPlugin;
import org.eclipse.ui.internal.wizards.newresource.ResourceMessages;
import org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard;

@SuppressWarnings("restriction")
public abstract class AbstractNewProjectWizard extends Wizard implements INewWizard, IExecutableExtension {

	private static final int TICKS = 1000;
	private static final int TOTAL_WORK = 2000;
	protected IProject newProject;
	protected AbstractProjectWizardPage page1;
	protected WizardPage page2;
	protected IConfigurationElement config;

	/**
	 * Creates the first wizard page.
	 * @return the first wizard page
	 */
	protected abstract AbstractProjectWizardPage createPage1();

	/**
	 * Creates the second wizard page.
	 * @return the second wizard page
	 */
	protected abstract WizardPage createPage2();

	protected abstract void addProjectSpecificParameters(Map<String, String> params);

	protected abstract void createProjectSpecificResources(IProject project, String descriptorTemplateName,
			Map<String, String> params, IProgressMonitor monitor) throws Exception;

	/**
	 * Returns the nature id to add new project.
	 * @return the nature id
	 */
	protected abstract String getNatureId();

	public void setInitializationData(IConfigurationElement config, String propertyName, Object data) {
		this.config = config;
	}

	public void init(IWorkbench workbench, IStructuredSelection selection) {
		// Nothing to do.
	}

	@Override
	public void addPages() {
		page1 = createPage1();
		addPage(page1);

		page2 = createPage2();
		if (page2 != null) {
			addPage(page2);
		}
	}

	@Override
	public boolean performFinish() {
		createNewProject();

		if (newProject == null) {
			return false;
		}
		BasicNewProjectResourceWizard.updatePerspective(config);
		//        updatePerspective();
		//        selectAndReveal(newProject);

		return true;
	}

	protected void createFile(IProject project, String path, InputStream in, Map<String, String> params,
			IProgressMonitor monitor) throws Exception {
		if (params == null) {
			IFile file = project.getFile(new Path(path));
			file.create(in, true, monitor);
			return;
		}
		try {
			byte[] buf = new byte[in.available()];
			in.read(buf);
			String xml = new String(buf, "UTF-8");

			// replaces variables
			for (Entry<String, String> entry : params.entrySet()) {
				xml = xml.replaceAll("\\$\\{" + entry.getKey() + "\\}", entry.getValue());
			}

			// creates file
			IFile file = project.getFile(new Path(path));
			file.create(new ByteArrayInputStream(xml.getBytes("UTF-8")), true, monitor);
		} finally {
			in.close();
		}
	}

	/**
	 * Creates a project resource given the project handle and description.
	 * 
	 * @param description
	 *            the project description to create a project resource for
	 * @param projectHandle
	 *            the project handle to create a project resource for
	 * @param monitor
	 *            the progress monitor to show visual progress with
	 * 
	 * @exception CoreException
	 *                if the operation fails
	 * @exception OperationCanceledException
	 *                if the operation is canceled
	 */
	private void createProject(IProjectDescription description, IProject projectHandle, IProgressMonitor monitor)
			throws CoreException, OperationCanceledException {
		try {
			monitor.beginTask("", TOTAL_WORK);
			projectHandle.create(description, new SubProgressMonitor(monitor, TICKS));
			if (monitor.isCanceled()) {
				throw new OperationCanceledException();
			}
			projectHandle.open(IResource.BACKGROUND_REFRESH, new SubProgressMonitor(monitor, TICKS));

		} finally {
			monitor.done();
		}
	}

	private IProject createNewProject() {
		if (newProject != null) {
			return newProject;
		}

		final IProject newProjectHandle = page1.getProjectHandle();
		URI location = null;
		if (!page1.useDefaults()) {
			location = page1.getLocationURI();
		}

		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		final IProjectDescription description = workspace.newProjectDescription(newProjectHandle.getName());
		description.setLocationURI(location);

		final Map<String, String> params = createParamMap(page2);
		addProjectSpecificParameters(params);

		// create the new project operation
		WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
			@Override
			protected void execute(IProgressMonitor monitor) throws CoreException {
				createProject(description, newProjectHandle, monitor);
				AIRPlugin.addNature(newProjectHandle, getNatureId());
				createResource(newProjectHandle, monitor, params);
			}
		};

		if (create(op, newProjectHandle.getName())) {
			newProject = newProjectHandle;
			return newProject;
		} else {
			return null;
		}

	}

	private boolean create(WorkspaceModifyOperation op, String name) {
		try {
			getContainer().run(true, true, op);
			return true;
		} catch (InterruptedException e) {
			return false;
		} catch (InvocationTargetException e) {
			// ie.- one of the steps resulted in a core exception
			Throwable t = e.getTargetException();
			if (t instanceof CoreException) {
				if (((CoreException) t).getStatus().getCode() == IResourceStatus.CASE_VARIANT_EXISTS) {
					MessageDialog.openError(getShell(), ResourceMessages.NewProject_errorMessage, NLS.bind(
							ResourceMessages.NewProject_caseVariantExistsError, name));
				} else {
					ErrorDialog.openError(getShell(), ResourceMessages.NewProject_errorMessage, null, // no special message
							((CoreException) t).getStatus());
				}
			} else {
				// CoreExceptions are handled above, but unexpected runtime
				// exceptions and errors may still occur.
				IDEWorkbenchPlugin.getDefault().getLog().log(
						new Status(IStatus.ERROR, IDEWorkbenchPlugin.IDE_WORKBENCH, 0, t.toString(), t));
				MessageDialog.openError(getShell(), ResourceMessages.NewProject_errorMessage, NLS.bind(
						ResourceMessages.NewProject_internalError, t.getMessage()));
			}
			return false;
		}
	}

	protected abstract Map<String, String> createParamMap(WizardPage page);

	protected abstract void createResource(IProject newProjectHandle, IProgressMonitor monitor,
			Map<String, String> params);
}
