package net.sf.amateras.air.wizards.file;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Map;

import net.sf.amateras.air.AIRPlugin;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard;

public abstract class AbstractFileWizard extends Wizard implements INewWizard {
	protected ISelection selection;
	protected AbstractFileCreationPage page;

	public abstract AbstractFileCreationPage createPage(ISelection selection);

	public abstract InputStream openContentStream(String fileName, String packageName, String templateName,
			Map<String, String> params);

	public void init(IWorkbench workbench, IStructuredSelection selection) {
		this.selection = selection;
		setNeedsProgressMonitor(true);
	}

	@Override
	public void addPages() {
		page = createPage(selection);
		addPage(page);
	}

	@Override
	public boolean performFinish() {
		IRunnableWithProgress op = getRunnableFinish();
		try {
			getContainer().run(true, false, op);
		} catch (InterruptedException e) {
			return false;
		} catch (InvocationTargetException e) {
			Throwable realException = e.getTargetException();
			MessageDialog.openError(getShell(), "Error", realException.getMessage());
			return false;
		}
		return true;
	}

	/**
	 * do finish by RunnableProgress.
	 */
	private IRunnableWithProgress getRunnableFinish() {
		final IPath path = page.getFilePath();
		final String packg = page.getPackageName();
		final String template = page.getTemplate();
		final Map<String, String> params = page.getParameters();
		return new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException {
				try {
					doFinish(path, packg, template, params, monitor);
				} catch (CoreException e) {
					throw new InvocationTargetException(e);
				} finally {
					monitor.done();
				}
			}
		};
	}

	/**
	 * do finish.
	 * @param path
	 * @param packg
	 * @param template
	 * @param Map
	 * @param monitor
	 * @throws CoreException
	 */
	protected void doFinish(IPath path, String packg, String template, Map<String, String> params,
			IProgressMonitor monitor) throws CoreException {
		String fileName = path.lastSegment();
		monitor.beginTask("Creating " + fileName, 2);
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IPath contPath = path.removeLastSegments(1);
		IResource resource = root.findMember(contPath);
		IContainer container = null;
		if (resource == null || !resource.exists()) {
			if (contPath.segmentCount() > 1 && root.getProject(contPath.segment(0)).exists()) {
				IProject project = root.getProject(contPath.segment(0));
				container = createFolder(project, contPath, monitor);
			}
		} else if (!(resource instanceof IContainer)) {
			throwCoreException("\"" + path + "\" is not a valid Container.");
		}
		if (container == null) {
			container = (IContainer) resource;
		}

		final IFile file = container.getFile(new Path(fileName));
		try {
			String fileString = path.removeFileExtension().lastSegment();

			InputStream stream = openContentStream(fileString, packg, template, params);
			if (file.exists()) {
				file.setContents(stream, true, true, monitor);
			} else {
				file.create(stream, true, monitor);
			}
			stream.close();
		} catch (IOException e) {
			AIRPlugin.logException(e);
		}

		doOther(container, path, params, monitor);
		monitor.worked(1);
		monitor.setTaskName("Opening file for editing...");
		asyncActivePage(file);
		monitor.worked(1);
	}

	protected abstract void doOther(IContainer container, IPath path, Map<String, String> params,
			IProgressMonitor moniter) throws CoreException;

	private IFolder createFolder(IContainer project, IPath containerPath, IProgressMonitor monitor)
			throws CoreException {
		IFolder folder = null;
		for (int i = 1; i < containerPath.segmentCount(); i++) {
			Path segPath = new Path(containerPath.segment(i));
			if (folder == null) {
				folder = project.getFolder(segPath);
			} else {
				folder = folder.getFolder(segPath);
			}
			if (!folder.exists()) {
				folder.create(false, true, monitor);
			}
		}
		return folder;
	}

	protected void throwCoreException(String message) throws CoreException {
		IStatus status = new Status(IStatus.ERROR, "AIR GEAR", IStatus.OK, message, null);
		throw new CoreException(status);
	}

	protected void asyncActivePage(final IFile file) {
		getShell().getDisplay().asyncExec(new Runnable() {
			public void run() {
				IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
				BasicNewProjectResourceWizard.selectAndReveal(file, window);
				try {
					IDE.openEditor(window.getActivePage(), file, true);
				} catch (PartInitException e) {
					AIRPlugin.logException(e);
				}
			}
		});
	}

}
