package net.sf.amateras.air.as.format;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.as.ActionScriptPartitionScanner;
import net.sf.amateras.air.preferences.FormatPropertyPage;

import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentPartitioner;
import org.eclipse.jface.text.ITypedRegion;
import org.eclipse.jface.text.rules.FastPartitioner;
import org.eclipse.text.edits.ReplaceEdit;

/**
 * Simple format... no syntax parse.
 * @author hideko ogawa
 */
public class ActionScriptCodeFormatter {

	public ReplaceEdit format(IDocument document, int documentOffset, int documentLength, String lineDelimiter) {

		IDocumentPartitioner partitioner = new FastPartitioner(new ActionScriptPartitionScanner(), new String[] {
				ActionScriptPartitionScanner.AS_COMMENT, ActionScriptPartitionScanner.AS_STRING });
		partitioner.connect(document);
		ITypedRegion[] typedRegions = partitioner.computePartitioning(0, document.getLength());

		String prefixSimaple = getPrefixSimple();
		StringBuilder sb = new StringBuilder();
		try {
			int indentIndex = 0;
			boolean isLineDelimiterAfter = false;

			char[] chars = document.get().toCharArray();

			for (int i = 0; i < chars.length; i++) {
				if (isString(i, typedRegions)) {
					sb.append(chars[i]);
					continue;
				}

				boolean isComment = isComment(i, typedRegions);
				if (!isComment && chars[i] == '}') {
					indentIndex--;
				}

				boolean isSelected = isSelected(i, documentOffset, documentLength);
				if (isLineDelimiterAfter && isSelected) {
					if (chars[i] == ' ' || chars[i] == '\t') {
						continue;
					} else if (chars[i] != '\n') {
						sb.append(getPrefix(indentIndex, prefixSimaple));
						isLineDelimiterAfter = false;
					}
				}

				sb.append(chars[i]);

				if (!isComment && chars[i] == '{') {
					indentIndex++;
				}

				if (chars[i] == '\n') {
					isLineDelimiterAfter = true;
				}
			}

			String newContents = sb.toString();
			if (newContents == null || document.get().equals(newContents)) {
				return null;
			} else {
				return new ReplaceEdit(0, newContents.length(), newContents);
			}

		} catch (Exception ex) {
			ex.printStackTrace();
			return null;
		}
	}

	private String getPrefixSimple() {
		IPreferenceStore store = AIRPlugin.getDefault().getPreferenceStore();
		int tabPolicy = store.getInt(AIRPlugin.PREF_FORMATTER_TAB_POLICY);
		if (tabPolicy == FormatPropertyPage.TAB_POLICY_SPACES_ONLY) {
			int spaceSize = store.getInt(AIRPlugin.PREF_FORMATTER_SPACE_INDENT_SIZE);
			StringBuilder sb = new StringBuilder();
			for (int i = 0; i < spaceSize; i++) {
				sb.append(' ');
			}
			return sb.toString();
		} else {
			return "\t";
		}
	}

	private String getPrefix(int indentIndex, String prefixSimple) {
		StringBuilder sb = new StringBuilder();
		for (int i = 0; i < indentIndex; i++) {
			sb.append(prefixSimple);
		}
		return sb.toString();
	}

	private boolean isComment(int offset, ITypedRegion[] typedRegions) {
		for (ITypedRegion region : typedRegions) {
			if (region.getType().equals(ActionScriptPartitionScanner.AS_COMMENT)) {
				if (region.getOffset() < offset && offset < (region.getOffset() + region.getLength())) {
					return true;
				}
			}
		}
		return false;
	}

	private boolean isString(int offset, ITypedRegion[] typedRegions) {
		for (ITypedRegion region : typedRegions) {
			if (region.getType().equals(ActionScriptPartitionScanner.AS_STRING)) {
				if (region.getOffset() < offset && offset < (region.getOffset() + region.getLength())) {
					return true;
				}
			}
		}
		return false;
	}

	private boolean isSelected(int offset, int documentOffset, int documentLength) {
		if (documentLength == 0) {
			return true;
		}
		return documentOffset <= offset && (documentOffset + documentLength) >= offset;
	}

}
