package net.sf.amateras.air.launch;

import net.sf.amateras.air.AIRPlugin;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugUIConstants;
import org.eclipse.debug.ui.ILaunchShortcut;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

/**
 * AIR application Launch Shortcut.
 * 
 * @author ogawahideko
 *
 */
public class LaunchAIRShortcut implements ILaunchShortcut {

	/**
	 * implements {@link ILaunchShortcut}
	 */
	public void launch(ISelection selection, String mode) {
		if (selection != null && selection instanceof IStructuredSelection && !selection.isEmpty()) {
			IProject targetProject = null;
			Object obj = ((IStructuredSelection) selection).getFirstElement();
			if (obj instanceof IProject) {
				targetProject = (IProject) obj;
			} else if (obj instanceof IResource) {
				targetProject = ((IResource) obj).getProject();
			} else if (obj instanceof IAdaptable) {
				targetProject = (IProject) ((IAdaptable) obj).getAdapter(IProject.class);
			}

			if (targetProject != null) {
				launchApplication(targetProject, mode);
			}
		}
	}

	/**
	 * implements {@link ILaunchShortcut}
	 */
	public void launch(IEditorPart editor, String mode) {
		IEditorInput input = editor.getEditorInput();
		if (input instanceof IFileEditorInput) {
			IProject targetProject = ((IFileEditorInput) input).getFile().getProject();
			launchApplication(targetProject, mode);
		}
	}

	/**
	 * build and launch for air application.
	 * @param targetProject
	 */
	private void launchApplication(IProject targetProject, String mode) {
		try {
			ILaunchConfiguration config = getLaunchConfiguration(targetProject);
			DebugUITools.launch(config, mode);
		} catch (Exception ex) {
			AIRPlugin.logException(ex);
		}
	}

	/**
	 * create and save for air launch configuration.
	 * @param project
	 * @return
	 * @throws CoreException
	 */
	private ILaunchConfiguration getLaunchConfiguration(IProject project) throws CoreException {
		ILaunchManager manager = DebugPlugin.getDefault().getLaunchManager();
		ILaunchConfiguration[] configs = manager.getLaunchConfigurations();

		for (int i = 0; i < configs.length; i++) {
			String value = configs[i].getAttribute(LaunchAIRMainTab.ATTR_PROJECT, "");
			if (value.equals(project.getName())) {
				return configs[i];
			}
		}

		ILaunchConfigurationType type = manager.getLaunchConfigurationType(LaunchAIRConfiguration.ID);

		ILaunchConfigurationWorkingCopy wc = type.newInstance(null, manager
				.generateUniqueLaunchConfigurationNameFrom(project.getName()));

		ScopedPreferenceStore projectStore = new ScopedPreferenceStore(new ProjectScope(project), AIRPlugin.PLUGIN_ID);
		String descriptor = projectStore.getString(AIRPlugin.PREF_DESCRIPTOR);

		wc.setAttribute(LaunchAIRMainTab.ATTR_PROJECT, project.getName());
		wc.setAttribute(LaunchAIRMainTab.ATTR_TARGET, descriptor);
		wc.setAttribute(IDebugUIConstants.ATTR_LAUNCH_IN_BACKGROUND, false);

		return wc.doSave();
	}

}
