package net.sf.amateras.air.util;

import java.io.File;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.debug.internal.ui.DebugUIPlugin;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;

/**
 * Provides utility methods about UI.
 * 
 * @author Naoki Takezoe
 */
@SuppressWarnings("restriction")
public class UIUtil {

	private UIUtil() {
	}

	/**
	 * Creates label that displays the given text.
	 * 
	 * @param text the display text
	 * @param parent the parent composite
	 * @return the created label
	 */
	public static Label createLabel(String text, Composite parent) {
		Label label = new Label(parent, SWT.NULL);
		label.setText(text);
		return label;
	}

	public static GridData createGridData(int span) {
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = span;
		return gd;
	}

	public static Button createBrowseFileButton(final Text text, Composite parent) {
		Button button = new Button(parent, SWT.PUSH);
		button.setText("Browse...");
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent evt) {
				FileDialog dialog = new FileDialog(text.getShell(), SWT.SINGLE);
				File file = new File(text.getText());
				dialog.setFilterPath(file.getParent());
				dialog.setFileName(file.getName());
				String result = dialog.open();
				if (result != null) {
					text.setText(result);
				}
			}
		});
		return button;
	}

	public static IProject browseProject() {
		ElementTreeSelectionDialog dialog = new ElementTreeSelectionDialog(PlatformUI.getWorkbench()
				.getActiveWorkbenchWindow().getShell(), new WorkbenchLabelProvider(), new WorkbenchContentProvider());
		dialog.setInput(ResourcesPlugin.getWorkspace());
		dialog.addFilter(new ViewerFilter() {
			@Override
			public boolean select(Viewer viewer, Object parentElement, Object element) {
				if (element instanceof IProject) {
					return true;
				}
				return false;
			}
		});
		dialog.setAllowMultiple(false);
		dialog.setTitle("Select Project");

		if (dialog.open() == Dialog.OK) {
			IProject project = (IProject) dialog.getFirstResult();
			return project;
		}

		return null;
	}

	/**
	 * Opens the error dialog.
	 * 
	 * @param message the message which will be displayed in the dialog.
	 */
	private static void openErrorDialogSub(String message) {
		Shell shell = DebugUIPlugin.getShell();
		if (shell == null) {
			DebugUIPlugin.logErrorMessage(message);
			return;
		}
		MessageDialog.openError(shell, "ERROR", message);
	}

	/**
	 * 
	 * @param message
	 */
	public static void openErrorDialog(final String message) {
		Display display = DebugUIPlugin.getStandardDisplay();
		if (display.getThread().equals(Thread.currentThread())) {
			openErrorDialogSub(message);
		} else {
			Runnable r = new Runnable() {
				public void run() {
					openErrorDialogSub(message);
				}
			};
			DebugUIPlugin.getStandardDisplay().syncExec(r);
		}
	}

	private static void openMessageDialogSub(String message) {
		Shell shell = DebugUIPlugin.getShell();
		MessageDialog.openInformation(shell, "MESSAGE", message);
	}

	public static void openMessageDialog(final String message) {
		Display display = DebugUIPlugin.getStandardDisplay();
		if (display.getThread().equals(Thread.currentThread())) {
			openMessageDialogSub(message);
		} else {
			Runnable r = new Runnable() {
				public void run() {
					openMessageDialogSub(message);
				}
			};
			DebugUIPlugin.getStandardDisplay().syncExec(r);
		}
	}

	/**
	 * Returns an active editor part in the workbench.
	 * 
	 * @return An instance of an active editorpart.
	 */
	public static IEditorPart getActiveEditor() {
		IWorkbench workbench = PlatformUI.getWorkbench();
		IWorkbenchWindow window = workbench.getActiveWorkbenchWindow();
		IWorkbenchPage page = window.getActivePage();
		IEditorPart editorPart = page.getActiveEditor();
		return editorPart;
	}

	public static IProject getProject(String projectName) {
		return ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
	}

}
