package net.sf.amateras.air.wizards.file;

import java.util.List;
import java.util.Map;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.builder.CompileProperty;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ContainerSelectionDialog;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

/**
 * 
 * @author ogawahideko
 *
 */
public abstract class AbstractFileCreationPage extends WizardPage {

	private ISelection selection;
	private Text containerText;
	private Text packageText;
	private Text fileText;
	private String template;

	public AbstractFileCreationPage(ISelection selection) {
		super("wizardPage");
		this.selection = selection;
	}

	public abstract String getExtension();

	public abstract Map<String, String> getParameters();

	public void createControl(Composite parent) {
		Composite container = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout();
		container.setLayout(layout);
		layout.numColumns = 3;
		layout.verticalSpacing = 9;
		Label label = new Label(container, SWT.NULL);
		label.setText("Source &folder:");
		containerText = new Text(container, SWT.BORDER | SWT.SINGLE);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		containerText.setLayoutData(gd);
		ModifyListener modifyListener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				dialogChanged();
			}
		};
		containerText.addModifyListener(modifyListener);

		Button button = new Button(container, SWT.PUSH);
		button.setText("&Browse...");
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				handleBrowse();
			}
		});

		label = new Label(container, SWT.NULL);
		label.setText("&Package:");

		packageText = new Text(container, SWT.BORDER | SWT.SINGLE);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		packageText.setLayoutData(gd);
		packageText.addModifyListener(modifyListener);
		new Label(container, SWT.NULL);

		label = new Label(container, SWT.NULL);
		label.setText("&FileName:");

		fileText = new Text(container, SWT.BORDER | SWT.SINGLE);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		fileText.setLayoutData(gd);
		fileText.addModifyListener(modifyListener);
		fileText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent evt) {
				modifyTextToFileName(fileText.getText());
			}
		});
		new Label(container, SWT.NULL);

		setLayoutText(container, modifyListener);

		label = new Label(container, SWT.NULL);
		label.setText("&Template:");

		SelectionAdapter adapter = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				if (e.widget instanceof Button) {
					Button b = (Button) e.widget;
					template = b.getText().replaceAll("&", "");
				}
			}
		};

		Composite group = new Composite(container, SWT.NONE);
		group.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		group.setLayout(new RowLayout());
		setTemplateLayout(group, adapter);
		new Label(container, SWT.NULL);

		setLayoutOption(container);

		initialize();
		dialogChanged();
		setControl(container);
	}

	protected void setLayoutText(Composite container, ModifyListener modifyListener) {
		//Nothong todo.
	}

	protected void setLayoutOption(Composite container) {
		//Nothong todo.
	}

	protected void modifyTextToFileName(String fileName) {
		//Nothing todo.
	}

	/**
	 * layout template buttons
	 * @param group
	 * @param adapter
	 */
	public abstract void setTemplateLayout(Composite group, SelectionAdapter adapter);

	private void initialize() {
		if (selection != null && selection.isEmpty() == false && selection instanceof IStructuredSelection) {
			IStructuredSelection ssel = (IStructuredSelection) selection;
			Object obj = ssel.getFirstElement();
			if (obj instanceof IResource) {
				setContainer((IResource) obj);
			}
		}
		packageText.setSelection(packageText.getText().length());
		packageText.setFocus();
	}

	/**
	 * set container and package.
	 * @param resource
	 */
	private void setContainer(IResource resource) {
		IContainer cont = getSourceFolder(resource);
		if (cont != null) {
			String pack = getPackageName(resource, cont);
			if (!pack.equals("")) {
				packageText.setText(pack);
			}
		} else {
			if (resource instanceof IContainer) {
				cont = (IContainer) resource;
			} else {
				cont = resource.getParent();
			}
		}
		containerText.setText(cont.getFullPath().toString());
	}

	/**
	 * parse soucefolder.
	 * @param resource
	 * @return
	 */
	private IContainer getSourceFolder(IResource resource) {
		ScopedPreferenceStore store = new ScopedPreferenceStore(new ProjectScope(resource.getProject()),
				AIRPlugin.PLUGIN_ID);
		List<CompileProperty> properties = CompileProperty.load(store);
		for (CompileProperty property : properties) {
			if (property.getDirectory() == null || property.getDirectory().length() == 0) {
				continue;
			}
			IFolder folder = resource.getProject().getFolder(property.getDirectory());
			if (resource.getProjectRelativePath().toString().startsWith(property.getDirectory())) {
				return folder;
			}
		}
		return resource.getProject();
	}

	/**
	 * parse package name.
	 * @param resource
	 * @param container
	 * @return
	 */
	private String getPackageName(IResource resource, IContainer container) {
		String folder;
		if (resource instanceof IFile) {
			folder = resource.getProjectRelativePath().removeLastSegments(1).toString();
		} else {
			folder = resource.getProjectRelativePath().toString();
		}

		folder = folder.substring(container.getProjectRelativePath().toString().length());
		if (folder.startsWith("/")) {
			folder = folder.substring(1);
		}
		if (folder.endsWith("/")) {
			folder = folder.substring(0, folder.length() - 1);
		}
		folder = folder.replaceAll("/", ".");
		return folder;
	}

	/**
	 * browse container dialog.
	 */
	private void handleBrowse() {
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		ContainerSelectionDialog dialog = new ContainerSelectionDialog(getShell(), root, false,
				"Select new file container");
		if (dialog.open() == ContainerSelectionDialog.OK) {
			Object[] result = dialog.getResult();
			if (result.length >= 1) {
				setContainer(root.getFolder((IPath) result[0]));
			}
		}
	}

	protected void dialogChanged() {
		Path contPath = new Path(getContainerName());
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IResource container = root.findMember(contPath);
		String fileName = getFileName();

		if (getContainerName().length() == 0) {
			updateStatus("File container must be specified");
			return;
		}
		if (container == null || (container.getType() & (IResource.PROJECT | IResource.FOLDER)) == 0) {
			updateStatus("File container has to be a existing container.");
			return;
		}
		if (!container.isAccessible()) {
			updateStatus("Project must be writable");
			return;
		}
		if (fileName.length() == 0) {
			updateStatus("File name must be specified");
			return;
		}
		if (fileName.replace('\\', '/').indexOf('/', 1) > 0) {
			updateStatus("File name must not contain slashes");
			return;
		}
		if (fileName.indexOf('.') > 0) {
			updateStatus("File name must not contain dots");
			return;
		}
		IResource file = root.findMember(getFilePath());
		if (file != null) {
			updateStatus("File already exists");
			return;
		}
		if (!checkOther()) {
			return;
		}
		updateStatus(null);
	}

	protected boolean checkOther() {
		return true;
	}

	private void updateStatus(String message) {
		setErrorMessage(message);
		setPageComplete(message == null);
	}

	public String getContainerName() {
		return containerText.getText();
	}

	public String getPackageName() {
		return packageText.getText();
	}

	public IPath getFilePath() {
		IPath path = new Path(getContainerName());
		path = path.append(getPackageName().replace('.', '/'));
		path = path.addTrailingSeparator();
		String fileName = getFileName();
		if (!(fileName.endsWith(getExtension()))) {
			fileName = fileName + getExtension();
		}
		path = path.append(fileName);
		return path;
	}

	public String getFileName() {
		return fileText.getText();
	}

	public String getTemplate() {
		return template;
	}

	public void setTemplate(String template) {
		this.template = template;
	}

}
