package tk.eclipse.plugin.jseditor.editors;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

import org.eclipse.core.resources.IFile;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.Region;
import org.eclipse.jface.text.contentassist.CompletionProposal;
import org.eclipse.jface.text.contentassist.ContextInformation;
import org.eclipse.jface.text.contentassist.ContextInformationValidator;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.jface.text.contentassist.IContextInformation;
import org.eclipse.jface.text.contentassist.IContextInformationValidator;
import org.eclipse.jface.text.templates.DocumentTemplateContext;
import org.eclipse.jface.text.templates.Template;
import org.eclipse.jface.text.templates.TemplateContext;
import org.eclipse.jface.text.templates.TemplateContextType;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.part.WorkbenchPart;
import org.eclipse.ui.texteditor.AbstractDecoratedTextEditor;

import tk.eclipse.plugin.htmleditor.HTMLPlugin;
import tk.eclipse.plugin.htmleditor.HTMLProjectParams;
import tk.eclipse.plugin.htmleditor.HTMLUtil;
import tk.eclipse.plugin.htmleditor.editors.HTMLEditor;
import tk.eclipse.plugin.htmleditor.template.HTMLTemplateAssistProcessor;
import tk.eclipse.plugin.htmleditor.template.HTMLTemplateManager;
import tk.eclipse.plugin.htmleditor.template.JavaScriptContextType;
import tk.eclipse.plugin.jseditor.editors.additional.AdditionalJavaScriptCompleterManager;
import tk.eclipse.plugin.jseditor.editors.additional.IAdditionalJavaScriptCompleter;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptContext;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptElement;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptFunction;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptModel;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptProperty;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptVariable;
import tk.eclipse.plugin.jseditor.editors.model.ModelManager;

/**
 * IContentAssistProcessor implementation for JavaScriptEditor.
 *
 * @author Naoki Takezoe
 */
public class JavaScriptAssistProcessor extends HTMLTemplateAssistProcessor { /* implements IContentAssistProcessor {*/

	public static List<AssistInfo> STATIC_ASSIST_INFO = new ArrayList<AssistInfo>();

	// assist proposal types
	public static final int VARIABLE =  0;
	public static final int FUNCTION =  1;
	public static final int KEYWORD  =  2;
	public static final int OBJECT   =  3;
	public static final int STATIC_VARIABLE = 4;
	public static final int STATIC_FUNCTION = 5;
	public static final int UNDEF    = 99;

	private static final String[] CLASSES = {
			"Arguments", "Array", "Boolean", "Date", "Error",
			"Function", "Math", "NativeError", "Number", "Object", "RegExp", "String"
	};

	public static Map<String, AssistInfo[]> INSTANCE_MEMBERS = new HashMap<String, AssistInfo[]>();
	public static Map<String, AssistInfo[]> STATIC_MEMBERS = new HashMap<String, AssistInfo[]>();

	static {
		// add keyword to static assist informations
		for(int i=0;i<JavaScriptScanner.KEYWORDS.length;i++){
			STATIC_ASSIST_INFO.add(new AssistInfo(KEYWORD, JavaScriptScanner.KEYWORDS[i]));
		}
		STATIC_ASSIST_INFO.add(new AssistInfo(VARIABLE, "window - Window", "window", "Window"));

		STATIC_ASSIST_INFO.add(new AssistInfo(FUNCTION, "decodeURI(uri) - Global", "decodeURI(${uri})", "String"));
		STATIC_ASSIST_INFO.add(new AssistInfo(FUNCTION, "decodeURIComponent(s) - Global", "decodeURIComponent(${s})", "String"));
		STATIC_ASSIST_INFO.add(new AssistInfo(FUNCTION, "encodeURI(uri) - Global", "encodeURI(${uri})", "String"));
		STATIC_ASSIST_INFO.add(new AssistInfo(FUNCTION, "encodeURIComponent(s) - Global", "encodeURIComponent(${s})", "String"));
		STATIC_ASSIST_INFO.add(new AssistInfo(FUNCTION, "unescape(s) - Global", "unescape(${s})", "String"));
		STATIC_ASSIST_INFO.add(new AssistInfo(FUNCTION, "eval(code) - Global", "eval(${code})", "Object"));
		STATIC_ASSIST_INFO.add(new AssistInfo(FUNCTION, "isNaN(n) - Global", "isNaN(${n})"));
		STATIC_ASSIST_INFO.add(new AssistInfo(FUNCTION, "parseFloat(s) - Global", "parseFloat(${s})", "Number"));
		STATIC_ASSIST_INFO.add(new AssistInfo(FUNCTION, "parseInt(s) - Global", "parseInt(${s})", "Number"));

		for(int i=0;i<CLASSES.length;i++){
			STATIC_ASSIST_INFO.add(new AssistInfo(OBJECT, CLASSES[i], CLASSES[i]));
		}

		AssistInfo[] math = {
				new AssistInfo(VARIABLE, "E - Math", "E", "Number"),
				new AssistInfo(VARIABLE, "LN10 - Math", "LN10", "Number"),
				new AssistInfo(VARIABLE, "LN2 - Math", "LN2", "Number"),
				new AssistInfo(VARIABLE, "LOG10E - Math", "LOG10E", "Number"),
				new AssistInfo(VARIABLE, "LOG2E - Math", "LOG2E", "Number"),
				new AssistInfo(VARIABLE, "PI - Math", "PI", "Number"),
				new AssistInfo(VARIABLE, "SQRT1_2 - Math", "SQRT1_2", "Number"),
				new AssistInfo(VARIABLE, "SQRT2 - Math", "SQRT2", "Number"),
				new AssistInfo(FUNCTION, "abs(n) - Math", "abs(${n})", "Number"),
				new AssistInfo(FUNCTION, "acos(x) - Math", "acos(${x})", "Number"),
				new AssistInfo(FUNCTION, "asin(x) - Math", "asin(${x})", "Number"),
				new AssistInfo(FUNCTION, "atan(x) - Math", "atan(${x})", "Number"),
				new AssistInfo(FUNCTION, "atan2(y, x) - Math", "atan2(${y}, ${x})", "Number"),
				new AssistInfo(FUNCTION, "ceil(n) - Math", "ceil(${n})", "Number"),
				new AssistInfo(FUNCTION, "cos(x) - Math", "cos(${x})", "Number"),
				new AssistInfo(FUNCTION, "exp(n) - Math", "exp(${n})", "Number"),
				new AssistInfo(FUNCTION, "floor(n) - Math", "floor(${n})", "Number"),
				new AssistInfo(FUNCTION, "log(n) - Math", "log(${n})", "Number"),
				new AssistInfo(FUNCTION, "max(x, y) - Math", "max(${x}, ${y})", "Number"),
				new AssistInfo(FUNCTION, "min(x, y) - Math", "min(${x}, ${y})", "Number"),
				new AssistInfo(FUNCTION, "pow(n, m) - Math", "pow(${n}, ${m})", "Number"),
				new AssistInfo(FUNCTION, "random() - Math", "random()", "Number"),
				new AssistInfo(FUNCTION, "round(n) - Math", "round(${n})", "Number"),
				new AssistInfo(FUNCTION, "sin(x) - Math", "sin(${x})", "Number"),
				new AssistInfo(FUNCTION, "sqrt(n) - Math", "sqrt(${n})", "Number"),
				new AssistInfo(FUNCTION, "tan(x) - Math", "tan(${x})", "Number"),
		};
		STATIC_MEMBERS.put("Math", math);

		AssistInfo[] object = {
				new AssistInfo(VARIABLE, "constructor - Object", "constructor"),
				new AssistInfo(VARIABLE, "prototype - Object", "prototype"),
				new AssistInfo(FUNCTION, "hasOwnProperty() - Object", "hasOwnProperty()"),
				new AssistInfo(FUNCTION, "isPrototyprOf() - Object", "isPrototyprOf()"),
				new AssistInfo(FUNCTION, "propertyIsEnumerable() - Object", "propertyIsEnumerable()"),
				new AssistInfo(FUNCTION, "toLocaleString() - Object", "toLocaleString()"),
				new AssistInfo(FUNCTION, "toSource() - Object", "toSource()"),
				new AssistInfo(FUNCTION, "toString() - Object", "toString()"),
				new AssistInfo(FUNCTION, "unwatch() - Object", "unwatch()"),
				new AssistInfo(FUNCTION, "valueOf() - Object", "valueOf()"),
				new AssistInfo(FUNCTION, "watch() - Object","watch()"),
		};
		INSTANCE_MEMBERS.put("Object", object);

		AssistInfo[] node = {
				new AssistInfo(VARIABLE, "firstChild - Node", "firstChild", "Node"),
				new AssistInfo(VARIABLE, "lastChild - Node", "lastChild", "Node"),
				new AssistInfo(VARIABLE, "nextSibling - Node", "nextSibling", "Node"),
				new AssistInfo(VARIABLE, "parentNode - Node", "parentNode", "Node"),
				new AssistInfo(VARIABLE, "previousSibling - Node", "previousSibling", "Node"),
				new AssistInfo(VARIABLE, "childNodes - Node", "childNodes", "NodeList"),
				new AssistInfo(VARIABLE, "nodeName - Node", "nodeName"),
				new AssistInfo(VARIABLE, "nodeType - Node", "nodeType"),
				new AssistInfo(VARIABLE, "nodeValue - Node", "nodeValue"),
				new AssistInfo(VARIABLE, "ownerDocument - Node", "ownerDocument", "HTMLDocument"),
				new AssistInfo(FUNCTION, "appendChild(newChild) - Node", "appendChild(${newChild})"),
				new AssistInfo(FUNCTION, "cloneNode(deep) - Node", "cloneNode(${deep})"),
				new AssistInfo(FUNCTION, "hasChildNodes() - Node", "hasChildNodes()"),
				new AssistInfo(FUNCTION, "insertBefore(newChild, refChild) - Node", "insertBefore(${newChild}, ${refChild})"),
				new AssistInfo(FUNCTION, "removeChild(oldChild) - Node", "removeChild(${oldChild})"),
		};
		INSTANCE_MEMBERS.put("Node", node);

		AssistInfo[] nodeList = {
				new AssistInfo(VARIABLE, "length - NodeList", "length"),
				new AssistInfo(FUNCTION, "item(index) - NodeList", "item(${index})", "Node"),
		};
		INSTANCE_MEMBERS.put("NodeList", nodeList);

		AssistInfo[] element = {
				new AssistInfo(VARIABLE, "firstChild - Node", "firstChild", "Node"),
				new AssistInfo(VARIABLE, "lastChild - Node", "lastChild", "Node"),
				new AssistInfo(VARIABLE, "nextSibling - Node", "nextSibling", "Node"),
				new AssistInfo(VARIABLE, "parentNode - Node", "parentNode", "Node"),
				new AssistInfo(VARIABLE, "previousSibling - Node", "previousSibling", "Node"),
				new AssistInfo(VARIABLE, "childNodes - Node", "childNodes", "NodeList"),
				new AssistInfo(VARIABLE, "nodeName - Node", "nodeName"),
				new AssistInfo(VARIABLE, "nodeType - Node", "nodeType"),
				new AssistInfo(VARIABLE, "nodeValue - Node", "nodeValue"),
				new AssistInfo(VARIABLE, "ownerDocument - Node", "ownerDocument", "HTMLDocument"),
				new AssistInfo(FUNCTION, "appendChild(newChild) - Node", "appendChild(${newChild})"),
				new AssistInfo(FUNCTION, "cloneNode(deep) - Node", "cloneNode(${deep})"),
				new AssistInfo(FUNCTION, "hasChildNodes() - Node", "hasChildNodes()"),
				new AssistInfo(FUNCTION, "insertBefore(newChild, refChild) - Node", "insertBefore(${newChild}, ${refChild})"),
				new AssistInfo(FUNCTION, "removeChild(oldChild) - Node", "removeChild(${oldChild})"),

				new AssistInfo(VARIABLE, "style - HTMLElement", "style"),
				new AssistInfo(VARIABLE, "id - HTMLElement", "id"),
				new AssistInfo(VARIABLE, "className - HTMLElement", "className"),
				new AssistInfo(VARIABLE, "offsetHeight - HTMLElement", "offsetHeight"),
				new AssistInfo(VARIABLE, "offsetLeft - HTMLElement", "offsetLeft"),
				new AssistInfo(VARIABLE, "offsetTop - HTMLElement", "offsetTop"),
				new AssistInfo(VARIABLE, "offsetWidth - HTMLElement", "offsetWidth"),
				new AssistInfo(VARIABLE, "attributes - HTMLElement", "attributes"),
				new AssistInfo(FUNCTION, "getAttribute(name) - HTMLElement", "getAttribute(${name})", "Attr"),
				new AssistInfo(FUNCTION, "hasChildNodes() - HTMLElement", "hasChildNodes()"),
				new AssistInfo(FUNCTION, "removeAttribute(name) - HTMLElement", "removeAttribute(${name})"),
				new AssistInfo(FUNCTION, "setAttribute(name, value) - HTMLElement", "setAttribute(${name}, ${value})"),
		};
		INSTANCE_MEMBERS.put("HTMLElement", element);

		AssistInfo[] attr = {
				new AssistInfo(VARIABLE, "name - Attr", "name", "String"),
				new AssistInfo(VARIABLE, "ownerElement - Attr", "ownerElement", "HTMLElement"),
				new AssistInfo(VARIABLE, "specified - Attr", "specified", "Boolean"),
				new AssistInfo(VARIABLE, "value - Attr", "value", "String"),
		};
		INSTANCE_MEMBERS.put("Attr", attr);

		AssistInfo[] document = {
				new AssistInfo(FUNCTION, "createAttribute(name) - HTMLDocument", "createAttribute(${name})", "Attr"),
				new AssistInfo(FUNCTION, "createCDATASection(data) - HTMLDocument", "createCDATASection(${data})"),
				new AssistInfo(FUNCTION, "createComment(data) - HTMLDocument", "createComment(${data})"),
				new AssistInfo(FUNCTION, "createDocumentFragment() - HTMLDocument", "createDocumentFragment()"),
				new AssistInfo(FUNCTION, "createElement(tagName) - HTMLDocument", "createElement(${tagName})", "HTMLElement"),
				new AssistInfo(FUNCTION, "createProcessingInstruction(target, data) - HTMLDocument", "createProcessingInstruction(${target}, ${data})"),
				new AssistInfo(FUNCTION, "createRange() - HTMLDocument", "createRange()"),
				new AssistInfo(FUNCTION, "createTextNode(data) - HTMLDocument", "createTextNode(${data})"),
				new AssistInfo(FUNCTION, "importNode(node, deep) - HTMLDocument", "importNode(${node}, ${deep})"),
				new AssistInfo(FUNCTION, "getElementById(id) - HTMLDocument", "getElementById(${id})", "HTMLElement"),
				new AssistInfo(FUNCTION, "getElementsByName(name) - HTMLDocument", "getElementsByName(${name})", "NodeList"),
				new AssistInfo(FUNCTION, "getElementsByTagName(tagName) - HTMLDocument", "getElementsByTagName(${tagName})", "NodeList"),
				new AssistInfo(VARIABLE, "alinkColor - HTMLDocument", "alinkColor"),
				new AssistInfo(VARIABLE, "bgColor - HTMLDocument", "bgColor"),
				new AssistInfo(VARIABLE, "body - HTMLDocument", "body", "HTMLElement"),
				new AssistInfo(VARIABLE, "cookie - HTMLDocument", "cookie"),
				new AssistInfo(VARIABLE, "defaultCharset - HTMLDocument", "defaultCharset"),
				new AssistInfo(VARIABLE, "domain - HTMLDocument", "domain"),
				new AssistInfo(VARIABLE, "forms - HTMLDocument", "forms"),
				new AssistInfo(VARIABLE, "images - HTMLDocument", "images"),
				new AssistInfo(VARIABLE, "links - HTMLDocument", "links"),
				new AssistInfo(VARIABLE, "defaultView - HTMLDocument", "defaultView", "Window"),
				new AssistInfo(VARIABLE, "defaultElement - HTMLDocument", "defaultElement", "HTMLElement"),
				new AssistInfo(VARIABLE, "fgColor - HTMLDocument", "fgColor"),
				new AssistInfo(VARIABLE, "fileCreatedDate - HTMLDocument", "fileCreatedDate"),
				new AssistInfo(VARIABLE, "fileModifiedDate - HTMLDocument", "fileModifiedDate"),
				new AssistInfo(VARIABLE, "fileSize - HTMLDocument", "fileSize"),
				new AssistInfo(VARIABLE, "lastMofified - HTMLDocument", "lastMofified"),
				new AssistInfo(VARIABLE, "linkColor - HTMLDocument", "linkColor"),
				new AssistInfo(VARIABLE, "readyState - HTMLDocument", "readyState"),
				new AssistInfo(VARIABLE, "referrer - HTMLDocument", "referrer"),
				new AssistInfo(VARIABLE, "styleSheets - HTMLDocument", "styleSheets"),
				new AssistInfo(VARIABLE, "title - HTMLDocument", "title"),
				new AssistInfo(VARIABLE, "URL - HTMLDocument", "URL"),
				new AssistInfo(VARIABLE, "vlinkColor - HTMLDocument", "vlinkColor"),
				new AssistInfo(FUNCTION, "clear() - HTMLDocument", "clear()"),
				new AssistInfo(FUNCTION, "close() - HTMLDocument", "close()"),
				new AssistInfo(FUNCTION, "getSelection() - HTMLDocument", "getSelection()"),
				new AssistInfo(FUNCTION, "open() - HTMLDocument", "open()"),
				new AssistInfo(FUNCTION, "routeEvent() - HTMLDocument", "routeEvent()"),
				new AssistInfo(FUNCTION, "write(msg) - HTMLDocument", "write(${msg})"),
				new AssistInfo(FUNCTION, "writeln(msg) - HTMLDocument", "writeln(${msg})"),
		};
		INSTANCE_MEMBERS.put("HTMLDocument", document);

		AssistInfo[] style = {
				new AssistInfo(VARIABLE, "background"),
				new AssistInfo(VARIABLE, "backgroundAttachment"),
				new AssistInfo(VARIABLE, "backgroundColor"),
				new AssistInfo(VARIABLE, "backgroundImage"),
				new AssistInfo(VARIABLE, "backgroundPosition"),
				new AssistInfo(VARIABLE, "backgroundRepeat"),
				new AssistInfo(VARIABLE, "border"),
				new AssistInfo(VARIABLE, "borderBottom"),
				new AssistInfo(VARIABLE, "borderBottomColor"),
				new AssistInfo(VARIABLE, "borderBottomStyle"),
				new AssistInfo(VARIABLE, "borderBottomWidth"),
				new AssistInfo(VARIABLE, "borderColor"),
				new AssistInfo(VARIABLE, "borderLeft"),
				new AssistInfo(VARIABLE, "borderLeftColor"),
				new AssistInfo(VARIABLE, "borderLeftStyle"),
				new AssistInfo(VARIABLE, "borderLeftWidth"),
				new AssistInfo(VARIABLE, "borderRight"),
				new AssistInfo(VARIABLE, "borderRightColor"),
				new AssistInfo(VARIABLE, "borderRightStyle"),
				new AssistInfo(VARIABLE, "borderRightWidth"),
				new AssistInfo(VARIABLE, "borderStyle"),
				new AssistInfo(VARIABLE, "borderTop"),
				new AssistInfo(VARIABLE, "borderTopColor"),
				new AssistInfo(VARIABLE, "borderTopStyle"),
				new AssistInfo(VARIABLE, "borderTopWidth"),
				new AssistInfo(VARIABLE, "clear"),
				new AssistInfo(VARIABLE, "clip"),
				new AssistInfo(VARIABLE, "color"),
				new AssistInfo(VARIABLE, "cursor"),
				new AssistInfo(VARIABLE, "display"),
				new AssistInfo(VARIABLE, "filter"),
				new AssistInfo(VARIABLE, "font"),
				new AssistInfo(VARIABLE, "fontFamily"),
				new AssistInfo(VARIABLE, "fontSize"),
				new AssistInfo(VARIABLE, "fontVariant"),
				new AssistInfo(VARIABLE, "fontWeight"),
				new AssistInfo(VARIABLE, "height"),
				new AssistInfo(VARIABLE, "left"),
				new AssistInfo(VARIABLE, "letterSpacing"),
				new AssistInfo(VARIABLE, "lineHeight"),
				new AssistInfo(VARIABLE, "lineStyle"),
				new AssistInfo(VARIABLE, "lineStyleImage"),
				new AssistInfo(VARIABLE, "lineStylePosition"),
				new AssistInfo(VARIABLE, "lineStyleType"),
				new AssistInfo(VARIABLE, "margin"),
				new AssistInfo(VARIABLE, "marginBottom"),
				new AssistInfo(VARIABLE, "marginLeft"),
				new AssistInfo(VARIABLE, "marginRight"),
				new AssistInfo(VARIABLE, "marginTop"),
				new AssistInfo(VARIABLE, "overflow"),
				new AssistInfo(VARIABLE, "padding"),
				new AssistInfo(VARIABLE, "paddingBottom"),
				new AssistInfo(VARIABLE, "paddingLeft"),
				new AssistInfo(VARIABLE, "paddingRight"),
				new AssistInfo(VARIABLE, "paddingTop"),
				new AssistInfo(VARIABLE, "pageBreakAfter"),
				new AssistInfo(VARIABLE, "pageBreakBefore"),
				new AssistInfo(VARIABLE, "position"),
				new AssistInfo(VARIABLE, "styleFloat"),
				new AssistInfo(VARIABLE, "textAlign"),
				new AssistInfo(VARIABLE, "textDecoration"),
				new AssistInfo(VARIABLE, "textDecorationBlink"),
				new AssistInfo(VARIABLE, "textDecorationLineThrough"),
				new AssistInfo(VARIABLE, "textDecorationNone"),
				new AssistInfo(VARIABLE, "textDecorationOverline"),
				new AssistInfo(VARIABLE, "textDecorationUnderline"),
				new AssistInfo(VARIABLE, "textIndent"),
				new AssistInfo(VARIABLE, "textTransform"),
				new AssistInfo(VARIABLE, "top"),
				new AssistInfo(VARIABLE, "verticalAlign"),
				new AssistInfo(VARIABLE, "visibility"),
				new AssistInfo(VARIABLE, "whiteSpace"),
				new AssistInfo(VARIABLE, "width"),
				new AssistInfo(VARIABLE, "wordBreak"),
				new AssistInfo(VARIABLE, "wordWrap"),
				new AssistInfo(VARIABLE, "zIndex"),
		};
		STATIC_MEMBERS.put("style", style);

		AssistInfo[] window = {
				new AssistInfo(VARIABLE, "location - Window", "location", "Location"),
				new AssistInfo(VARIABLE, "history - Window", "history", "History"),
				new AssistInfo(VARIABLE, "document - Window", "document", "HTMLDocument"),
				new AssistInfo(VARIABLE, "navigator - Window", "navigator", "Navigator"),
				new AssistInfo(VARIABLE, "screen - Window", "screen", "Screen"),
				new AssistInfo(FUNCTION, "alert(message) - Window", "alert(${message})", "void"),
				new AssistInfo(FUNCTION, "atob(s) - Window", "atob(${s})", "String"),
		};
		INSTANCE_MEMBERS.put("Window", window);
		for(AssistInfo info: window){
			STATIC_ASSIST_INFO.add(info);
		}

		AssistInfo[] history = {
				new AssistInfo(VARIABLE, "length - History", "length"),
				new AssistInfo(FUNCTION, "back() - History", "back()"),
				new AssistInfo(FUNCTION, "forward() - History", "forward()"),
				new AssistInfo(FUNCTION, "go(n) - History", "go(${n})"),
		};
		INSTANCE_MEMBERS.put("History", history);

		AssistInfo[] location = {
				new AssistInfo(VARIABLE, "hash - Location", "hash"),
				new AssistInfo(VARIABLE, "host - Location", "host"),
				new AssistInfo(VARIABLE, "hostname - Location", "hostname"),
				new AssistInfo(VARIABLE, "href - Location", "href"),
				new AssistInfo(VARIABLE, "port - Location", "port"),
				new AssistInfo(VARIABLE, "protocol - Location", "protocol"),
				new AssistInfo(VARIABLE, "search - Location", "search"),
				new AssistInfo(FUNCTION, "reload() - Location", "reload()"),
				new AssistInfo(FUNCTION, "replace(url) - Location", "replace(${url})")
		};
		INSTANCE_MEMBERS.put("Location", location);

		AssistInfo[] screen = {
				new AssistInfo(VARIABLE, "colorDepth - Screen", "colorDepth"),
				new AssistInfo(VARIABLE, "fontSmoothingEnabled - Screen", "fontSmoothingEnabled"),
				new AssistInfo(VARIABLE, "height - Screen", "height"),
				new AssistInfo(VARIABLE, "updateInterval - Screen", "updateInterval"),
				new AssistInfo(VARIABLE, "width - Screen", "width"),
		};
		INSTANCE_MEMBERS.put("Screen", screen);

		AssistInfo[] date = {
				new AssistInfo(FUNCTION, "getFullYear() - Date", "getFullYear()"),
				new AssistInfo(FUNCTION, "setFullYear(year) - Date", "setFullYear(${year})"),
				new AssistInfo(FUNCTION, "getYear() - Date", "getYear()"),
				new AssistInfo(FUNCTION, "setYear(year) - Date", "setYear(${year})"),
				new AssistInfo(FUNCTION, "getMonth() - Date", "getMonth()"),
				new AssistInfo(FUNCTION, "setMonth(mon) - Date", "setMonth(${mon})"),
				new AssistInfo(FUNCTION, "getDate() - Date", "getDate()"),
				new AssistInfo(FUNCTION, "setDate(date) - Date", "setDate(${date})"),
				new AssistInfo(FUNCTION, "getHours() - Date", "getHours()"),
				new AssistInfo(FUNCTION, "setHours(hour) - Date", "setHours(${hour})"),
				new AssistInfo(FUNCTION, "getMinutes() - Date", "getMinutes()"),
				new AssistInfo(FUNCTION, "setMinutes(min) - Date", "setMinutes(${min})"),
				new AssistInfo(FUNCTION, "getSeconds() - Date", "getSeconds()"),
				new AssistInfo(FUNCTION, "setSeconds(sec) - Date", "setSeconds(${sec})"),
				new AssistInfo(FUNCTION, "getMilliseconds() - Date", "getMilliseconds()"),
				new AssistInfo(FUNCTION, "setMilliseconds(msec) - Date", "setMilliseconds(${msec})"),
				new AssistInfo(FUNCTION, "getTimezoneOffset() - Date", "getTimezoneOffset()"),
				new AssistInfo(FUNCTION, "toString() - Date", "toString()", "String"),
				new AssistInfo(FUNCTION, "toLocaleString() - Date", "toLocaleString()", "String"),
				new AssistInfo(FUNCTION, "toGMTString() - Date", "toGMTString()", "String"),
				new AssistInfo(FUNCTION, "toUTCString() - Date", "toUTCString()", "String"),
				new AssistInfo(FUNCTION, "toVarDate() - Date", "toVarDate()"),
				new AssistInfo(FUNCTION, "getTime() - Date", "getTime()"),
				new AssistInfo(FUNCTION, "setTime(time) - Date", "setTime(${time})"),
		};
		INSTANCE_MEMBERS.put("Date", date);

		AssistInfo[] dateStatic = {
				new AssistInfo(FUNCTION, "parse(str) - Date", "parse(${str})"),
				new AssistInfo(FUNCTION, "UTC(year, mon, day, hour, min, sec) - Date", "UTC(${year}, ${mon}, ${day}, ${hour}, ${min}, ${sec})"),
		};
		STATIC_MEMBERS.put("Date", dateStatic);

		AssistInfo[] string = {
				new AssistInfo(VARIABLE, "length - String", "length"),
				new AssistInfo(FUNCTION, "charAt(n) - String", "charAt(${n})", "String"),
				new AssistInfo(FUNCTION, "substr(from[, to]) - String", "substr(${from}, ${to})", "String"),
				new AssistInfo(FUNCTION, "substring(from[, to]) - String", "substring(${from}, ${to})", "String"),
				new AssistInfo(FUNCTION, "slice(from[, to]) - String", "slise(${from}, ${to})", "String"),
				new AssistInfo(FUNCTION, "split([separator[, limit]]) - String", "split(${separator}, ${limit})", "Array"),
				new AssistInfo(FUNCTION, "concat(str) - String", "concat(${str})", "String"),
				new AssistInfo(FUNCTION, "replace(regexp, newstr) - String", "replace(${regexp}, ${newstr})", "String"),
				new AssistInfo(FUNCTION, "toUpperCase() - String", "toUpperCase()"),
				new AssistInfo(FUNCTION, "toLowerCase() - String", "toLowerCase()"),
				new AssistInfo(FUNCTION, "indexOf(keyword[, from]) - String", "indexOf(${keyword}, ${from})"),
				new AssistInfo(FUNCTION, "lastIndexOf(keyword[, from]) - String", "lastIndexOf(${keyword}, ${from})"),
				new AssistInfo(FUNCTION, "match(regexp) - String", "match(${regexp})", "Array"),
				new AssistInfo(FUNCTION, "search(regexp) - String", "search(${regexp})"),
				new AssistInfo(FUNCTION, "charCodeAt(n) - String", "charCodeAt(${n})"),
				new AssistInfo(FUNCTION, "bold() - String", "bold()", "String"),
				new AssistInfo(FUNCTION, "italics() - String", "italics()", "String"),
				new AssistInfo(FUNCTION, "fixed() - String", "fixed()", "String"),
				new AssistInfo(FUNCTION, "big() - String", "big()", "String"),
				new AssistInfo(FUNCTION, "small() - String", "small()", "String"),
				new AssistInfo(FUNCTION, "blink() - String", "blink()", "String"),
				new AssistInfo(FUNCTION, "strike() - String", "strike()", "String"),
				new AssistInfo(FUNCTION, "sup() - String", "sup()", "String"),
				new AssistInfo(FUNCTION, "sub() - String", "sub()", "String"),
				new AssistInfo(FUNCTION, "fontcolor(color) - String", "fontcolor(${color})", "String"),
				new AssistInfo(FUNCTION, "fontsizr(size) - String", "fontsize(${size})", "String"),
				new AssistInfo(FUNCTION, "anchor(name) - String", "anchor(${name})", "String"),
				new AssistInfo(FUNCTION, "link(url) - String", "link(${url})", "String"),
		};
		INSTANCE_MEMBERS.put("String", string);

		AssistInfo[] function = {
				new AssistInfo(VARIABLE, "length - Function", "length"),
				new AssistInfo(FUNCTION, "call(thisArg, args) - Function", "call(${thisArg}, ${args})"),
				new AssistInfo(FUNCTION, "aplly(thisArg, argArray) - Function", "apply(${thisArg}, ${argArray})"),
		};
		INSTANCE_MEMBERS.put("Function", function);

		AssistInfo[] array = {
				new AssistInfo(VARIABLE, "length - Array", "length"),
				new AssistInfo(FUNCTION, "concat(values) - Array", "concat(${values})"),
				new AssistInfo(FUNCTION, "join([separator]) - Array", "join(${separator})"),
				new AssistInfo(FUNCTION, "pop() - Array", "pop()"),
				new AssistInfo(FUNCTION, "push(values) - Array", "push(${values})"),
				new AssistInfo(FUNCTION, "reverse() - Array", "reverse()"),
				new AssistInfo(FUNCTION, "slice(start, end) - Array", "slice(${start}, ${end})"),
				new AssistInfo(FUNCTION, "sort([orderfunc]) - Array", "sort(${orderfunc})"),
				new AssistInfo(FUNCTION, "splice(start, deleteCount, values) - Array", "splice(${start}, ${deleteCount}, ${values})"),
				new AssistInfo(FUNCTION, "unshift(values) - Array", "unshift(${values})"),
		};
		INSTANCE_MEMBERS.put("Array", array);

		AssistInfo[] number = {
				new AssistInfo(FUNCTION, "toExponential(digits) - Number", "toExponential(${digits})"),
				new AssistInfo(FUNCTION, "toFixed(digits) - Number", "toFixed(${digits})"),
				new AssistInfo(FUNCTION, "toPrecision(precision) - Number", "toPrecision(${precision})"),
				new AssistInfo(FUNCTION, "toString(radix) - Number", "toString(${redix})"),
		};
		INSTANCE_MEMBERS.put("Number", number);

		AssistInfo[] numberStatic = {
				new AssistInfo(VARIABLE, "MAX_VALUE - Number", "MAX_VALUE", "Number"),
				new AssistInfo(VARIABLE, "MIN_VALUE - Number", "MIN_VALUE", "Number"),
				new AssistInfo(VARIABLE, "NaN - Number", "NaN"),
				new AssistInfo(VARIABLE, "NEGATIVE_INFINITY - Number", "NEGATIVE_INFINITY", "Number"),
				new AssistInfo(VARIABLE, "POSITIVE_INFINITY - Number", "POSITIVE_INFINITY", "Number"),
		};
		STATIC_MEMBERS.put("Number", numberStatic);

		AssistInfo[] regexp = {
				new AssistInfo(VARIABLE, "global - RegExp", "global"),
				new AssistInfo(VARIABLE, "ignoreCase - RegExp", "ignoreCase"),
				new AssistInfo(VARIABLE, "lastIndex - RegExp", "lastIndex"),
				new AssistInfo(VARIABLE, "multiline - RegExp", "multiline"),
				new AssistInfo(VARIABLE, "source - RegExp", "source"),
				new AssistInfo(FUNCTION, "exec(string) - RegExp", "exec(${string})", "Array"),
				new AssistInfo(FUNCTION, "test(string) - RegExp", "test(${string})", "Array"),
		};
		INSTANCE_MEMBERS.put("RegExp", regexp);

	}

	private List<IAdditionalJavaScriptCompleter> completers = new ArrayList<IAdditionalJavaScriptCompleter>();
//	private List<JavaScriptModel> libraryModels = new ArrayList<JavaScriptModel>();

	/**
	 * Returns source code to parse from <code>ITextViewer</code>.
	 * <p>
	 * If you want to use this class with the your own editor
	 * which supports the document contains JavaScript such as HTML/JSP,
	 * override this method as returns only JavaScript code.
	 *
	 * @param viewer <code>ITextViewer</code>
	 * @return JavaScript source code
	 */
	protected String getSource(ITextViewer viewer){
		return viewer.getDocument().get();
	}

	/**
	 * Returns <code>AssistInfo[]</code> from <code>INSTANCE_MEMBERS</code>
	 * and <code>IAdditionalJavaScriptCompleter</code>s.
	 */
	private AssistInfo[] getInstanceMembers(String className, JavaScriptModel model){
		List<AssistInfo> result = new ArrayList<AssistInfo>();
		AssistInfo[] info = (AssistInfo[])INSTANCE_MEMBERS.get(className);
		if(info!=null){
			for(int i=0;i<info.length;i++){
				result.add(info[i]);
			}
		}

		for(JavaScriptElement element: model.getChildren()){
			if(element instanceof JavaScriptContext && element.getName().equals(className)){
				for(JavaScriptProperty property: ((JavaScriptContext) element).getProperties()){
					if(!property.isStatic()){
						if(property.getType().equals("Function")){
							AssistInfo assistInfo = new AssistInfo(FUNCTION,
									property.toString() + " - " + element.getName(), property.getReplaceString());
							if(property.getJsDoc() != null){
								assistInfo.setAdditionalInfo(property.getJsDoc().text);
							}
							result.add(assistInfo);
						} else {
							AssistInfo assistInfo = new AssistInfo(VARIABLE,
									property.toString() + " - " + element.getName(), property.getName());
							result.add(assistInfo);
						}
					}
				}
			}
		}

		for(int i=0;i<completers.size();i++){
			IAdditionalJavaScriptCompleter completer = completers.get(i);
			AssistInfo[] additionalInfo = (AssistInfo[]) completer.getInstanceMembers().get(className);
			if(additionalInfo != null){
				for(int j=0;j<additionalInfo.length;j++){
					result.add(additionalInfo[j]);
				}
			}
		}

		if(result.size()==0){
			return null;
		}
		return result.toArray(new AssistInfo[result.size()]);
	}

	/**
	 * Returns <code>AssistInfo[]</code> from <code>STATIC_MEMBERS</code>
	 * and <code>IAdditionalJavaScriptCompleter</code>s.
	 */
	private AssistInfo[] getStaticMembers(String className, JavaScriptModel model){
		List<AssistInfo> result = new ArrayList<AssistInfo>();
		AssistInfo[] info = (AssistInfo[])STATIC_MEMBERS.get(className);
		if(info!=null){
			for(int i=0;i<info.length;i++){
				result.add(info[i]);
			}
		}

		for(JavaScriptElement element: model.getChildren()){
			if(element instanceof JavaScriptContext && element.getName().equals(className)){
				for(JavaScriptProperty property: ((JavaScriptContext) element).getProperties()){
					if(property.isStatic()){
						if(property.getType().equals("Function")){
							AssistInfo assistInfo = new AssistInfo(FUNCTION,
									property.toString() + " - " + element.getName(), property.getReplaceString());
							result.add(assistInfo);
						} else {
							AssistInfo assistInfo = new AssistInfo(VARIABLE,
									property.toString() + " - " + element.getName(), property.getName());
							result.add(assistInfo);
						}
					}
				}
			}
		}

		for(int i=0;i<completers.size();i++){
			IAdditionalJavaScriptCompleter completer = completers.get(i);
			AssistInfo[] additionalInfo = (AssistInfo[]) completer.getStaticMembers().get(className);
			if(additionalInfo != null){
				for(int j=0;j<additionalInfo.length;j++){
					result.add(additionalInfo[j]);
				}
			}
		}
		if(result.size()==0){
			return null;
		}
		return result.toArray(new AssistInfo[result.size()]);
	}

	private JavaScriptModel createModel(String source){
		JavaScriptModel model = new JavaScriptModel(HTMLUtil.getActiveFile(), source);

		for(JavaScriptModel libModel: ModelManager.getInstance().getProjectLibModels(HTMLUtil.getActiveFile().getProject())){
			for(JavaScriptElement element: libModel.getChildren()){
				if(element instanceof JavaScriptVariable){
					model.add((JavaScriptVariable) element);

				} else if(element instanceof JavaScriptFunction){
					model.add((JavaScriptFunction) element);

				} else if(element instanceof JavaScriptProperty){
					model.add((JavaScriptProperty) element);
				}
			}
		}

		return model;
	}

	public ICompletionProposal[] computeCompletionProposals(ITextViewer viewer, int offset) {
		List<ICompletionProposal> proposalList = new ArrayList<ICompletionProposal>();
		String source = getSource(viewer);

		String[] words = getLastWord(viewer.getDocument().get(), offset);
		String last = words[0];
		String word = words[1];

		List<String> addedStrings = new ArrayList<String>();

		if(last.endsWith(".")){
			String objName = last.substring(0, last.length()-1);
			String objType = null;

			JavaScriptModel model = createModel(source);
			JavaScriptContext context = model.getContextFromOffset(offset);

			if(context != null){
				for(JavaScriptElement element: context.getVisibleElements()){
					if(element instanceof JavaScriptVariable &&
							((JavaScriptVariable) element).getName().equals(objName)){
						String type = ((JavaScriptVariable) element).getType();
						if(!type.equals("Object")){
							objType = type;
						}
						break;
					}
					if(element instanceof JavaScriptFunction &&
							((JavaScriptFunction) element).getName().equals(objName)){
						String type = ((JavaScriptFunction) element).getReturnType();
						if(type != null){
							if(!type.equals("Object")){
								objType = type;
							}
						}
						break;
					}
				}
			}

			// static property
			for(JavaScriptElement element: model.getChildren()){
				if(element instanceof JavaScriptContext){
					for(JavaScriptProperty property: ((JavaScriptContext) element).getProperties()){
						if(property.getName().equals(objName)){
							objType = property.getType();
							break;
						}
					}
				}
			}

			String type = null;
			if(objName.startsWith("\"") || objName.startsWith("'")){
				type = "String";
			}
			if(type == null){
				for(AssistInfo info: STATIC_ASSIST_INFO){
					if(info.getReplaceString().equals(objName) && info.getReturnType() != null){
						type = info.getReturnType();
						break;
					}
				}
			}
			if(type == null){
				for(AssistInfo[] infos: INSTANCE_MEMBERS.values()){
					for(AssistInfo info: infos){
						if((info.getReplaceString().equals(objName) || info.getReplaceString().startsWith(objName + "("))
								&& info.getReturnType() != null){
							type = info.getReturnType();
							break;
						}
					}
				}
			}
			if(type == null) {
				for(int i=0;i<completers.size();i++){
					IAdditionalJavaScriptCompleter completer = (IAdditionalJavaScriptCompleter) completers.get(i);
					type = completer.resolveType(objName);
					if(type != null){
						break;
					}
				}
			}
			// fuzzy type matching by variable name
			if(type == null){
				String varName = objName;
				if(varName.startsWith("$")){
					varName = varName.substring(1);
					for(JavaScriptElement element: model.getChildren()){
						if(element instanceof JavaScriptFunction && element.getName().toLowerCase().equals(varName.toLowerCase())){
							type = element.getName();
							break;
						}
					}
				}
			}
			if(type != null){
				objType = type;
			}

			if(objType != null && !objType.equals("Object")){
				addAssistInfo(offset, proposalList, word, getInstanceMembers(objType, model));
			} else {
				addAssistInfo(offset, proposalList, word, getStaticMembers(objName, model));
			}
			addAssistInfo(offset, proposalList, word, getInstanceMembers("Object", model));

		} else {
			addAssistInfo(offset, proposalList, word,
					STATIC_ASSIST_INFO.toArray(new AssistInfo[STATIC_ASSIST_INFO.size()]));
			for(int i=0;i<completers.size();i++){
				IAdditionalJavaScriptCompleter completer = (IAdditionalJavaScriptCompleter) completers.get(i);
				addAssistInfo(offset, proposalList, word, completer.getGlobalAssistInformations());
			}

			JavaScriptModel model = createModel(source);
			JavaScriptContext context = model.getContextFromOffset(offset);

			if(context != null){
				JavaScriptElement[] children = context.getVisibleElements();
				for(int i=0;i<children.length;i++){
					if(children[i].getName().toLowerCase().startsWith(word.toLowerCase())){
						int type = UNDEF;
						String replace = null;

						if(children[i] instanceof JavaScriptFunction){
							JavaScriptFunction func = (JavaScriptFunction) children[i];
							if(func.isMethod()){
								continue;
							}
							type = FUNCTION;
							replace = func.getReplaceString();

							if(!addedStrings.contains(replace)){
								HTMLTemplateManager manager = HTMLTemplateManager.getInstance();
								TemplateContextType contextType = manager.getContextTypeRegistry().getContextType(
										JavaScriptContextType.CONTEXT_TYPE);

								IEditorPart editor = HTMLUtil.getActiveEditor();

								Region region = new Region(offset - word.length(), word.length());
								Template template = new Template(
										children[i].toString(), ((WorkbenchPart) editor).getPartName(), contextType.getId(), replace, true);

								IDocument document = null;
								if(editor instanceof HTMLEditor){
									document = ((HTMLEditor) editor).getPaletteTarget().getDocumentProvider().getDocument(editor.getEditorInput());
								} else if(editor instanceof AbstractDecoratedTextEditor){
									document = ((AbstractDecoratedTextEditor) editor).getDocumentProvider().getDocument(editor.getEditorInput());
								}

								TemplateContext templateContext = new DocumentTemplateContext(
										contextType, document, region.getOffset(), region.getLength());

								ICompletionProposal proposal = new JavaScriptFunctionProposal(
										template, templateContext, region, getImageFromType(type), func.getDescription());

								proposalList.add(proposal);
								addedStrings.add(replace);
							}

							if(func.getProperties().length > 0){
								type = VARIABLE;
								replace = func.getName();

								if(!addedStrings.contains(replace)){
									ICompletionProposal proposal = new CompletionProposal(
											replace, offset - word.length(), word.length(), replace.length(),
											getImageFromType(type), replace, null, null);

									proposalList.add(proposal);
									addedStrings.add(replace);
								}
							}

						} else if(children[i] instanceof JavaScriptVariable){
							type = VARIABLE;
							replace = children[i].getName();

							if(!addedStrings.contains(replace)){
								ICompletionProposal proposal = new CompletionProposal(
										replace, offset - word.length(), word.length(), replace.length(),
										getImageFromType(type), children[i].toString(), null, null);

								proposalList.add(proposal);
								addedStrings.add(replace);
							}
						}
					}
				}
			}
		}

		for(ICompletionProposal template: super.computeCompletionProposals(viewer, offset)){
			proposalList.add(template);
		}

		// sort
		HTMLUtil.sortCompilationProposal(proposalList);
		ICompletionProposal[] prop = proposalList.toArray(new ICompletionProposal[proposalList.size()]);

		return prop;
	}

	private static void addAssistInfo(int offset, List<ICompletionProposal> proposalList,
			String word, AssistInfo[] info) {
		if(info!=null){
			for(int i=0;i<info.length;i++){
				if(info[i].replaceString.toLowerCase().startsWith(word.toLowerCase())){
					proposalList.add(info[i].createCompletionProposal(offset, word));
				}
			}
		}
	}

	/**
	 * Returns <code>Image</code> from the assist proposal type.
	 *
	 * @param type KEYWORD, VARIABLE or FUNCTION
	 * @return Image from the type
	 */
	private static Image getImageFromType(int type){
		switch(type){
			case KEYWORD:
				return HTMLPlugin.getDefault().getImageRegistry().get(HTMLPlugin.ICON_VALUE);
			case VARIABLE:
				return HTMLPlugin.getDefault().getImageRegistry().get(HTMLPlugin.ICON_VARIABLE);
			case FUNCTION:
				return HTMLPlugin.getDefault().getImageRegistry().get(HTMLPlugin.ICON_FUNCTION);
			case OBJECT:
				return HTMLPlugin.getDefault().getImageRegistry().get(HTMLPlugin.ICON_CLASS);
			default:
				return null;
		}
	}

	/**
	 * Cuts out the last word of caret position.
	 *
	 * @param viewer ITextViewer
	 * @param offset the caret offset
	 * @return the last word of caret position
	 */
	public static String[] getLastWord(String source, int offset){
		source = source.substring(0, offset);
		String last = "";

		int start = source.lastIndexOf('\n');
		if(start == -1){
			start = source.lastIndexOf('\r');
		}
		if(start == -1){
			start = 0;
		}

		Stack<String> stack = new Stack<String>();

		StringBuffer sb = new StringBuffer();
		for(int i = start; i < offset; i++){
			char c = source.charAt(i);
			if(c == ')' && !stack.isEmpty()){
				sb.setLength(0);
				sb.append(stack.pop());
				continue;
			}
			if((Character.isWhitespace(c) || !Character.isJavaIdentifierPart(c)) && (c != '"' && c != '\'')){
				if(sb.length() != 0){
					if(c=='.'){
						sb.append(c);
					}
					last = sb.toString();
					sb.setLength(0);
					if(c == '('){
						stack.push(last);
					}
				}
			} else {
				sb.append(c);
			}
		}
		return new String[]{last, sb.toString()};
	}

	/**
	 * The structure for assist informations.
	 */
	public static class AssistInfo {

		private int type;
		private String displayString;
		private String replaceString;
		private String returnType;
		private String additionalInfo;

		public AssistInfo(int type, String string){
			this(type, string, string);
		}

		public AssistInfo(int type, String displayString, String replaceString){
			this(type, displayString, replaceString, null);
		}

		public AssistInfo(int type, String displayString, String replaceString, String returnType){
			this.type = type;
			this.displayString = displayString;
			this.replaceString = replaceString;
			this.returnType = returnType;
		}

		/**
		 * Sets the text which is displayed as additional information of the completion proposal.
		 *
		 * @param additionalInfo additional information of the completion proposal
		 */
		public void setAdditionalInfo(String additionalInfo){
			this.additionalInfo = additionalInfo;
		}

		public String getReplaceString(){
			return this.replaceString;
		}

		public String getReturnType(){
			return this.returnType;
		}

		public ICompletionProposal createCompletionProposal(int offset, String word){
			if(type == FUNCTION){
				Region region = new Region(offset - word.length(), word.length());

				HTMLTemplateManager manager = HTMLTemplateManager.getInstance();
				TemplateContextType contextType = manager.getContextTypeRegistry().getContextType(
						JavaScriptContextType.CONTEXT_TYPE);

				IEditorPart editor = HTMLUtil.getActiveEditor();
				IDocument document = null;
				if(editor instanceof HTMLEditor){
					document = ((HTMLEditor) editor).getPaletteTarget().getDocumentProvider().getDocument(editor.getEditorInput());
				} else {
					document = ((AbstractDecoratedTextEditor) editor).getDocumentProvider().getDocument(editor.getEditorInput());

				}

				TemplateContext context = new DocumentTemplateContext(contextType, document, region.getOffset(), region.getLength());

				String[] dim = displayString.split("-");
				String templateName = dim[0].trim();

				String description = "";
				if(dim.length > 1){
					description = dim[1].trim();
				}

				Template template = new Template(templateName, description, contextType.getId(), replaceString, true);

				return new JavaScriptFunctionProposal(template, context, region, getImageFromType(type), additionalInfo);

			} else {
				return new CompletionProposal(
						replaceString, offset - word.length(), word.length(), replaceString.length(),
						getImageFromType(type), displayString, null, null);
			}
		}
	}

	public IContextInformation[] computeContextInformation(ITextViewer viewer, int offset) {
		return new ContextInformation[0];
	}

	public char[] getCompletionProposalAutoActivationCharacters() {
		return new char[0];
	}

	public char[] getContextInformationAutoActivationCharacters() {
		return new char[0];
	}

	public String getErrorMessage() {
		return "error";
	}

	public IContextInformationValidator getContextInformationValidator() {
		return new ContextInformationValidator(this);
	}

	protected TemplateContextType getContextType(ITextViewer viewer, IRegion region) {
		HTMLTemplateManager manager = HTMLTemplateManager.getInstance();
		return manager.getContextTypeRegistry().getContextType(JavaScriptContextType.CONTEXT_TYPE);
	}

	/**
	 * Updates internal informations.
	 *
	 * @param file the editing file
	 */
	public void update(IFile file){
		try {
			HTMLProjectParams params = new HTMLProjectParams(file.getProject());

			String[] names = params.getJavaScriptCompleters();
			completers.clear();
			for(int i=0;i<names.length;i++){
				IAdditionalJavaScriptCompleter completer =
					AdditionalJavaScriptCompleterManager.getAdditionalJavaSCriptCompleter(names[i]);
				completers.add(completer);
			}
		} catch(Exception ex){
			HTMLPlugin.logException(ex);
		}
	}

}
