package tk.eclipse.plugin.jseditor.editors;

import java.io.File;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.Region;
import org.eclipse.jface.text.hyperlink.IHyperlink;
import org.eclipse.jface.text.hyperlink.IHyperlinkDetector;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.ide.IGotoMarker;

import tk.eclipse.plugin.htmleditor.HTMLPlugin;
import tk.eclipse.plugin.htmleditor.HTMLUtil;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptCallable;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptContext;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptElement;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptModel;
import tk.eclipse.plugin.jseditor.editors.model.ModelManager;

/**
 * <code>IHyperlinkDetector</code> for JavaScript.
 * <p>
 * This detector detects available functions and variables
 * at the caret position.
 *
 * @author Naoki Takezoe
 */
public class JavaScriptHyperlinkDetector implements IHyperlinkDetector {

	/**
	 * Returns hyperlinks or null.
	 */
	public IHyperlink[] detectHyperlinks(ITextViewer textViewer,
			IRegion region, boolean canShowMultipleHyperlinks) {

		String source = textViewer.getDocument().get();
		int offset = region.getOffset();

		// extracts the word at the caret position
		String word = String.valueOf(source.charAt(offset));
		int index = offset -1;
		while(index >= 0){
			char c = source.charAt(index);
			if(Character.isJavaIdentifierPart(c)){
				word = c + word;
				index--;
			} else {
				break;
			}
		}
		index = offset + 1;
		offset = offset - word.length() + 1;
		while(source.length()>index){
			char c = source.charAt(index);
			if(Character.isJavaIdentifierPart(c)){
				word = word + c;
				index++;
			} else if(c=='('){
				break;
			} else {
				return null;
			}
		}

		ModelManager manager = ModelManager.getInstance();

		// itselfs
		{
			JavaScriptModel model = manager.getModel(HTMLUtil.getActiveFile());
			JavaScriptElement element = findElement(model, word);
			if(element != null && element instanceof JavaScriptCallable){
				IRegion hyperlinkRegion = new Region(offset, word.length());
				return new IHyperlink[]{
						new JavaScriptHyperlink(hyperlinkRegion, textViewer, ((JavaScriptCallable) element).getStartOffset())
				};
			}
		}

		// libraries
		{
			for(JavaScriptModel model: manager.getProjectLibModels(HTMLUtil.getActiveFile().getProject())){
				JavaScriptElement element = findElement(model, word);
				if(element != null && element instanceof JavaScriptCallable){
					IRegion hyperlinkRegion = new Region(offset, word.length());
					return new IHyperlink[]{
							new JavaScriptHyperlink(hyperlinkRegion, manager.getResourceByModel(model),
									((JavaScriptCallable) element).getStartOffset())
					};
				}
			}
		}

		return null;
	}

	private static JavaScriptElement findElement(JavaScriptContext context, String word){
		JavaScriptElement[] children = context.getChildren();

		for(int i=0;i<children.length;i++){
			if(children[i] instanceof JavaScriptCallable){
				if(children[i].getName().equals(word)){
					return children[i];
				}
			}

			if(children[i] instanceof JavaScriptContext){
				JavaScriptElement element = findElement((JavaScriptContext) children[i], word);
				if(element != null){
					return element;
				}
			}
		}

		return null;
	}

	/**
	 * <code>IHyperlink</code> implementation to jump the target resource.
	 */
	private class JavaScriptHyperlink implements IHyperlink {

		private IRegion region;
		private Object resource;
		private int beginOffset;

		/**
		 * The constructor.
		 *
		 * @param region the hyperlink region
		 * @param resource the target resource
		 *   (<code>IFile</code>, <code>ITextViewer</code> or <code>java.io.File</code>)
		 * @param beginOffset the begin offset
		 */
		public JavaScriptHyperlink(IRegion region, Object resource, int beginOffset){
			this.region = region;
			this.resource = resource;
			this.beginOffset = beginOffset;
		}

		public IRegion getHyperlinkRegion() {
			return region;
		}

		public String getTypeLabel() {
			return null;
		}

		public String getHyperlinkText() {
			return null;
		}

		/**
		 * TODO <code>java.io.File</code> is unsupported.
		 */
		public void open() {
			// IFile
			if(resource instanceof IFile){
				IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
				try {
					IEditorPart editor = IDE.openEditor(page, (IFile) resource, true);
					IGotoMarker gotoMarker = (IGotoMarker) editor.getAdapter(IGotoMarker.class);
					if(gotoMarker!=null){
						IMarker marker= ((IFile) resource).createMarker(IMarker.TEXT);
						marker.setAttribute(IMarker.CHAR_START, beginOffset);
						marker.setAttribute(IMarker.CHAR_END, beginOffset);
						gotoMarker.gotoMarker(marker);
						marker.delete();
					}
				} catch (Exception ex) {
					HTMLPlugin.logException(ex);
				}
			}
			if(resource instanceof ITextViewer){
				ITextViewer viewer = (ITextViewer) resource;
				viewer.setSelectedRange(beginOffset, 0);

				try {
					int line = viewer.getDocument().getLineOfOffset(beginOffset);
					if(viewer.getTopIndex() > line || viewer.getBottomIndex() < line){
						viewer.setTopIndex(line);
					}
				} catch(BadLocationException ex){
					;
				}
			}
			// java.io.File (external file)
			if(resource instanceof File){

			}
		}
	}

}
