package tk.eclipse.plugin.jseditor.editors;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.views.contentoutline.ContentOutlinePage;

import tk.eclipse.plugin.htmleditor.HTMLPlugin;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptContext;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptElement;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptFunction;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptModel;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptProperty;
import tk.eclipse.plugin.jseditor.editors.model.JavaScriptVariable;

/**
 * ContentOutlinePage implementation for JavaScriptEditor.
 *
 * @see tk.eclipse.plugin.jseditor.editors.JavaScriptEditor
 * @author Naoki Takezoe
 */
public class JavaScriptOutlinePage extends ContentOutlinePage {

	private JavaScriptModel model;
	private JavaScriptEditor editor;
	private String filterText = "";
	private boolean select = true;
	private JavaScriptSelectionChangedListener selectionChangedListener = new JavaScriptSelectionChangedListener();


	public JavaScriptOutlinePage(JavaScriptEditor editor) {
		super();
		this.editor = editor;
	}

	public void setSelect(boolean select){
		this.select = select;
	}

	public void setFilterText(String filterText){
		if(filterText == null){
			filterText = "";
		}
		this.filterText = filterText;
		getTreeViewer().refresh();
		getTreeViewer().expandAll();

		ITreeContentProvider provider = (ITreeContentProvider) getTreeViewer().getContentProvider();
		Object[] children = (Object[]) provider.getChildren(model);
		if(children.length > 0){
			getViewer().setSelection(new StructuredSelection(children[0]), true);
		}
	}

	public void createControl(Composite parent) {
		super.createControl(parent);

		IFile file = null;
		IEditorInput input = editor.getEditorInput();
		if(input instanceof IFileEditorInput){
			file = ((IFileEditorInput) input).getFile();
		}

		model = new JavaScriptModel(file, editor.getDocumentProvider().getDocument(editor.getEditorInput()).get());
		TreeViewer viewer = getTreeViewer();
		viewer.setContentProvider(new JavaScriptContentProvider());
		viewer.setLabelProvider(new JavaScriptLabelProvider());
		viewer.addSelectionChangedListener(selectionChangedListener);
		viewer.setInput(model);
		update();
	}

	public TreeViewer getViewer(){
		return getTreeViewer();
	}

	public void setSelection(int offset){
		if(model != null){
			try {
				getTreeViewer().removeSelectionChangedListener(selectionChangedListener);
				JavaScriptContext context = model.getContextFromOffset(offset);
				setSelection(new StructuredSelection(context));
				getTreeViewer().addSelectionChangedListener(selectionChangedListener);
			} catch(Exception ex){
				;
			}
		}
	}

	public void update(){
		try {
			model.update(editor.getDocumentProvider().getDocument(editor.getEditorInput()).get());
			getTreeViewer().refresh();
		} catch(Throwable t){
		}
	}

	public void selectSelection(){
		IStructuredSelection sel = (IStructuredSelection) getViewer().getSelection();
		JavaScriptElement element = (JavaScriptElement) sel.getFirstElement();
		if(element != null){
//			if(element instanceof JavaScriptCallable){
//				// TODO for Debug
//				editor.selectAndReveal(element.getOffset(),
//						((JavaScriptCallable) element).getEndOffset() - element.getOffset());
//			} else {
				editor.selectAndReveal(element.getOffset(), 0);
//			}
		}
	}

	/** ITreeContentProvider implementation for JavaScriptOutlinePage. */
	private class JavaScriptContentProvider implements ITreeContentProvider {
		public Object[] getChildren(Object parentElement) {
			if(parentElement instanceof JavaScriptContext){
				List<Object> result = new ArrayList<Object>();
				JavaScriptElement[] children = ((JavaScriptContext) parentElement).getChildren();
				if(children != null){
					for(JavaScriptElement e: children){
						if(getChildren(e).length > 0){
							result.add(e);
						} else if(filterText.length() > 0 &&
								e.getName().toLowerCase().startsWith(filterText.toLowerCase())){
							result.add(e);
						} else if(filterText.length() == 0){
							result.add(e);
						}
					}
				}
				return result.toArray(new Object[result.size()]);
			}
			return new Object[0];
		}
		public Object getParent(Object element) {
			if(element instanceof JavaScriptContext){
				return ((JavaScriptContext)element).getParent();
			}
			return null;
		}
		public boolean hasChildren(Object element) {
			if(getChildren(element).length==0){
				return false;
			} else {
				return true;
			}
		}
		public Object[] getElements(Object inputElement) {
			return getChildren(inputElement);
		}
		public void dispose() {
		}
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
	}

	/** ISelectionChangedListener implementation for JavaScriptOutlinePage. */
	private class JavaScriptSelectionChangedListener implements ISelectionChangedListener {
		public void selectionChanged(SelectionChangedEvent event) {
			if(!select){
				return;
			}
			selectSelection();
		}
	}

	/** LabelProvider implementation for JavaScriptOutlinePage. */
	private class JavaScriptLabelProvider extends LabelProvider {
		public Image getImage(Object element){
			if(element instanceof JavaScriptFunction){
				JavaScriptFunction func = (JavaScriptFunction) element;
				if(func.getProperties().length == 0){
					return HTMLPlugin.getDefault().getImageRegistry().get(HTMLPlugin.ICON_FUNCTION);
				} else {
					return HTMLPlugin.getDefault().getImageRegistry().get(HTMLPlugin.ICON_CLASS);
				}
			}
			if(element instanceof JavaScriptVariable){
				return HTMLPlugin.getDefault().getImageRegistry().get(HTMLPlugin.ICON_VARIABLE);
			}
			if(element instanceof JavaScriptProperty){
				JavaScriptProperty prop = (JavaScriptProperty) element;
				if(prop.getType().equals("Function")){
					return HTMLPlugin.getDefault().getImageRegistry().get(HTMLPlugin.ICON_FUNCTION);
				} else {
					return HTMLPlugin.getDefault().getImageRegistry().get(HTMLPlugin.ICON_VARIABLE);
				}
			}
			return null;
		}
	}
}
