package tk.eclipse.plugin.jseditor.editors.model;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IFile;

import tk.eclipse.plugin.htmleditor.HTMLProjectParams;
import tk.eclipse.plugin.jseditor.editors.JavaScriptAssistProcessor;
import tk.eclipse.plugin.jseditor.editors.JavaScriptUtil;
import tk.eclipse.plugin.jseditor.editors.JavaScriptAssistProcessor.AssistInfo;
import tk.eclipse.plugin.jseditor.editors.JsDocParser.JsDoc;
import tk.eclipse.plugin.jseditor.editors.additional.AdditionalJavaScriptCompleterManager;
import tk.eclipse.plugin.jseditor.editors.additional.IAdditionalJavaScriptCompleter;

/**
 *
 * @author Naoki Takezoe
 */
public class JavaScriptModel implements JavaScriptContext {

	private List<JavaScriptElement> children = new ArrayList<JavaScriptElement>();
	private List<JavaScriptComment> comments = new ArrayList<JavaScriptComment>();
	private JavaScriptContext context;
	private int endOffset;
	private IFile file;

	public JavaScriptModel(IFile file, String source){
		this.file = file;
		update(source);
	}

	public int getStartOffset(){
		return 0;
	}

	public int getEndOffset(){
		return endOffset;
	}

	public void add(JavaScriptProperty prop){
		this.children.add(prop);
	}

	public JavaScriptProperty[] getProperties(){
		List<JavaScriptProperty> list = new ArrayList<JavaScriptProperty>();
		for(JavaScriptElement element: getChildren()){
			if(element instanceof JavaScriptProperty){
				list.add(JavaScriptProperty.class.cast(element));
			}
		}
		return list.toArray(new JavaScriptProperty[list.size()]);
	}

	public JavaScriptContext getContextFromOffset(int offset){
		return getContextFromOffset(this, offset, true);
	}

	private JavaScriptContext getContextFromOffset(JavaScriptContext context, int offset, boolean topLevel){
		JavaScriptElement[] children = context.getChildren();
		for(int i=0;i<children.length;i++){
			if(children[i] instanceof JavaScriptContext){
				JavaScriptContext result = getContextFromOffset((JavaScriptContext) children[i], offset, false);
				if(result != null){
					return result;
				}
			}
		}

		if(context.getStartOffset() < offset && context.getEndOffset() > offset){
			return context;
		}

		return topLevel ? this : null;
	}

	/**
	 * Updates model structure by the specified source code.
	 *
	 * @param source JavaScript
	 */
	public void update(String source){
		this.children.clear();
		this.comments.clear();
		this.endOffset = source.length();
		context = this;

		boolean whitespace = true;
		char quote = 0;
		boolean escape = false;
		int brace = 0;

		for(int i=0;i<source.length();i++){
			char c = source.charAt(i);
			// TODO String literal
			if((c == '"' || c == '\'') && (quote == 0 || quote == c)){
				if(!escape){
					quote = (quote == c ? 0 : c);
				}
				escape = false;
				continue;
			}
			if(quote != 0){
				escape = (c == '\\');
				continue;
			}
			// skip comment
			if(c == '/' && source.length() > i+1){
				char nc = source.charAt(i+1);
				if(nc == '/'){
					int start = i;
					while(nc!='\r' && nc!='\n' && source.length() > i + 2){
						i++;
						nc = source.charAt(i + 1);
					}
					comments.add(new JavaScriptComment(start, i+1, source.substring(start, i + 1)));
				}
				if(nc == '*'){
					int start = i;
					i = source.indexOf("*/", i);
					if(i==-1){
						break;
					}
					comments.add(new JavaScriptComment(start, i+2, source.substring(start, i + 2)));
				}
			}
			// var
			if(whitespace && c == 'v'){
				int result = parseVariable(source, i);
				if(result!=0){
					whitespace = true;
					i += result;
					continue;
				}
			}
			// function
			if(whitespace && c == 'f'){
				int result = parseFunction(source, i);
				if(result != 0){
					whitespace = true;
					i += result;
					continue;
				}
			}
			if(c == '{'){
				brace++;
			}
			// end function
			if(c == '}'){
				if(context.getParent() != null && brace == 0){
					if(context instanceof JavaScriptFunction){
						((JavaScriptFunction)context).setEndOffset(i);
						context = context.getParent();
					}
				} else if(brace > 0){
					brace--;
				}
			}
			if(c == ';' && i > 0 && source.charAt(i - 1) == '}'){
				if(context.getParent() != null && brace == 0){
					if(context instanceof JavaScriptProperty){
						((JavaScriptProperty)context).setEndOffset(i);
						context = context.getParent();
					}
				} else if(brace > 0){
					brace--;
				}
			}
			// property
			// TODO Refactoring
			if(c == '='){
				// extract name
				boolean flag = false;
				int backCount = 0;
				StringBuilder sb = new StringBuilder();
				for(int j = i - 1; j >= 0; j--){
					backCount++;
					char prev = source.charAt(j);
					if(JavaScriptUtil.isWhitespace(prev)){
						if(flag == false){
							continue;
						} else {
							break;
						}
					} else if(flag == false){
						sb.insert(0, prev);
						flag = true;
					} else {
						sb.insert(0, prev);
					}
				}
				String name = sb.toString();
				if(name.startsWith("this.")){
					String replace = name.substring(5);
					String varSource = source.substring(i + 1).trim();
					JavaScriptProperty prop = new JavaScriptProperty(resolveType(varSource), replace, i, false);

					JsDoc jsDoc = JavaScriptUtil.extractJsDoc(source, i - backCount);
					prop.setJsDoc(jsDoc);

					context.add(prop);

					// extract function arguments
					if(prop.getType().equals("Function")){
						prop.setArguments(JavaScriptUtil.extractAnonFuncArguments(varSource));
					}

					prop.setParent(context);
					context = prop;

				} else {
					for(JavaScriptElement element: getVisibleElements()){
						if(element instanceof JavaScriptContext && name.startsWith(element.getName() + ".")){
							String replace = name.substring(element.getName().length() + 1);
							String varSource = source.substring(i + 1).trim();
							if(replace.startsWith("prototype.")){
								// instance members
								replace = replace.substring(10);
								JavaScriptProperty prop = new JavaScriptProperty(resolveType(varSource), replace, i, false);

								// extracts function arguments
								if(prop.getType().equals("Function")){
									prop.setArguments(JavaScriptUtil.extractAnonFuncArguments(varSource));
									prop.setJsDoc(JavaScriptUtil.extractJsDoc(source, i - backCount));

									prop.setParent(context);
									context = prop;
								}

								((JavaScriptContext) element).add(prop);
								break;
							}
							// static members
							JavaScriptProperty prop = new JavaScriptProperty(resolveType(varSource), replace, i, true);

							// extracts function arguments
							if(prop.getType().equals("Function")){
								prop.setArguments(JavaScriptUtil.extractAnonFuncArguments(varSource));
								prop.setJsDoc(JavaScriptUtil.extractJsDoc(source, i - backCount));

								prop.setParent(context);
								context = prop;
							}

							((JavaScriptContext) element).add(prop);
							break;
						}
					}
				}
			}

			// whitespace
			if(JavaScriptUtil.isWhitespace(c)){
				whitespace = true;
			} else {
				whitespace = false;
			}
		}
	}

	private static final Pattern VAR_PATTERN = Pattern.compile("var[\\s\r\n]+(.+?)[\\s\r\n]*?[;=]");
	private static final Pattern VAR_TYPE_PATTERN = Pattern.compile("new[\\s\r\n]*?(.+?)[\\s\r\n(]");

	private String resolveType(String varSource){
		String type = null;
		if(varSource.startsWith("function")){
			type = "Function";
		} else if(varSource.startsWith("[")){
			type = "Array";
		} else if(varSource.startsWith("/")){
			type = "RegExp";
		} else if(varSource.startsWith("\"") || varSource.startsWith("'")){
			type = "String";
		} else {
			Matcher typeMatcher = VAR_TYPE_PATTERN.matcher(varSource);
			if(typeMatcher.find() && typeMatcher.start()==0){
				type = typeMatcher.group(1).trim();
			}

			int index = varSource.indexOf(";");
			if(index >= 0){
				varSource = varSource.substring(0, index);
			}
			String[] words = JavaScriptAssistProcessor.getLastWord(varSource, varSource.length());
			varSource = words[1];

			if(type == null){
				// resolves global variables
				for(AssistInfo info: JavaScriptAssistProcessor.STATIC_ASSIST_INFO){
					if(info.getReplaceString().equals(varSource) && info.getReturnType() != null){
						type = info.getReturnType();
						break;
					}
				}
			}
			if(type == null){
				// resolves return type of function
				for(AssistInfo[] infos: JavaScriptAssistProcessor.INSTANCE_MEMBERS.values()){
					for(AssistInfo info: infos){
						if(info.getReplaceString().startsWith(varSource + "(") && info.getReturnType() != null){
							type = info.getReturnType();
							break;
						}
					}
				}
			}

			// TODO resolves by additional completers
			if(type == null && file != null) {
				try {
					HTMLProjectParams params = new HTMLProjectParams(file.getProject());
					String[] names = params.getJavaScriptCompleters();

					for(int i=0;i<names.length;i++){
						IAdditionalJavaScriptCompleter completer =
							AdditionalJavaScriptCompleterManager.getAdditionalJavaSCriptCompleter(names[i]);
						type = completer.resolveType(varSource);
						if(type != null){
							break;
						}
					}
				} catch(Exception ex){
				}
			}

			if(type == null){
				// scan other variables
				for(JavaScriptElement element: context.getChildren()){
					if(element instanceof JavaScriptVariable && element.getName().equals(varSource)){
						type = ((JavaScriptVariable) element).getType();
					}
				}
			}

		}
		return type;
	}

	private int parseVariable(String source, int position){
		if(source.indexOf("var", position) == position){
			String subSource = source.substring(position);
			int index = Math.min(subSource.indexOf(';'), subSource.indexOf('{'));
			if(index >= 0){
				subSource = subSource.substring(0, index + 1);
			}

			Matcher matcher = VAR_PATTERN.matcher(subSource);
			if(matcher.find() && matcher.start()==0){
				String name = matcher.group(1);
				String type = null;
				if(matcher.group().endsWith("=")){
					String varSource = subSource.substring(matcher.end()).trim();
					if(varSource.startsWith("{")){
						index = source.substring(position).indexOf("};");
					}
					type = resolveType(varSource);
				}
				if(type != null && type.equals("Function")){
					String args = JavaScriptUtil.extractAnonFuncArguments(subSource.substring(matcher.end()).trim());
					JavaScriptFunction func = new JavaScriptFunction(name, args, position, false);
					func.setParent(context);
					context.add(func);
					context = func;

				} else {
					JavaScriptVariable var = new JavaScriptVariable(type, name, position);
					context.add(var);
				}

				if(index >= 0){
					return index;
				}
				return matcher.end();
			}
		}
		return 0;
	}

	private static final Pattern FUNCTION_PATTERN = Pattern.compile(
			"function[\\s\r\n]+?(.+?)[\\s\r\n]*?\\((.*?)\\)[\\s\r\n]*?\\{", Pattern.DOTALL);

	private int parseFunction(String source, int position){
		if(source.indexOf("function", position) == position){
			Matcher matcher = FUNCTION_PATTERN.matcher(source.substring(position));
			if(matcher.find() && matcher.start() == 0){
				JsDoc jsDoc = JavaScriptUtil.extractJsDoc(source, position);
				String args = matcher.group(2).replaceAll("[\\s\r\n]*,[\\s\r\n]*",", ").trim();
				JavaScriptFunction func = new JavaScriptFunction(matcher.group(1), args, position, false);
				func.setJsDoc(jsDoc);
				func.setParent(context);
				context.add(func);
				context = func;
				return matcher.end();
			}
		}
		return 0;
	}


	public void add(JavaScriptFunction func) {
		this.children.add(func);
	}

	public void add(JavaScriptVariable var) {
		this.children.add(var);
	}

	public JavaScriptElement[] getChildren(){
		return this.children.toArray(new JavaScriptElement[this.children.size()]);
	}

	public JavaScriptElement[] getVisibleElements(){
		return getChildren();
	}

	public JavaScriptContext getParent(){
		return null;
	}

	public JavaScriptComment[] getComments(){
		return this.comments.toArray(new JavaScriptComment[this.comments.size()]);
	}

}
