package tk.eclipse.plugin.struts.editors;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.text.IDocument;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IGotoMarker;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.part.MultiPageEditorPart;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;

import tk.eclipse.plugin.htmleditor.HTMLUtil;
import tk.eclipse.plugin.htmleditor.editors.HTMLSourceEditor;
import tk.eclipse.plugin.htmleditor.editors.MultiPageEditorOutlinePage;
import tk.eclipse.plugin.struts.StrutsConfigModel2XML;
import tk.eclipse.plugin.struts.StrutsConfigXML2Model;
import tk.eclipse.plugin.struts.StrutsPlugin;
import tk.eclipse.plugin.struts.StrutsProject;
import tk.eclipse.plugin.struts.TilesDefsXML2Model;
import tk.eclipse.plugin.struts.Util;
import tk.eclipse.plugin.struts.editors.models.RootModel;
import tk.eclipse.plugin.xmleditor.editors.XMLEditor;

/**
 * This is the multi-page editor to editing struts-config.xml.
 * 
 * @author Naoki Takezoe
 */
public class MultiPageStrutsConfigEditor extends MultiPageEditorPart implements IResourceChangeListener {
	
	private MultiPageEditorOutlinePage outlinePage;
	private GraphicalStrutsConfigEditor graphicalEditor;
	private FormStrutsConfigEditor formEditor;
	private StrutsConfigXMLEditor textEditor;
	private ResourceBundle resource = StrutsPlugin.getDefault().getResourceBundle();
	
	public static final int FLOW_PAGE = 0;
	public static final int FORM_PAGE = 1;
	public static final int XML_PAGE = 2;
	
	/**
	 * The constructor.
	 */
	public MultiPageStrutsConfigEditor() {
		super();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		outlinePage = new MultiPageEditorOutlinePage();
	}
	
	/**
	 * Creates the graphical editor.
	 */
	private void createPage0() {
		try {
			if(getEditorInput() instanceof IFileEditorInput){
				graphicalEditor = new GraphicalStrutsConfigEditor(this);
				int index = addPage(graphicalEditor, getEditorInput());
				setPageText(index, resource.getString("editor.tab.flow"));
			}
		} catch (PartInitException e) {
			Util.openErrorDialog(e);
		}
	}
	
	/**
	 * Creates the form editor.
	 */
	private void createPage1() {
		try {
			if(getEditorInput() instanceof IFileEditorInput){
				formEditor = new FormStrutsConfigEditor(this);
				int index = addPage(formEditor, getEditorInput());
				setPageText(index, resource.getString("editor.tab.form"));
			}
		} catch (PartInitException e) {
			Util.openErrorDialog(e);
		}
	}	
	
	/**
	 * Creates the source editor.
	 */
	private void createPage2() {
		try {
			textEditor = new StrutsConfigXMLEditor();
			int index = addPage(textEditor, getEditorInput());
			setPageText(index, resource.getString("editor.tab.source"));
			
		} catch (PartInitException e) {
			Util.openErrorDialog(e);
		}
	}
	
//	protected IEditorSite createSite(IEditorPart editor) {
//		if(editor instanceof StrutsConfigXMLEditor){
//			return new SourceEditorSite(this,editor,getEditorSite());
//		} else {
//			return super.createSite(editor);
//		}
//	}
	
	protected void createPages() {
		createPage0();
		createPage1();
		createPage2();
		syncEditors(XML_PAGE);
	}
	
	public RootModel getRoot(){
	    return graphicalEditor.getRoot();
	}
	
	public void dispose() {
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		super.dispose();
	}
	
	/**
	 * Save struts-config.xml by the following sequence.
	 * 
	 * <ol>
	 *   <li>synchronization between the graphical editor and the source editor</li>
	 *   <li>save the content of the source editor as struts-config.xml</li>
	 *   <li>serizalize the model of the graphical editor</li>
	 *   <li>execute validation</li>
	 * </ol>
	 * 
	 * @param monitor IProgressMonitor
	 */
	public void doSave(IProgressMonitor monitor) {
	    syncEditors(getActivePage());
		
	    // (MVL) added: get Tiles information, if needed
	    if(graphicalEditor!=null){
			IProject project = ((FileEditorInput) textEditor.getEditorInput()).getFile().getProject();
			getRoot().setTilesRoot(TilesDefsXML2Model.createModel(project, getRoot()));
	    }
		// (MVL) end
		
		textEditor.doSave(monitor);
		setInput(textEditor.getEditorInput());
		if(graphicalEditor!=null){
			graphicalEditor.doSave(textEditor.getEditorInput());
		}
		setPartName(textEditor.getEditorInput().getName());
	}
	
	
	
	/**
	 * Save struts-config.xml by the following sequence.
	 * 
	 * <ol>
	 *   <li>synchronization between the graphical editor and the source editor</li>
	 *   <li>save the content of the source editor as struts-config.xml</li>
	 *   <li>serizalize the model of the graphical editor</li>
	 *   <li>execute validation</li>
	 * </ol>
	 */
	public void doSaveAs() {
		doSave(new NullProgressMonitor());
	}
	
	public void init(IEditorSite site, IEditorInput editorInput) throws PartInitException {
//		if (!(editorInput instanceof IFileEditorInput)){
//			throw new PartInitException(resource.getString("error.invalidinput"));
//		}
		super.init(site, editorInput);
		setPartName(editorInput.getName());
	}
	
	public boolean isSaveAsAllowed() {
		return true;
	}
	
	protected void pageChange(int newPageIndex) {
		if(isDirty()){
			if(newPageIndex==FLOW_PAGE){
				graphicalEditor.flushCommandStack();
			    syncEditors(XML_PAGE);
			} else if(newPageIndex==XML_PAGE){
			    syncEditors(FLOW_PAGE);
			}
		}
		super.pageChange(newPageIndex);
		outlinePage.setActiveEditor(getEditor(newPageIndex));
	}
	
	/**
	 * Synchronizes between the graphical editor and the source editor.
	 * 
	 * @param activePage Index of an active page.
	 * <ul>
	 *   <li><code>FLOW_PAGE</code> (0) - Graphical editor</li>
	 *   <li><code>FORM_PAGE</code> (1) - Form editor</li>
	 *   <li><code>XML_PAGE</code> (2) - Source editor</li>
	 * </ul>
	 */
	public void syncEditors(int activePage){
		if(graphicalEditor==null){
			return;
		}
		if(activePage==FLOW_PAGE){
			// the graphical editor to the source editor
			IFile file = ((FileEditorInput)graphicalEditor.getEditorInput()).getFile();
			StrutsProject strutsProject = new StrutsProject(file.getProject());
			String xml = StrutsConfigModel2XML.createStrutsConfig(graphicalEditor.getRoot(), strutsProject.getExtension());
			textEditor.setDiff(xml);
			formEditor.update();
			
		} else if(activePage==FORM_PAGE){
			IFile file = ((FileEditorInput)graphicalEditor.getEditorInput()).getFile();
			StrutsProject strutsProject = new StrutsProject(file.getProject());
			String xml = StrutsConfigModel2XML.createStrutsConfig(graphicalEditor.getRoot(), strutsProject.getExtension());
			textEditor.setDiff(xml);
			
		} else if(activePage==XML_PAGE){
			// the source editor to the graphical editor
			IEditorInput input = textEditor.getEditorInput();
			IDocument doc = textEditor.getDocumentProvider().getDocument(input);
			String xml = doc.get();
			try {
				IFile file = ((FileEditorInput)graphicalEditor.getEditorInput()).getFile();
				InputStream in = new ByteArrayInputStream(xml.getBytes(file.getCharset()));
				RootModel model = StrutsConfigXML2Model.createModel(in, graphicalEditor.getRoot(), file);
				graphicalEditor.setRoot(model);
			} catch(Exception ex){
				Util.logException(ex);
			}
			formEditor.update();
		}
	}
	
	public IDocument getDocument(){
		IEditorInput input = textEditor.getEditorInput();
		IDocument doc = textEditor.getDocumentProvider().getDocument(input);
		return doc;
	}
	
	/** First flag */
	boolean firstFlag = true;
	
	public void setFocus(){
		if(graphicalEditor!=null){
			// Open the property view and the outline view
			if(firstFlag){
				IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
				try {
					window.getActivePage().showView("org.eclipse.ui.views.PropertySheet");
					window.getActivePage().showView("org.eclipse.ui.views.ContentOutline");
				} catch(Exception ex){
					// TODO Exception is caused here in Eclipse 3.1
					//Util.logException(ex);
				}
				firstFlag = false;
			}
			
		    // (MVL) added: get Tiles information, if needed
			IProject project = ((FileEditorInput) textEditor.getEditorInput()).getFile().getProject();
			getRoot().setTilesRoot(TilesDefsXML2Model.createModel(project, getRoot()));
			// (MVL) end
		}		
		super.setFocus();
	}
	
	public void resourceChanged(final IResourceChangeEvent event){
		if(event.getType() == IResourceChangeEvent.POST_CHANGE){
			final IEditorInput input = textEditor.getEditorInput();
			if(input instanceof IFileEditorInput){
				Display.getDefault().asyncExec(new Runnable(){
					public void run(){
						IFile file = ((IFileEditorInput)input).getFile();
						if(!file.exists()){
							IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
							page.closeEditor(MultiPageStrutsConfigEditor.this, false);
						} else if(!getPartName().equals(file.getName())){
							setPartName(file.getName());
						} else if(HTMLUtil.contains(event.getDelta(), file)){
							updateGraphicalEditor();
						}
					}        
				});
			}
		}
	}
	
	@Override
	@SuppressWarnings("unchecked")
	public Object getAdapter(Class adapter) {
		if(IContentOutlinePage.class.equals(adapter) && graphicalEditor!=null){
			return outlinePage;
		} else if(IGotoMarker.class.equals(adapter)){
			setActivePage(1);
			return textEditor.getAdapter(adapter);
		} else if(HTMLSourceEditor.class.equals(adapter)){
			return textEditor;
		} else if(getActiveEditor()==textEditor){
			return textEditor.getAdapter(adapter);
		} else if(graphicalEditor!=null){
			return graphicalEditor.getAdapter(adapter);
		}
		return super.getAdapter(adapter);
	}
	
	/**
	 * Updates waring information of the graphical editor.
	 */
	private void updateGraphicalEditor(){
		if(graphicalEditor!=null){
			syncEditors(XML_PAGE); // source to model synchronization
			graphicalEditor.refresh();
		}
	}
	
	public XMLEditor getXMLEditor(){
		return this.textEditor;
	}
	
	public GraphicalStrutsConfigEditor getGraphicalEditor() {
		return this.graphicalEditor;
	}
	
    public void setActivePage(int pageIndex) {
        super.setActivePage(pageIndex);
    }
	
//	/** IEditorSite for the source editor. */
//	private class SourceEditorSite extends MultiPageEditorSite {
//		
//		private HTMLSourceEditor editor = null;
//		private IEditorSite site;
//		private ArrayList menuExtenders;
//		
//		public SourceEditorSite(MultiPageEditorPart multiPageEditor,IEditorPart editor,IEditorSite site) {
//			super(multiPageEditor, editor);
//			this.site = site;
//			this.editor = (HTMLSourceEditor)editor;
//		}
//		
//		public IEditorActionBarContributor getActionBarContributor() {
//			return site.getActionBarContributor();
//		}
//		
//		public void registerContextMenu(String menuId, MenuManager menuManager, ISelectionProvider selectionProvider) {
//			if(editor != null){
//				if (menuExtenders == null) {
//					menuExtenders = new ArrayList(1);
//				}
//				menuExtenders.add(new PopupMenuExtender(menuId, menuManager, selectionProvider, editor));
//			}
//		}
//		
//		public void dispose(){
//			if (menuExtenders != null) {
//				for (int i = 0; i < menuExtenders.size(); i++) {
//					((PopupMenuExtender)menuExtenders.get(i)).dispose();
//				}
//				menuExtenders = null;
//			}
//			super.dispose();
//		}
//	}
}
