;;; anthy.el -- Anthy

;; Copyright (C) 2001
;; Author: Yusuke Tabata<yusuke@kmc.kyoto-u.ac.jp>
;;         Tomoharu Ugawa
;; Keywords: japanese

;; This file is part of Anthy

;;; Commentary:
;;
;; $B$+$J4A;zJQ49%(%s%8%s(B Anthy$B$r(B emacs$B$+$i;H$&$?$a$N%W%m%0%i%`(B
;; Anthy$B%i%$%V%i%j$r;H$&$?$a$N%3%^%s%I(Banthy-agent$B$r5/F0$7$F!"(B
;; anthy-agent$B$H%Q%$%W$GDL?.$r$9$k$3$H$K$h$C$FJQ49$NF0:n$r(B
;; $B9T$&(B
;;
;; Funded by IPA$BL$F'%=%U%H%&%'%"AOB$;v6H(B 2001 10/10
;;
;; $B3+H/$O(Bemacs20.7$B>e$G9T$C$F$$$F(Bminor-mode
;; $B$b$7$/$O(Bleim$B$H$7$F$b;HMQ$G$-$k(B
;; (set-input-method 'japanese-anthy)
;;
;;
;;
;; 2001-11-16 EUC-JP -> ISO-2022-JP
;;
;; TODO
;;  multi-context$B2=(B
;;  minibufffer$B$N07$$(B
;;  modeline
;;

;;; Code:
;(setq debug-on-error t)

(defvar anthy-default-enable-enum-candidate-p t)

(defconst anthy-working-buffer " *anthy*")
(defvar anthy-agent-process nil)
;;
(defvar anthy-agent-command-list '("anthy-agent"))

;; face
(defvar anthy-hilight-face nil)
(defvar anthy-underline-face nil)
(copy-face 'highlight 'anthy-highlight-face)
(set-face-underline-p 'anthy-highlight-face t)
(copy-face 'underline 'anthy-underline-face)

;;
(defvar anthy-xemacs
  (if (featurep 'xemacs)
      t nil))
(if anthy-xemacs
    (require 'overlay))
;;
(defvar anthy-mode-map nil "*Anthy$B$N(BASCII$B%b!<%I$N%-!<%^%C%W(B")
(or anthy-mode-map
    (let ((map (make-keymap))
	  (i 33))
      (define-key map (char-to-string 10) 'anthy-insert)
      (while (< i 127)
	(define-key map (char-to-string i) 'anthy-insert)
	(setq i (+ 1 i)))
      (setq anthy-mode-map map)))

(defvar anthy-preedit-keymap nil "*Anthy$B$N(Bpreedit$B$N%-!<%^%C%W(B")
(or anthy-preedit-keymap
    (let ((map (make-keymap))
	  (i 0))
      (while (< i 128)
	(define-key map (char-to-string i) 'anthy-insert)
	(setq i (+ 1 i)))
      (if anthy-xemacs
	  (progn
	    (define-key map '(shift left) 'anthy-insert)
	    (define-key map '(shift right) 'anthy-insert)
	    (define-key map '(left) 'anthy-insert)
	    (define-key map '(right) 'anthy-insert))
	(progn
	  (define-key map [S-left] 'anthy-insert)
	  (define-key map [S-right] 'anthy-insert)
	  (define-key map [left] 'anthy-insert)
	  (define-key map [right] 'anthy-insert)))
      (setq anthy-preedit-keymap map)))

(defvar anthy-keyencode-alist
  '((1 . "(ctrl A)") ;; \C-a
    (2 . "(left)") ;; \C-b
    (4 . "(ctrl D)") ;; \C-d
    (5 . "(ctrl E)") ;; \C-e
    (6 . "(right)") ;; \C-f
    (7 . "(esc)") ;; \C-g
    (8 . "(ctrl H)") ;; \C-h
    (9 . "(shift left)") ;; \C-i
    (10 . "(ctrl J)")
    (11 . "(ctrl K)")
    (13 . "(enter)") ;; \C-m
    (14 . "(space)") ;; \C-n
    (15 . "(shift right)") ;; \C-o
    (16 . "(up)") ;; \C-p
    (32 . "(space)")
    (40 . "(opar)") ;; '('
    (41 . "(cpar)") ;; ')'
    (127 . "(ctrl H)")
    ;; emacs map
    (S-right . "(shift right)")
    (S-left . "(shift left)")
    (right . "(right)")
    (left . "(left)")
    (up . "(up)")
    ;; xemacs
    ((shift right) . "(shift right)")
    ((shift left) . "(shift left)")
    ((right) . "(right)")
    ((left) . "(left)")
    ((up) . "(up)")))

;;
(defvar anthy-mode-line-string-alist
  '(("hiragana" . " Anthy: $B$"(B")
    ("katakana" . " Anthy: $B%"(B")
    ("alphabet" . " Anthy: A")
    ("walphabet" . " Anthy: $B#A(B")))



;; buffer local variables
(defvar anthy-minor-mode nil)
(make-variable-buffer-local 'anthy-minor-mode)
(defvar anthy-mode nil)
(make-variable-buffer-local 'anthy-mode)
(defvar anthy-leim-active-p nil)
(make-variable-buffer-local 'anthy-leim-active-p)
(defvar anthy-saved-mode nil)
(make-variable-buffer-local 'anthy-saved-mode)
(defvar anthy-preedit "")
(make-variable-buffer-local 'anthy-preedit)
(defvar anthy-preedit-start 0)
(make-variable-buffer-local 'anthy-preedit-start)
(defvar anthy-preedit-overlays '())
(make-variable-buffer-local 'anthy-preedit-overlays)
(defvar anthy-mode-line-string " Anthy:")
(make-variable-buffer-local 'anthy-mode-line-string)
(defvar anthy-enum-candidate-p nil)
(make-variable-buffer-local 'anthy-enum-candidate-p)
(defvar anthy-enum-rcandidate-p nil)
(make-variable-buffer-local 'anthy-enum-rcandidate-p)
(defvar anthy-candidate-minibuffer "")
(make-variable-buffer-local 'anthy-candidate-minibuffer)
(defvar anthy-enum-candidate-list '())
(make-variable-buffer-local 'anthy-enum-candidate-list)
(defvar anthy-current-rkmap "hiragana")
(defvar anthy-enable-enum-candidate-p 
  (cons anthy-default-enable-enum-candidate-p nil))

;;; setup minor-mode
;; minor-mode-alist
(if (not
     (assq 'anthy-minor-mode minor-mode-alist))
    (setq minor-mode-alist
       (cons
	(cons 'anthy-minor-mode '(anthy-mode-line-string))
	minor-mode-alist)))
;; minor-mode-map-alist
(if (not
     (assq 'anthy-minor-mode minor-mode-map-alist))
    (setq minor-mode-map-alist
       (cons
	(cons 'anthy-minor-mode anthy-mode-map)
	minor-mode-map-alist)))

(defun anthy-process-sentinel (proc stat)
  (message stat)
  (setq anthy-agent-process nil))

;;; status
(defun anthy-update-mode-line ()
  (let ((a (assoc anthy-current-rkmap anthy-mode-line-string-alist)))
    (if a
      (setq anthy-mode-line-string (cdr a))))
  (force-mode-line-update))

;;; preedit control
(defun anthy-erase-preedit ()
  (if (> (string-width anthy-preedit) 0)
      (let* ((str anthy-preedit)
	     (len (length str))
	     (start anthy-preedit-start))
	(delete-region start (+ start len))
	(goto-char start)))
  (setq anthy-preedit "")
  (mapcar 'delete-overlay anthy-preedit-overlays)
  (setq anthy-preedit-overlays nil))

(defun anthy-select-face-by-attr (attr)
  (if (memq 'RV attr)
      'anthy-highlight-face
    'anthy-underline-face))

(defun anthy-enable-preedit-keymap ()
  (setcdr
   (assq 'anthy-minor-mode minor-mode-map-alist)
   anthy-preedit-keymap))

(defun anthy-disable-preedit-keymap ()
  (setcdr
   (assq 'anthy-minor-mode minor-mode-map-alist)
   anthy-mode-map)
  (anthy-update-mode-line))

;(defun anthy-ntimes-str (str n)
;  (let ((ans ""))
;    (while (> n 0)
;      (setq ans (concat ans str))
;      (setq n (1- n)))
;    ans))

(defun anthy-insert-preedit-segment (str attr)
  (let ((start (point))
	(end) (ol))
    (cond ((or (memq 'ENUM attr) (memq 'ENUMR attr))
	   (setq str (concat "<" str ">")))
         ; ((memq 'ENUM attr)
	 ;  (setq str 
	 ;	 (concat "<" (anthy-ntimes-str " " (string-width str)) ">")))
	  ((memq 'RV attr) (setq str (concat "[" str "]"))))
    (insert-and-inherit str)
    (setq end (point))
    (setq ol (make-overlay start end))
    (overlay-put ol 'face (anthy-select-face-by-attr attr))
    (setq anthy-preedit-overlays
	  (cons ol anthy-preedit-overlays))
    str))

(defvar anthy-select-candidate-keybind
  '((0 . "1")
    (1 . "2")
    (2 . "3")
    (3 . "4")
    (4 . "5")
    (5 . "6")
    (6 . "7")
    (7 . "8")
    (8 . "9")
    (9 . "0")))

(defun anthy-get-candidate (idx)
  (anthy-send-recv-command 
   (concat " GET_CANDIDATE "
	   (number-to-string idx) "\n")))

(defun anthy-layout-candidate (idx nr)
  (let ((w (frame-width))
	(endp nil)
	(i 0)
	(firstp t)
	(errorp nil))
    (setq anthy-candidate-minibuffer "")
    (setq anthy-enum-candidate-list '())
    (cond (anthy-enum-candidate-p
	   (while (and (not endp)
		       (< i 10)
		       (< (+ idx i) nr)
		       (not errorp))
	     (let ((repl (anthy-get-candidate (+ idx i))))
	       (if (listp repl)
		   (let ((str (concat anthy-candidate-minibuffer 
				      (if firstp "" " ") 
				      (cdr (assq
					    i anthy-select-candidate-keybind))
				      ":"
				      (car repl))))
		     (if (or firstp (<= (string-width str) w))
			 (progn
			   (setq anthy-candidate-minibuffer str)
			   (setq anthy-enum-candidate-list
				 (cons (cons i (+ idx i)) 
				       anthy-enum-candidate-list)))
		       (setq endp t)))
		 (setq errorp t)))
	     (setq firstp nil)
	     (setq i (1+ i)))
	   ;; set current candidate properly
	   (anthy-get-candidate
	    (cdr (car anthy-enum-candidate-list))))

	  (anthy-enum-rcandidate-p
	   (let ((cand-list '()))
	     (while (and (< i 10)
			 (<= i idx)
			 (> w 0)
			 (not errorp))
	       (let ((repl (anthy-get-candidate (- idx i))))
		 (if (listp repl)
		     (progn
		       (setq w (- w 
				  (if (= i 0) 0 1) ; ' '
				  (string-width (car repl))
				  1               ; ':'
				  (string-width
				   (cdr (assq 
					 i anthy-select-candidate-keybind)))))
		       (if (or (= i 0) (>= w 0))
			   (setq cand-list (cons (cons i (car repl))
						 cand-list))))
		   (setq errorp t)))
	       (setq i (1+ i)))
	     (if (or (>= i 10) (> i idx) (and (< w 0) (> i 0)))
		 (setq i (1- i)))
	     (let ((lst cand-list))
	       (while lst
		 (let ((c (car lst)))
		   (setq anthy-candidate-minibuffer
			 (concat anthy-candidate-minibuffer
				 (if (= (car c) i) "" " ")
				 (cdr (assq (- i (car c))
					    anthy-select-candidate-keybind))
				 ":"
				 (cdr c)))
		   (setq anthy-enum-candidate-list
			 (cons (cons (- i (car c)) (- idx (car c)))
			       anthy-enum-candidate-list)))
		 (setq lst (cdr lst)))))))
    (if (not errorp)
	(message anthy-candidate-minibuffer))))

(defun anthy-update-preedit (stat ps)
  (let ((cursor-pos nil)
	(num-candidate 0)
	(idx-candidate 0))
    ;; erase old preedit
    (anthy-erase-preedit)
    (anthy-disable-preedit-keymap)
    ;; insert new preedit
    (setq anthy-preedit-start (point))
    (setq anthy-enum-candidate-p nil)
    (setq anthy-enum-rcandidate-p nil)
    (if (member stat '(2 3 4))
	(progn
	  (setq anthy-preedit
		(concat anthy-preedit "|"))
	  (anthy-insert-preedit-segment "|" '())))
    (while ps
      (let ((cur (car ps)))
	(setq ps (cdr ps))
	(cond
	 ((eq cur 'cursor)
	  (setq cursor-pos (point)))
	 ((string-equal (car (cdr cur)) "")
	  nil)
	 (t
	  (let ((nr (car (cdr (cdr (cdr cur)))))
		(idx (car (cdr (cdr cur))))
		(str (car (cdr cur)))
		(attr (car cur)))
	    (setq str (anthy-insert-preedit-segment str attr))
	    (cond ((and (car anthy-enable-enum-candidate-p) (memq 'ENUM attr))
		   (setq anthy-enum-candidate-p t)
		   (setq idx-candidate idx)
		   (setq num-candidate nr))
		  ((and (car anthy-enable-enum-candidate-p) (memq 'ENUMR attr))
		   (setq anthy-enum-rcandidate-p t)
		   (setq idx-candidate idx)
		   (setq num-candidate nr)))
	    (setq anthy-preedit
		  (concat anthy-preedit str))
	    (if (and (member stat '(3 4)) (not (eq ps '())))
		(progn
		  (setq anthy-preedit
			(concat anthy-preedit "|"))
		  (anthy-insert-preedit-segment "|" '()))))))))
    ;; enum candidate
    (if (or anthy-enum-candidate-p anthy-enum-rcandidate-p)
	(anthy-layout-candidate idx-candidate num-candidate))
    ;; update preedit keymap
    (if (member stat '(2 3 4))
	(anthy-enable-preedit-keymap))
    (if cursor-pos (goto-char cursor-pos))))

(if anthy-xemacs
    (defun anthy-encode-key (ch)
      (let* ((ccode (char-to-int ch)) (c))
	(setq c (assoc ccode anthy-keyencode-alist))
	(if c
	    (cdr c)
	  (setq c (assoc c anthy-keyencode-alist))
	  (if c
	      (cdr c)
	    (char-to-string ch)))))
  (defun anthy-encode-key (ch)
    (let ((c (assoc ch anthy-keyencode-alist)))
      (if c
	  (cdr c)
	(if (and
	     (integerp ch)
	     (> ch 32))
	    (char-to-string ch)
	  nil)))))

(defun anthy-proc-agent-reply (repl)
  (let*
      ((stat (car repl))
       (body (cdr repl))
       (commit "")
       (preedit nil))
    (while body
      (let ((cur (car body)))
	(setq body (cdr body))
	(if (and (listp cur) (listp (car cur)))
	    (cond ((eq (car (car cur)) 'COMMIT)
		   (setq commit (concat commit (car (cdr cur)))))
		  ((eq (car (car cur)) 'CUTBUF)
		   (let ((len (length (car (cdr cur)))))
		     (copy-region-as-kill (point) (+ (point) len))))
		  ((memq 'UL (car cur))
		   (setq preedit (append preedit (list cur)))))
	  (setq preedit (append preedit (list cur))))))
    (if (> (string-width commit) 0)
	(progn
	  (anthy-erase-preedit)
	  (anthy-disable-preedit-keymap)
	  (insert-and-inherit commit)))
    (anthy-update-preedit stat preedit)
    (anthy-update-mode-line)))

(defun anthy-insert-select-candidate (ch)
  (let ((idx (cdr (assq (car (rassoc (char-to-string ch)
				     anthy-select-candidate-keybind))
			anthy-enum-candidate-list))))
    (if idx
	(progn
	  (let ((repl (anthy-send-recv-command
		       (concat " SELECT_CANDIDATE "
			       (number-to-string idx)
			       "\n"))))
	    (anthy-proc-agent-reply repl))
	  (setq anthy-enum-candidate-p nil)
	  (setq anthy-enum-rcandidate-p nil))
      (message anthy-candidate-minibuffer))))

(defvar anthy-rkmap-keybind
  '((("hiragana" . 113) . "katakana")
    (("katakana" . 113) . "hiragana")
    (("hiragana" . 108) . "alphabet")
    (("katakana" . 108) . "alphabet")
    (("hiragana" . 76) . "walphabet")
    (("katakana" . 76) . "walphabet")
    (("alphabet" . 10) . "hiragana")
    (("walphabet" . 10) . "hiragana")))

(defun anthy-insert (&optional arg)
  "Anthy$B$N%-!<%O%s%I%i(B"
  (interactive "p*")
  (let* ((ch last-command-char)
	 (chenc (anthy-encode-key ch)))
    (cond ((and (or anthy-enum-candidate-p anthy-enum-rcandidate-p)
		(integerp ch)
		(assq (car (rassoc (char-to-string ch)
				   anthy-select-candidate-keybind))
		      anthy-enum-candidate-list))
	   (anthy-insert-select-candidate ch))
	  ((and (assoc (cons anthy-current-rkmap ch) anthy-rkmap-keybind)
		(string-equal anthy-preedit ""))
	   (let ((mapname (cdr (assoc (cons anthy-current-rkmap ch)
				      anthy-rkmap-keybind))))
	     (let ((repl (anthy-send-recv-command 
			  (concat " MAP_SELECT " mapname "\n"))))
	       (if (eq repl 'OK)
		   (progn
		     (setq anthy-current-rkmap
			   (cdr (assoc (cons anthy-current-rkmap ch)
				       anthy-rkmap-keybind)))
		     (anthy-update-mode-line))))))
	  ((and (string-equal anthy-current-rkmap "alphabet")
		(string-equal anthy-preedit ""))
	   (self-insert-command 1))
	  (t
	   (let* ((cmd (if ch 
			   (anthy-encode-key ch)
			 nil))
		  (repl
		   (if cmd (anthy-send-recv-command 
			    (concat cmd "\n"))
		     nil)))
	     (if repl
		 (anthy-proc-agent-reply repl)))))))

;;
(defun anthy-invoke-agent ()
  (let ((list anthy-agent-command-list)
	(proc nil))
    (while (and list (not proc))
      (setq proc 
	    (start-process "anthy-agent" anthy-working-buffer (car list)))
      (if (not (boundp 'proc))
	  (setq proc nil))
      (setq list (cdr list)))
    proc))
;;
;;
;;
(defun anthy-check-agent ()
  ;; check and do invoke
  (if (not anthy-agent-process)
      (let
	  ((proc (anthy-invoke-agent)))
	(if anthy-agent-process
	    (kill-process anthy-agent-process))
	(setq anthy-agent-process proc)
	(process-kill-without-query proc)
	(cond ((coding-system-p 'euc-japan)
	       (set-process-coding-system proc 'euc-japan 'euc-japan))
	      ((coding-system-p '*euc-japan*)
	       (set-process-coding-system proc '*euc-japan* '*euc-japan*)))
	(set-process-sentinel proc 'anthy-process-sentinel))))
;;
(defun anthy-send-recv-command (cmd)
  (if (not anthy-agent-process)
      (anthy-check-agent))
  (let ((old-buffer (current-buffer)))
    (unwind-protect
	(progn
	  (set-buffer anthy-working-buffer)
	  (erase-buffer)
	  (process-send-string anthy-agent-process cmd)
	  (while (= (buffer-size) 0)
	    (accept-process-output))
	  (read (buffer-string)))
      (set-buffer old-buffer))))
;;
(defun anthy-minibuffer-enter ()
  (setq anthy-saved-mode anthy-mode)
  (setq anthy-mode nil)
  (setq anthy-enable-enum-candidate-p 
	(cons nil anthy-enable-enum-candidate-p))
  (anthy-update-mode))
;;
(defun anthy-minibuffer-exit ()
  (setq anthy-mode anthy-saved-mode)
  (setq anthy-enable-enum-candidate-p 
	(cdr anthy-enable-enum-candidate-p))
  (anthy-update-mode))
;;
(defun anthy-mode-on ()
  (add-hook 'minibuffer-setup-hook 'anthy-minibuffer-enter)
  (add-hook 'minibuffer-exit-hook 'anthy-minibuffer-exit)
  (setq anthy-minor-mode t)
  (anthy-update-mode-line))
;;
(defun anthy-mode-off ()
  (setq anthy-minor-mode nil)
  (anthy-update-mode-line))
;;
(defun anthy-update-mode ()
  (if (or anthy-mode anthy-leim-active-p)
      (progn
	(anthy-check-agent)
	(anthy-mode-on))
    (anthy-mode-off)))

(defun anthy-mode (&optional arg)
  "Start Anthy conversion system."
  (interactive "P")
  (setq anthy-mode
        (if (null arg)
            (not anthy-mode)
          (> (prefix-numeric-value arg) 0)))
  (anthy-update-mode))
;;
(defun anthy-hiragana-map (&optional arg)
  "Hiragana mode"
  (interactive "P")
  (anthy-send-recv-command " MAP_SELECT hiragana\n"))
;;
(defun anthy-katakana-map (&optional arg)
  "Hiragana mode"
  (interactive "P")
  (anthy-send-recv-command " MAP_SELECT katakana\n"))
;;
(defun anthy-alpha-map (arg)
  "Alphabet mode"
  (interactive "P")
  (anthy-send-recv-command " MAP_SELECT alphabet\n"))
;;
(defun anthy-wide-alpha-map (arg)
  "Wide Alphabet mode"
  (interactive "P")
  (anthy-send-recv-command " MAP_SELECT walphabet\n"))
;;
;; switch map edit
;;
(defun anthy-send-map-edit-command (key type)
  (if (not (stringp key))
      (setq key (char-to-string key)))
  (anthy-send-recv-command
   (concat " MAP_EDIT " key " " type "\n")))
;;
(defun anthy-wide-char (arg)
  "Wide char"
  (interactive "c key:")
  (anthy-send-map-edit-command arg "wide"))
;;
(defun anthy-narrow-char (arg)
  "Narrow char"
  (interactive "c key:")
  (anthy-send-map-edit-command arg "narrow"))
;;
(defun anthy-jp-char (arg)
  "Japanese char"
  (interactive "c key:")
  (anthy-send-map-edit-command arg "jp"))
;;
;; leim $B$N(B inactivate
;;
(defun anthy-leim-inactivate ()
  (setq anthy-leim-active-p nil)
  (anthy-update-mode))
;;
;; leim $B$N(B activate
;;
(defun anthy-leim-activate (&optional name)
  (setq inactivate-current-input-method-function 'anthy-leim-inactivate)
  (setq anthy-leim-active-p t)
  (anthy-update-mode))
;;
;;
;;
(if (fboundp 'register-input-method)
    (register-input-method "japanese-anthy" "Japanese"
			   'anthy-leim-activate "[anthy]"
			   "Anthy Kana Kanji conversion system"))
;;
(if anthy-xemacs
    nil
  (global-set-key [M-escape] 'anthy-mode))
(provide 'anthy)
(if (boundp 'default-input-method)
    (setq-default default-input-method "japanese-anthy"))
(setq default-input-method "japanese-anthy")
(load "anthy-dic")
;;;
;;; anthy.el ends here
