/*
 * roma kana converter
 *
 * $Id: rkconv.c,v 1.1.1.1 2001/12/18 08:32:07 yusuke Exp $
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "rkconv.h"

#define MAX_CONV_CHARS  1024
#define MAX_MAP_PALETTE 10
#define SPECIAL_CHAR '\xff'

struct rk_rule_set
{
	struct rk_rule* rules;
	int nr_rules;
};

struct rk_slr_closure
{
	char* prefix;
	struct rk_rule* r;
	int is_reduction_only;
	struct rk_slr_closure* next[128];
};

struct rk_map
{
	struct rk_rule_set* rs;
	struct rk_slr_closure* cl;
	int refcount;
};

struct rk_conv_context
{
	struct rk_map* map;
	int map_no;
	int old_map_no;
	struct rk_slr_closure* cur_state;
	char cur_str[MAX_CONV_CHARS + 1];
	int cur_str_len;
	struct rk_map* map_palette[MAX_MAP_PALETTE];
};

static int rk_rule_copy_to(struct rk_rule* from, struct rk_rule* to);
static struct rk_rule_set* rk_rule_set_create(struct rk_rule* rules);
static void rk_rule_set_free(struct rk_rule_set* rs);
static struct rk_slr_closure* rk_slr_closure_create(struct rk_rule_set* rs, char* prefix, int pflen);
static void rk_slr_closure_free(struct rk_slr_closure* cl);
static void rk_convert_iterative(struct rk_conv_context* cc, 
				 int c, char* buf, int size);
static int rk_reduce(struct rk_conv_context* cc,
		     struct rk_slr_closure* cur_state, char* buf, int size);


static int
rk_rule_copy_to(struct rk_rule* from, struct rk_rule* to)
{
	to->lhs = strdup(from->lhs);
	to->rhs = strdup(from->rhs);
	to->follow = from->follow ? strdup(from->follow) : NULL;

	if (to->lhs == NULL || to->rhs == NULL ||
	    (from->follow != NULL && to->follow == NULL)) {
		free(to->lhs);
		free(to->rhs);
		free(to->follow);

		to->lhs    = NULL;
		to->rhs    = NULL;
		to->follow = NULL;
		return -1;
	}
	return 0;
}

static struct rk_rule_set*
rk_rule_set_create(struct rk_rule* rules)
{
	int i;
	struct rk_rule_set* rs;

	rs = (struct rk_rule_set*) malloc(sizeof(struct rk_rule_set));
	if (rs == NULL) {
		errno = ENOMEM;
		return NULL;
	}

	for (i = 0; rules[i].lhs != NULL; i++);
	rs->nr_rules = i;
	rs->rules = (struct rk_rule*) malloc(sizeof(struct rk_rule) * i);
	if (rs->rules == NULL) {
		free(rs);
		errno = ENOMEM;
		return NULL;
	}
	for (i = 0; i < rs->nr_rules; i++) {
		if (rk_rule_copy_to(rules + i, rs->rules + i) != 0) {
			rs->nr_rules = i;
			rk_rule_set_free(rs);
			return NULL;
		}
	}
	return rs;
}

static void
rk_rule_set_free(struct rk_rule_set* rs)
{
	int i;

	for (i = 0; i < rs->nr_rules; i++) {
		free(rs->rules[i].lhs);
		free(rs->rules[i].rhs);
		free(rs->rules[i].follow);
	}
	free(rs->rules);
	free(rs);
}

static struct rk_slr_closure* 
rk_slr_closure_create(struct rk_rule_set* rs, char* prefix, int pflen)
{
	struct rk_slr_closure* cl;
	int i;

	cl = (struct rk_slr_closure*) malloc(sizeof(struct rk_slr_closure));
	if (cl == NULL) {
		errno = ENOMEM;
		return NULL;
	}

	if (prefix != NULL) {
		cl->prefix = (char*) malloc(pflen + 1);
		if (cl->prefix == NULL) {
			errno = ENOMEM;
			free(cl);
			return NULL;
		}
		memcpy(cl->prefix, prefix, pflen);
		cl->prefix[pflen] = '\0';
	} else {
		cl->prefix = strdup("");
		if (cl->prefix == NULL) {
			free(cl);
			return NULL;
		}
	}
    
	cl->r = NULL;
	cl->is_reduction_only = 1;
	memset(cl->next, 0, sizeof cl->next);

	for (i = 0; i < rs->nr_rules; i++) {
		struct rk_rule* r;
		int c;
		r = rs->rules + i;
		if (pflen > 0 && strncmp(prefix, r->lhs, pflen) != 0)
			continue;

		c = r->lhs[pflen] & 0x7f;
		if (c == '\0') { /* reduce */
			cl->r = r;
			if (r->follow != NULL)
				cl->is_reduction_only = 0;
		} else {
			cl->is_reduction_only = 0;
			if (cl->next[c] == NULL) {
				cl->next[c] = rk_slr_closure_create(
					rs, r->lhs, pflen + 1);
				if (cl->next[c] == NULL) {
					rk_slr_closure_free(cl);
					return NULL;
				}
			}
		}
	}

	return cl;
}

static void
rk_slr_closure_free(struct rk_slr_closure* cl)
{
	int i;
	free(cl->prefix);
	for (i = 0; i < 128; i++)
		free(cl->next[i]);
	free(cl);
}

struct rk_map*
rk_map_create(struct rk_rule* rules)
{
	struct rk_map* map;

	map = (struct rk_map*) malloc(sizeof(struct rk_map));
	if (map == NULL) {
		errno = ENOMEM;
		return NULL;
	}

	map->rs = rk_rule_set_create(rules);
	if (map->rs == NULL) {
		free(map);
		return NULL;
	}

	map->cl = rk_slr_closure_create(map->rs, NULL, 0);
	if (map->cl == NULL) {
		rk_rule_set_free(map->rs);
		free(map);
		return NULL;
	}

	return map;
}

int
rk_map_free(struct rk_map* map)
{
	if (map->refcount > 0)
		return -1;
	rk_rule_set_free(map->rs);
	rk_slr_closure_free(map->cl);
	return 0;
}

void
rk_convert_iterative(struct rk_conv_context* cc, int c, char* buf, int size)
{
	struct rk_slr_closure* cur_state = cc->cur_state;

	if (cc->map == NULL)
		return;
	if (size > 0)
		*buf = '\0';
 AGAIN:

	if (cur_state->next[c]) {
		struct rk_slr_closure* next_state = cur_state->next[c];

		if (next_state->is_reduction_only) {
			rk_reduce(cc, next_state, buf, size);
			if (cc->map == NULL) {
				cc->cur_state = NULL;
				return;
			}
			cur_state = cc->map->cl;
		} else 
			cur_state = next_state;
	} else if (cur_state->r != NULL &&
		   (cur_state->r->follow == NULL || 
		    strchr(cur_state->r->follow, c))) {
		int len;
		len = rk_reduce(cc, cur_state, buf, size);
		if (cc->map == NULL) {
			cc->cur_state = NULL;
			return;
		}
		cur_state = cc->map->cl;
		buf += len;
		size -= len;
		goto AGAIN;
	} else if (cur_state != cc->map->cl) {
		cur_state = cc->map->cl;
		goto AGAIN;
	}
	cc->cur_state = cur_state;
}

static int
rk_reduce(struct rk_conv_context* cc,
	  struct rk_slr_closure* cur_state, char* buf, int size)
{
	struct rk_rule* r;
	char* p, * q, * end;

	r = cur_state->r;
	if (r == NULL || size <= 0)
		return 0;
  
	if (r->rhs[0] == SPECIAL_CHAR) {
		if (r->rhs[1] == 'o') 
			rk_select_registered_map(cc, cc->old_map_no);
		else {
			int mapn = r->rhs[1] - '0';
			rk_select_registered_map(cc, mapn);
		}
		return 0;
	}

	p = r->rhs;
	q = buf;
	end = buf + size - 1;
	while (*p && q < end)
		*q ++ = *p++;
	*q = '\0';

	return q - buf;
}

struct rk_conv_context*
rk_context_create(void)
{
	struct rk_conv_context* cc;

	cc = (struct rk_conv_context*) malloc(sizeof(struct rk_conv_context));
	if (cc == NULL) {
		errno = ENOMEM;
		return NULL;
	}

	cc->map = NULL;
	memset(&cc->map_palette, 0, sizeof(struct rk_map*) * MAX_MAP_PALETTE);
	cc->map_no = -1;
	cc->old_map_no = -1;
	rk_flush(cc);

	return cc;
}

void
rk_context_free(struct rk_conv_context* cc)
{
	int i;
	rk_select_map(cc, NULL);
	for (i = 0; i < MAX_MAP_PALETTE; i++)
		rk_register_map(cc, i, NULL);
	free(cc);
}

int
rk_push_key(struct rk_conv_context* cc, int c)
{
	c &= 0x7f;
	if (cc->cur_state == NULL)
		return -1;

	rk_convert_iterative(cc, c, 
			     cc->cur_str + cc->cur_str_len, 
			     MAX_CONV_CHARS + 1 - cc->cur_str_len);
	cc->cur_str_len += strlen(cc->cur_str + cc->cur_str_len);
  
	return 0;
}

void
rk_flush(struct rk_conv_context* cc)
{
	cc->cur_state = (cc->map == NULL) ? NULL : cc->map->cl;
	cc->cur_str[0] = '\0';
	cc->cur_str_len = 0;
}

int
rk_result(struct rk_conv_context* cc, char* buf, int size)
{
	int copy_len;
  
	if (size <= 0)
		return cc->cur_str_len;
	copy_len = (size - 1 < cc->cur_str_len) ? size - 1 : cc->cur_str_len;
	memcpy(buf, cc->cur_str, copy_len);
	buf[copy_len] = '\0';
	if (copy_len < cc->cur_str_len)
		memmove(cc->cur_str, cc->cur_str + copy_len, 
			cc->cur_str_len - copy_len + 1);
	cc->cur_str_len -= copy_len;
  
	return cc->cur_str_len;
}

struct rk_map*
rk_select_map(struct rk_conv_context* cc, struct rk_map* map)
{
	struct rk_map* old_map;
  
	cc->old_map_no = cc->map_no;
	old_map = cc->map;
	if (old_map)
		old_map->refcount--;

	cc->map = map;
	if (cc->map == NULL)
		cc->cur_state = NULL;
	else {
		map->refcount++;
		cc->cur_state = map->cl;
		rk_flush(cc);
	}
	cc->map_no = -1;

	return old_map;
}


int
rk_get_pending_str(struct rk_conv_context* cc, char* buf, int size)
{
	char* p, * q, * end;

	p = (cc->cur_state == NULL) ? "" : cc->cur_state->prefix;

	if (size <= 0)
		return strlen(p) + 1;

	q = buf;
	end = buf + size - 1;
	while (*p && q < end)
		*q++ = *p++;
	*q = '\0';
	return strlen(p);
}

struct rk_map* 
rk_register_map(struct rk_conv_context* cc, int mapn, struct rk_map* map)
{
	struct rk_map* old_map;

	if (mapn < 0 || MAX_MAP_PALETTE <= mapn)
		return NULL;

	old_map = cc->map_palette[mapn];
	if (old_map)
		old_map->refcount--;

	cc->map_palette[mapn] = map;
	if (map)
		map->refcount++;

	return old_map;
}

void
rk_select_registered_map(struct rk_conv_context* cc, int mapn)
{
	if (0 <= mapn && mapn < 0 + MAX_MAP_PALETTE) {
		rk_select_map(cc, cc->map_palette[mapn]);
		cc->map_no = mapn;
	} else {
		rk_select_map(cc, NULL);
		cc->map_no = -1;
	}
}

int
rk_selected_map(struct rk_conv_context* cc)
{
	return cc->map_no;
}

/* some utitlity functions to merge rk_rule */
static int rk_rule_length(struct rk_rule* rules)
{
	int i;
	for (i = 0; rules[i].lhs != NULL; i++);
	return i;
}

struct rk_rule* rk_merge_rules(struct rk_rule* r1, struct rk_rule* r2)
{
	int size;
	int ret;
	struct rk_rule* rules;
	struct rk_rule* p, * q;

	size = rk_rule_length(r1) + rk_rule_length(r2);
	rules = (struct rk_rule*) malloc(sizeof(struct rk_rule) * (size + 1));
	if (rules == NULL)
		return NULL;

	q = rules;
	for (p = r1; p->lhs != NULL; p++) {
		ret = rk_rule_copy_to (p, q++);
		if (ret == -1)
			goto ERROR;
	}
	for (p = r2; p->lhs != NULL; p++) {
		ret = rk_rule_copy_to (p, q++);
		if (ret == -1)
			goto ERROR;
	}
	q->lhs = NULL;

	return rules;

ERROR:
	rk_rules_free (rules);
	return NULL;
}

void rk_rules_free(struct rk_rule* rules)
{
	struct rk_rule* p;

	for (p = rules; p->lhs != NULL; p++) {
		free(p->lhs);
		free(p->rhs);
		free(p->follow);
	}
  
	free(rules);
}

#ifdef RKCONV_DEBUG
#include "rkmap.h"

int
main (void)
{
	char buf[1024];
	char buf2[1024];
	struct rk_map* m;
	struct rk_conv_context* cc;

	m = rk_map_create(rk_rule_hiragana);
	cc = rk_context_create();

	rk_select_map(cc, m);
	{
		char* p = "aho";
		char* src = p;

		while (*p) rk_push_key (cc, *p++);
		rk_result (cc, buf, sizeof (buf));
		rk_get_pending_str (cc, buf2, sizeof (buf2));

		printf ("%s => %s %s\n", src, buf, buf2);
	}
	{
		char* p = "hogehgoefonyunaninnunenoky";
		char* src = p;
		while (*p) rk_push_key(cc, *p++);
		rk_result (cc, buf, sizeof (buf));
		rk_get_pending_str (cc, buf2, sizeof (buf2));
		printf ("%s => %s %s\n", src, buf, buf2);
	}

	rk_context_free(cc);
	rk_map_free(m);

	return 0;
}
#endif /* RKCONV_DEBUG */
