/**/
#include <stdlib.h>
#include <anthy/anthy.h>
#include "context.h"

#define MAX_CONTEXT 32

struct context {
  struct anthy_context *ac;
} *context_slot;

static struct anthy_context *
get_anthy_context(int id)
{
  if (id < 0 || id >= MAX_CONTEXT) {
    return NULL;
  }
  return context_slot[id].ac;
}

static LISP
init_anthy_lib()
{
  int i;
  anthy_init();
  context_slot = malloc(sizeof(struct context) *
			MAX_CONTEXT);
  for (i = 0; i < MAX_CONTEXT; i++) {
    context_slot[i].ac = NULL;
  }
  return NIL;
}

static LISP
create_context()
{
  int i;
  for (i = 0; i < MAX_CONTEXT; i++) {
    if (!context_slot[i].ac) {
      context_slot[i].ac = anthy_create_context();
      return flocons(i);
    }
  }
  return NIL;
}


static LISP
release_context(LISP id_)
{
  int id = get_c_long(id_);
  if (context_slot[id].ac) {
    anthy_release_context(context_slot[id].ac);
    context_slot[id].ac = NULL;
  }
  return NIL;
}

static LISP
set_string(LISP id_, LISP str_)
{
  int id = get_c_long(id_);
  char *str;
  anthy_context_t ac = get_anthy_context(id);
  if (!ac) {
    return NIL;
  }
  str = uim_get_c_string(str_);
  anthy_set_string(ac, str);
  free(str);
  return NIL;
}

static LISP
get_nr_segments(LISP id_)
{
  int id = get_c_long(id_);
  struct anthy_conv_stat acs;
  anthy_context_t ac = get_anthy_context(id);
  if (!ac) {
    return NIL;
  }
  anthy_get_stat(ac, &acs);

  return flocons(acs.nr_segment);
}

static LISP
get_nr_candidates(LISP id_, LISP nth_)
{
  int id, nth;
  anthy_context_t ac;
  struct anthy_conv_stat cs;
  id = get_c_long(id_);
  nth = get_c_long(nth_);
  ac = get_anthy_context(id);
  if (!ac) {
    return NIL;
  }
  anthy_get_stat(ac, &cs);
  if (nth < cs.nr_segment) {
    struct anthy_segment_stat ss;
    anthy_get_segment_stat(ac, nth, &ss);
    return flocons(ss.nr_candidate);
  }
  return NIL;
}

static LISP
get_nth_candidate(LISP id_, LISP seg_, LISP nth_)
{
  int id = get_c_long(id_);
  int seg = get_c_long(seg_);
  int nth  = get_c_long(nth_);
  int buflen;
  char *buf;
  anthy_context_t ac = get_anthy_context(id);
  struct anthy_conv_stat cs;
  if (!ac) {
    return NIL;
  }
  buflen = anthy_get_segment(ac, seg, nth, NULL, 0);
  if (buflen == -1) {
    return NIL;
  }
  buf = alloca(buflen+1);
  anthy_get_segment(ac, seg, nth, buf, buflen+1);
  return strcons(buflen, buf);
}

static LISP
resize_segment(LISP id_, LISP seg_, LISP cnt_)
{
  int id = get_c_long(id_);
  int seg = get_c_long(seg_);
  int cnt = get_c_long(cnt_);
  anthy_context_t ac = get_anthy_context(id);
  anthy_resize_segment(ac, seg, cnt);
  return NIL;
}

void
uim_init_anthy()
{
  init_subr_0("anthy-lib-init", init_anthy_lib);

  init_subr_0("anthy-lib-alloc-context", create_context);
  init_subr_1("anthy-lib-free-context", release_context);

  init_subr_2("anthy-lib-set-string", set_string);
  init_subr_1("anthy-lib-get-nr-segments",get_nr_segments);
  init_subr_2("anthy-lib-get-nr-candidates", get_nr_candidates);
  init_subr_3("anthy-lib-get-nth-candidate", get_nth_candidate);
  init_subr_3("anthy-lib-resize-segment", resize_segment);
}
