/*
 * SKK is a simple Japanese input method
 */
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>

#include "context.h"

extern LISP sym_t;

struct skk_line {
  char *head;
  char okuri;
  int nr_cands;
  char **cands;
  struct skk_line *next;
};

static struct dic_info {
  void *addr;
  int border;
  int size;
  struct skk_line head;
} *skk_dic;

static int
calc_line_len(char *s)
{
  int i;
  for (i = 0; s[i] != '\n'; i++);
  return i;
}


static int
is_okuri(char *str)
{
  char *b;
  b = strchr(str, ' ');
  if (!b) {
    return 0;
  }
  b--;
  if (isalpha(*b)) {
    return 1;
  }
  return 0;
}

static int
find_border(struct dic_info *di)
{
  char *s = di->addr;
  int off = 0;
  while (1) {
    int l = calc_line_len(&s[off]);
    if (s[off] == ';') {
      off += l +1;
      continue;
    }
    if (!is_okuri(&s[off])) {
      return off;
    }
    off += l + 1;
  }
  return 0;
}

static struct dic_info *
open_dic(const char *fn)
{
  struct dic_info *di;
  struct stat st;
  int fd;
  void *addr;
  if (lstat(fn, &st) == -1) {
    return NULL;
  }
  fd = open(fn, O_RDONLY);
  if (fd == -1) {
    return NULL;
  }
  addr = mmap(0, st.st_size, PROT_READ, MAP_SHARED, fd, 0);
  if (addr == MAP_FAILED) {
    close(fd);
    return NULL;
  }
  di = (struct dic_info *)malloc(sizeof(struct dic_info));
  di->addr = addr;
  di->size = st.st_size;
  di->border = find_border(di);
  di->head.next = NULL;
  close(fd);
  return di;
}

static char *
find_line(struct dic_info *di, int off)
{
  char *ptr = di->addr;
  while (off > 0 && ptr[off] != '\n') {
    off --;
  }
  if (off) {
    off ++;
  }
  return &ptr[off];
}

static char *
extract_entry(struct dic_info *di, int off, char *buf, int len)
{
  char *p = find_line(di, off);
  int i;
  if (p[0] == ';') {
    return NULL;
  }
  for (i = 0; i < len && p[i] != ' '; i++) {
    buf[i] = p[i];
  }
  buf[i] = 0;
  return buf;
}

static int
do_search(struct dic_info *di, char *s, int min,
	  int max, int d)
{
  char buf[256];
  char *r;
  int idx = (min + max) / 2;
  int c = 0;

  if (abs(max-min) < 4) {
    return -1;
  }
  r = extract_entry(di, idx, buf, 256);
  if (r) {
    c = strcmp(s,r);
  } else {
    return -1;
  }

  if (!c) {
    return idx;
  }
  if (c * d> 0) {
    return do_search(di, s, idx, max, d);
  } else {
    return do_search(di, s, min, idx, d);
  }
  return -1;
}

static char *
skk_search_file(struct dic_info *di, char *s, char okuri)
{
  int n;
  char *p;
  int len;
  char *line;
  char *idx = alloca(strlen(s) + 2);
  if (!di) {
    return "";
  }
  sprintf(idx, "%s%c",s, okuri);
  printf("INDEX=%s\n", idx);
  if (okuri) {
    n = do_search(di, idx, 0, di->border - 1, -1);
  } else {
    n = do_search(di, idx, di->border, di->size - 1, 1);
  }
  if (n == -1) {
    printf("not found\n");
    return NULL;
  }
  p = find_line(di, n);
  len = calc_line_len(p);
  line = malloc(len+1);
  line[0] = 0;
  strncat(line, p, len);
  return line;
}

char *next_slash(char *str)
{
  while (*str && *str != '/') {
    str ++;
  }
  return str;
}

static char *
nth_candidate(char *str, int nth)
{
  char *p , *term;
  int i;
  for (i = 0; i <= nth; i++) {
    str = next_slash(str);
    if (*str == '/') {
      str++;
    }
  }
  if (!str) {
    return NULL;
  }
  if (*str == '/') {
    str++;
  }
  p = strdup(str);
  term = next_slash(p);
  *term = 0;
  return p;
}

static LISP
skk_dic_open(LISP fn)
{
  char *s = uim_get_c_string(fn);
  if (!skk_dic) {
    skk_dic = open_dic(s);
  }
  free(s);
  return NIL;
}

static struct skk_line *
skk_search(struct dic_info *di, char *s, char okuri)
{
  struct skk_line *sl;
  char *res;
  char **cands;
  char *tmp;
  int i, nr;
  if (!di) {
    return NULL;
  }
  for (sl = di->head.next; sl; sl = sl->next) {
    if (!strcmp(sl->head, s) &&
	sl->okuri == okuri) {
      return sl;
    }
  }
  res = skk_search_file(di, s, okuri);
  if (!res) {
    return NULL;
  }
  sl = malloc(sizeof(struct skk_line));
  sl->head = strdup(s);
  sl->okuri = okuri;
  cands = alloca(sizeof(char*)*strlen(res));

  nr = 0;
  do {
    tmp = nth_candidate(res, nr);
    if (tmp && strlen(tmp)) {
      cands[nr] = tmp;
      nr++;
    } else {
      break;
    }
  } while (1);
  sl->cands = malloc(sizeof(char *) * nr);
  sl->nr_cands = nr;
  for (i = 0; i < nr; i++) {
    sl->cands[i] = cands[i];
  }

  /* link */
  sl->next = di->head.next;
  di->head.next = sl;
  return sl;
}

static LISP
skk_get_entry(LISP head, LISP okuri)
{
  char *hs, o = 0;
  struct skk_line *sl;
  if (okuri == NIL) {
    o = 0;
  } else {
    char *os = uim_get_c_string(okuri);
    o = os[0];
    free(os);
  }
  hs = uim_get_c_string(head);
  sl = skk_search(skk_dic, hs, o);
  free(hs);

  if (!sl) {
    return NIL;
  }

  return sym_t;
}

static LISP
skk_get_nth_candidate(LISP head, LISP okuri, LISP nth)
{
  int n;
  char o;
  char *hs;
  LISP ret;
  struct skk_line *sl;

  hs = get_c_string(head);
  n = get_c_long(nth);
  if (okuri == NIL) {
    o = 0;
  } else {
    char *os= get_c_string(okuri);
    o = os[0];
  }

  sl = skk_search(skk_dic, hs, o);
  if (sl && sl->nr_cands > n) {
    return strcons(strlen(sl->cands[n]), sl->cands[n]);
  }
   return NIL;
}

static LISP
skk_get_nr_candidates(LISP head, LISP okuri)
{
  int n;
  char o;
  char *hs;
  LISP ret;
  struct skk_line *sl;

  hs = get_c_string(head);
  if (okuri == NIL) {
    o = 0;
  } else {
    char *os= get_c_string(okuri);
    o = os[0];
  }

  sl = skk_search(skk_dic, hs, o);
  return flocons(sl->nr_cands);
}

static LISP
skk_commit_candidate(LISP head_, LISP okuri, LISP nth_)
{
  int nth;
  char o;
  char *head;
  struct skk_line *sl;

  head = get_c_string(head_);
  nth = get_c_long(nth_);
  if (okuri == NIL) {
    o = 0;
  } else {
    char *os= get_c_string(okuri);
    o = os[0];
  }

  sl = skk_search(skk_dic, head, o);
  if (sl && sl->nr_cands > nth) {
    char *tmp;
    tmp = sl->cands[0];
    sl->cands[0] = sl->cands[nth];
    sl->cands[nth] = tmp;
  }
  return NIL;
}


void uim_init_skk_dic()
{
  init_subr_1("skk-lib-dic-open", skk_dic_open);
  init_subr_2("skk-lib-get-entry", skk_get_entry);
  init_subr_3("skk-lib-get-nth-candidate", skk_get_nth_candidate);
  init_subr_2("skk-lib-get-nr-candidates", skk_get_nr_candidates);
  init_subr_3("skk-lib-commit-candidate", skk_commit_candidate);
}
