/*
 * gtk+-immodule
 */
#include <gtk/gtk.h>
#include <gtk/gtkimcontext.h>
#include <gtk/gtkimmodule.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>
#include <glib/gprintf.h>
#include <string.h>
#include <stdlib.h>
#include <uim/uim.h>
#include <uim/gettext.h>

#define NR_CANDIDATES 10

struct preedit_segment {
  int attr;
  char *str;
};

struct candidate_window {
  GtkWidget *top_win;
  GtkWidget *clist;
  GtkWidget *numlabel;
  int nr_candidates;
  int candidate_index;
  int layout_begin;
  int is_active;
  int select_count;
  GdkRectangle cursor;
};

typedef struct _IMUIMContext {
  /**/
  struct _GtkIMContext parent;
  struct _GtkIMContext *slave;
  uim_context uc;
  struct candidate_window cwin;
  int nr_psegs;
  struct preedit_segment *pseg;
  /**/
  GtkWidget *stat_win;
  GtkWidget *menu;
  GdkWindow *win;
  GdkRectangle preedit_pos; /* preedit_pos not always point the cursor location */
  int mode;
  /**/
  struct _IMUIMContext *prev, *next;
} IMUIMContext;

static IMUIMContext context_list;

static GObjectClass *parent_class;

typedef struct _IMContextUIMClass 
{
  GtkIMContextClass parent_class;
} IMContextUIMClass;

static void im_uim_class_init (GtkIMContextClass *class);

static void show_preedit(GtkIMContext *ic, GtkWidget *preedit_label);


static const GTypeInfo class_info = {
  sizeof(IMContextUIMClass),
  (GBaseInitFunc) NULL,
  (GBaseFinalizeFunc) NULL,
  (GClassInitFunc) im_uim_class_init,
  NULL,
  NULL,
  sizeof(IMUIMContext),
  0,
  (GtkObjectInitFunc) NULL,
};

static GType type_im_uim;
#define IM_UIM_CONTEXT(obj) (GTK_CHECK_CAST((obj),type_im_uim,IMUIMContext))
static const GtkIMContextInfo im_uim_info = {
  "uim",
  "UIM()",
  "gtk+",
  "",
  ""
};

static
void
im_uim_commit_string(void *ptr, char *str)
{
  IMUIMContext *uic = (IMUIMContext *)ptr;
  g_signal_emit_by_name(uic, "commit", str);
}

static void
clear_cb(void *ptr)
{
  IMUIMContext *uic = ptr;
  int i;
  for (i = 0; i < uic->nr_psegs; i++) {
    free(uic->pseg[i].str);
  }
  free(uic->pseg);
  uic->pseg = 0;
  uic->nr_psegs = 0;
}

static void
pushback_cb(void *ptr, int attr, char *str)
{
  if(!str)
    return;

  IMUIMContext *uic = ptr;
  uic->pseg = realloc(uic->pseg,
		      sizeof(struct preedit_segment) *
		     (uic->nr_psegs + 1));
  uic->pseg[uic->nr_psegs].str = strdup(str);
  uic->pseg[uic->nr_psegs].attr = attr;
  uic->nr_psegs ++;
}

static void
update_cb(void *ptr)
{
  IMUIMContext *uic = ptr;
  g_signal_emit_by_name(uic, "preedit_changed");
}

static void
mode_cb(void *ptr, int mode)
{
  IMUIMContext *uic = ptr;
  if (uic->mode == mode) {
    return ;
  }
  gtk_option_menu_set_history(GTK_OPTION_MENU(uic->menu), mode);
  uic->mode = mode;
}

static int
convert_keyval(int key)
{
  switch (key) {
  case GDK_BackSpace: return UKey_Backspace;
  case GDK_Delete: return UKey_Delete;
  case GDK_Escape: return UKey_Escape;
  case GDK_Tab: return UKey_Tab;
  case GDK_Return: return UKey_Return;
  case GDK_Left: return UKey_Left;
  case GDK_Up: return UKey_Up;
  case GDK_Right: return UKey_Right;
  case GDK_Down: return UKey_Down;
  case GDK_Prior: return UKey_Prior;
  case GDK_Next: return UKey_Next;
  case GDK_Home: return UKey_Home;
  case GDK_End: return UKey_End;
  case GDK_Zenkaku_Hankaku: return UKey_Zenkaku_Hankaku;
  case GDK_F1: return UKey_F1;
  case GDK_F2: return UKey_F2;
  case GDK_F3: return UKey_F3;
  case GDK_F4: return UKey_F4;
  case GDK_F5: return UKey_F5;
  case GDK_F6: return UKey_F6;
  case GDK_F7: return UKey_F7;
  case GDK_F8: return UKey_F8;
  case GDK_F9: return UKey_F9;
  case GDK_F10: return UKey_F10;
  case GDK_F11: return UKey_F11;
  case GDK_F12: return UKey_F12;
  }

  if(key >= GDK_KP_0 && key <= GDK_KP_9) {
    return key - GDK_KP_0 + UKey_0;
  }
  if (key < 256) {
    return key;
  }
  return UKey_Other;
}

static int
convert_modifier(int mod)
{
  int rv = 0;
  if (mod & GDK_SHIFT_MASK) {
    rv |= UKey_Shift;
  }
  if (mod & GDK_CONTROL_MASK) {
    rv |= UKey_Control;
  }
  if (mod & GDK_MOD1_MASK) {
    rv |= UKey_Alt;
  }
  return rv;
}


/*
 * KEY EVENT HANDLER
 */
static gboolean
filter_keypress(GtkIMContext *ic,
		GdkEventKey *key)
{
  int rv;
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);
  int kv = convert_keyval(key->keyval);
  int mod = convert_modifier(key->state);

  if (key->type == GDK_KEY_RELEASE) {
    rv = uim_release_key(uic->uc, kv, mod);
  } else {
    rv = uim_press_key(uic->uc, kv, mod);
  }
  if (rv) {
    return gtk_im_context_filter_keypress(uic->slave, key);
  }
  return TRUE;
}

static char *
get_preedit_segment(struct preedit_segment *ps,
		    PangoAttrList *attrs,
		    char *str)
{
  PangoAttribute *attr;
  if (attrs) {
    int begin, end;
    begin = strlen(str);
    end = begin + strlen(ps->str);
    if (ps->attr & UPeAttr_UnderLine) {
      attr = pango_attr_underline_new(PANGO_UNDERLINE_SINGLE);
      attr->start_index = begin;
      attr->end_index = end;
      pango_attr_list_change(attrs, attr);
    }
    if (ps->attr & UPeAttr_Reverse) {
        attr = pango_attr_foreground_new (0xffff, 0xffff, 0xffff);
        attr->start_index = begin;
        attr->end_index = end;
        pango_attr_list_change (attrs, attr);

        attr = pango_attr_background_new (0, 0, 0);
        attr->start_index = begin;
        attr->end_index = end;
        pango_attr_list_change (attrs, attr);
    }
  }
  str = (char *)realloc(str, strlen(str) + strlen(ps->str)+1);
  strcat(str, ps->str);
  return str;
}


static void
im_uim_get_preedit_string(GtkIMContext *ic, gchar **str,
			  PangoAttrList **attrs,
			  gint *cursor_pos)
{
  char *tmp;
  int i, pos = 0;
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);

  if (attrs) {
    *attrs = pango_attr_list_new();
  }
  tmp = strdup("");
  for (i = 0; i < uic->nr_psegs; i++) {
    if (uic->pseg[i].attr & UPeAttr_Cursor) {
      pos = g_utf8_strlen(tmp,-1);
    }
    if (attrs) {
      tmp = get_preedit_segment(&uic->pseg[i], *attrs, tmp);
    } else {
      tmp = get_preedit_segment(&uic->pseg[i], NULL, tmp);
    }
  }
  if (cursor_pos) {
    *cursor_pos = pos;
  }
  if (str) {
    *str = tmp;
  } else {
    free(tmp);
  }
}

static void
im_uim_set_cursor_location(GtkIMContext *ic,
			   GdkRectangle *area)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);
  uic->cwin.cursor.x= area->x;
  uic->cwin.cursor.y= area->y;
  uic->cwin.cursor.height= area->height;
  uic->cwin.cursor.width= area->width;

  uic->preedit_pos.x = area->x;
  uic->preedit_pos.x = area->x;

}


static void
im_uim_commit_cb(GtkIMContext *ic,
		 const gchar *str,
		 IMUIMContext *is)
{
  g_signal_emit_by_name(is, "commit", str);
}

static void
update_status_window(IMUIMContext *uic)
{
  if (uic->win && uic->stat_win) {
    gint x,y,w,h,d;
    gdk_window_get_geometry(uic->win, &x, &y,&w,&h,&d);
    gdk_window_get_origin(uic->win, &x, &y);
    gtk_window_move(GTK_WINDOW(uic->stat_win), x+w , y+h);
  }
  if(uic->cwin.top_win && uic->cwin.is_active){
    gtk_widget_show(uic->cwin.top_win);
  }
}

void focus_in(GtkIMContext *ic)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);
  IMUIMContext *cc;
  for (cc = context_list.next; cc != &context_list; cc = cc->next) {
    if (cc != uic) {
      if (uic->stat_win) {
	gtk_widget_hide(uic->stat_win);
      }
      if (uic->cwin.top_win) {
	gtk_widget_hide(uic->cwin.top_win);
      }
    }
  }
  update_status_window(uic);
  if (uic->stat_win) {
    gtk_widget_show(uic->stat_win);
  }
}

void focus_out(GtkIMContext *ic)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);
  if (uic->win && uic->stat_win) {
    gtk_widget_hide(uic->stat_win);
  }
  if(uic->cwin.top_win){
    gtk_widget_hide(uic->cwin.top_win);
  }
}

void im_uim_reset(GtkIMContext *ic)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);

  uim_reset_context(uic->uc);

  g_warning (gettext("reset called: implement me"));

}

void
set_use_preedit(GtkIMContext *ic, gboolean use_preedit)
{
  GtkWidget *preedit_window;
  GtkWidget *preedit_label;
  if( use_preedit == FALSE ) {
    preedit_window = gtk_window_new(GTK_WINDOW_POPUP);
    preedit_label = gtk_label_new("");
    gtk_container_add(GTK_CONTAINER(preedit_window), preedit_label);
    
    g_signal_connect ( G_OBJECT(ic), "preedit-changed",
		       G_CALLBACK (show_preedit), preedit_label );
    gtk_widget_show_all(preedit_window);
  }
}


static void
show_preedit(GtkIMContext *ic, GtkWidget *preedit_label)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);
  GtkWidget *preedit_window;
  gchar *str;
  gint cursor_pos;
  PangoAttrList *attrs;

  preedit_window = gtk_widget_get_parent(preedit_label);
  
  gtk_im_context_get_preedit_string (ic, &str, &attrs, &cursor_pos);

  
  if ( strlen (str) > 0)
    {
      gint x,y,w,h;
      PangoLayout *layout;

      gtk_label_set_text(GTK_LABEL(preedit_label), str);
      gtk_label_set_attributes(GTK_LABEL(preedit_label), attrs);

      gdk_window_get_origin(uic->win, &x, &y);
      
      gtk_window_move( GTK_WINDOW(preedit_window),
		       x + uic->preedit_pos.x,
		       y + uic->preedit_pos.y );

      layout = gtk_label_get_layout(GTK_LABEL(preedit_label));

      pango_layout_get_cursor_pos (layout, 0, NULL, NULL);

      pango_layout_get_pixel_size (layout, &w, &h);
      gtk_window_resize (GTK_WINDOW(preedit_window), w, h);

      gtk_widget_show( preedit_window );

    }
  else {
      gtk_label_set_text(GTK_LABEL(preedit_label), "");
      gtk_widget_hide( preedit_window );
      gtk_window_resize( GTK_WINDOW(preedit_window), 0, 0 );
  }
  g_free (str);
  pango_attr_list_unref (attrs);
  
}


static GdkFilterReturn
resize_window_cb (GdkXEvent *xevent, GdkEvent *ev, gpointer data)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(data);
  update_status_window(uic);
  return GDK_FILTER_CONTINUE;
}


static void
set_client_window(GtkIMContext *ic, GdkWindow *w)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(ic);
  if (uic->win != w) {
    if (uic->win) {
      gdk_window_remove_filter(uic->win, resize_window_cb, uic);
    }
    if (w) {
      GdkEventMask mask;
      mask = gdk_window_get_events(w);
      gdk_window_set_events(w, mask | GDK_STRUCTURE_MASK);
      gdk_window_add_filter(w, resize_window_cb, uic);
    }
  }
  uic->win = w;
  update_status_window(uic);
}


/*
 * DESTRUCTOR
 */
static void
im_uim_finalize(GObject *obj)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(obj);
  set_client_window(GTK_IM_CONTEXT(uic), NULL);
  uic->next->prev = uic->prev;
  uic->prev->next = uic->next;

  if (uic->menu) {
    gtk_widget_destroy(uic->menu);
  }
  if (uic->stat_win) {
    gtk_widget_destroy(uic->stat_win);
  }
  if (uic->cwin.top_win) {
    gtk_widget_destroy(uic->cwin.top_win);
  }
  uic->stat_win = NULL;
  uic->menu = NULL;

  uim_release_context(uic->uc);

  g_signal_handlers_disconnect_by_func(uic->slave,
				       im_uim_commit_cb,
				       uic);
  g_object_unref(uic->slave);
  parent_class->finalize (obj);
}

static void
im_uim_class_init (GtkIMContextClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);
  parent_class = g_type_class_peek_parent(class);
  class->set_client_window = set_client_window;
  class->filter_keypress = filter_keypress;
  class->get_preedit_string = im_uim_get_preedit_string;
  class->set_cursor_location = im_uim_set_cursor_location;
  class->focus_in = focus_in;
  class->focus_out = focus_out;
  class->reset = im_uim_reset;
  class->set_use_preedit = set_use_preedit;

  object_class->finalize = im_uim_finalize;
}

static gint
select_mode(GtkWidget *w, gpointer p)
{
  IMUIMContext *is = p;
  int nth = gtk_option_menu_get_history(GTK_OPTION_MENU(is->menu));
  uim_set_mode(is->uc, nth);
  return 0;
}

static void
update_status_menu(IMUIMContext *uic)
{
  int i, nr;
  GtkWidget *menu;
  GtkWidget *hbox;

  nr = uim_get_nr_modes(uic->uc);
  if (!nr) {
    uic->menu = NULL;
    uic->stat_win = NULL;
    return ;
  }

  /* menu */
  uic->menu = gtk_option_menu_new();
  uic->stat_win = gtk_window_new(GTK_WINDOW_POPUP);
  menu = gtk_menu_new();
  for (i = 0; i < nr; i++) {
    GtkWidget *item;
    char *name;
    name = uim_get_mode_name(uic->uc, i);
    item = gtk_menu_item_new_with_label(name);
    gtk_menu_append(menu, item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(select_mode), uic);
    gtk_widget_show(item);
  }
  /**/
  hbox = gtk_hbox_new(FALSE, 0);
  /**/
  gtk_container_add(GTK_CONTAINER(uic->stat_win), hbox);
  gtk_option_menu_set_menu(GTK_OPTION_MENU(uic->menu), menu);
  gtk_box_pack_start(GTK_BOX(hbox), uic->menu, FALSE, FALSE, 2);
  gtk_option_menu_set_history(GTK_OPTION_MENU(uic->menu),
			      uim_get_current_mode(uic->uc));
  gtk_widget_show(uic->menu);
  gtk_widget_show(hbox);
}

static void
update_mode_list_cb(void *ptr)
{
  IMUIMContext *uic = ptr;
  if (uic->menu) {
    gtk_widget_destroy(uic->menu);
  }
  if (uic->stat_win) {
    gtk_widget_destroy(uic->stat_win);
  }
  uic->menu = NULL;
  uic->stat_win = NULL;
  update_status_menu(uic);
}


static void
select_candidate_cb(GtkWidget *clist,
		    gint row, gint column,
		    GdkEventButton *event,
		    gpointer data)
{
  IMUIMContext *uic = IM_UIM_CONTEXT(data);
  if (uic->cwin.select_count == 0) {
    int idx = row + uic->cwin.layout_begin;
    
    if( idx >= uic->cwin.nr_candidates ) {
      gtk_clist_select_row(GTK_CLIST(uic->cwin.clist),
			   uic->cwin.candidate_index - uic->cwin.layout_begin,
			   0);
      return ;
    }

    uim_set_candidate_index(uic->uc, idx);
    uic->cwin.candidate_index = idx;
  } else {
    uic->cwin.select_count --;
  }
}

static void
init_candidate_win(IMUIMContext *uic)
{
  struct candidate_window *cwin = &uic->cwin;
  GtkWidget *vbox;
  GtkWidget *numlabel;

  if (cwin->top_win) {
    return ;
  }
  cwin->top_win = gtk_window_new(GTK_WINDOW_POPUP);
  cwin->clist = gtk_clist_new(2);
  cwin->is_active = 0;
  g_signal_connect(G_OBJECT(cwin->clist), "select_row",
		   G_CALLBACK(select_candidate_cb),
		   uic);
  vbox = gtk_vbox_new(FALSE, 0);
  numlabel = gtk_label_new("");
  cwin->numlabel = numlabel;

  gtk_box_pack_start(GTK_BOX(vbox), cwin->clist, FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), numlabel, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(cwin->top_win),vbox);
  {
    gint x,y,w,h,d;
    gdk_window_get_geometry(uic->win, &x, &y,&w,&h,&d);
    gdk_window_get_origin(uic->win, &x, &y);
    gtk_window_move(GTK_WINDOW(cwin->top_win), x , y+h);
  }

  gtk_clist_set_column_auto_resize( GTK_CLIST(cwin->clist), 0, TRUE );
  gtk_clist_set_column_auto_resize( GTK_CLIST(cwin->clist), 1, TRUE );

  gtk_clist_set_shadow_type( GTK_CLIST(cwin->clist), GTK_SHADOW_NONE );
  
  gtk_widget_show(cwin->clist);
  gtk_widget_show(numlabel);
  gtk_widget_show(vbox);
}

static int
trim_index(IMUIMContext *uic)
{
  struct candidate_window *cwin = &uic->cwin;
  int changed = 0;
  while (cwin->layout_begin + NR_CANDIDATES <= cwin->candidate_index) {
    changed = 1;
    cwin->layout_begin += NR_CANDIDATES;
  }  
  while (cwin->layout_begin > cwin->candidate_index) {
    changed = 1;
    cwin->layout_begin -= NR_CANDIDATES;
  }
  if (cwin->layout_begin < 0) {
    changed = 1;
    cwin->layout_begin = 0;
  }
  return changed;
}

static void
layout_candidate(IMUIMContext *uic)
{
  char *buf[2];
  char idx[20];
  int  i, x, y, topwin_x, topwin_y;
  int  sc_he, cw_he; /*screen height, candidate window height*/
  int  sc_wi, cw_wi;
  char *new_first_cand = NULL;
  char *old_first_cand = NULL;

  trim_index(uic);

  gtk_clist_get_text(GTK_CLIST(uic->cwin.clist), 0, 1, &new_first_cand);

  sc_he = gdk_screen_get_height(gdk_screen_get_default ());
  sc_wi = gdk_screen_get_width (gdk_screen_get_default ());

  gtk_window_get_size(GTK_WINDOW(uic->cwin.top_win), &cw_wi, &cw_he);

  gdk_window_get_origin(uic->win, &topwin_x, &topwin_y);

  if(sc_wi <  topwin_x + uic->cwin.cursor.x + cw_wi ) {
    x = topwin_x + uic->cwin.cursor.x - cw_wi;
  } else {
    x = topwin_x + uic->cwin.cursor.x;
  }

  if(sc_he <  topwin_y + uic->cwin.cursor.y +  uic->cwin.cursor.height + cw_he ) {
    y = topwin_y + uic->cwin.cursor.y - cw_he;
  } else {
    y = topwin_y + uic->cwin.cursor.y +  uic->cwin.cursor.height;
  }

  gtk_window_move(GTK_WINDOW(uic->cwin.top_win), x, y );

  new_first_cand = uim_get_candidate(uic->uc,uic->cwin.layout_begin);

  /* This condition works almost case, but not perfect.
     We must think out more good way.  */

  if( !old_first_cand ||
      strcmp( old_first_cand, new_first_cand) !=0 ) {

    gtk_clist_freeze( GTK_CLIST(uic->cwin.clist));
    gtk_clist_clear(GTK_CLIST(uic->cwin.clist));

    buf[0] = "1";
    buf[1] = new_first_cand;
    gtk_clist_append(GTK_CLIST(uic->cwin.clist), buf);
    free( new_first_cand );

    for (i = 1; i < NR_CANDIDATES; i++) {
      if (i < uic->cwin.nr_candidates - uic->cwin.layout_begin) {
	g_sprintf(idx, "%d", i + uic->cwin.layout_begin + 1 );
	buf[0] = idx;
	buf[1] = uim_get_candidate(uic->uc, i + uic->cwin.layout_begin);
      } else {
	buf[0] = "";
	buf[1] = "";
      }
      gtk_clist_append(GTK_CLIST(uic->cwin.clist), buf);

      if(strcmp(buf[1], "") != 0 ) {
	free(buf[1]);
      }
    }
    gtk_clist_thaw( GTK_CLIST(uic->cwin.clist) );
  }
  
  uic->cwin.select_count ++;
  gtk_clist_select_row(GTK_CLIST(uic->cwin.clist),
		       uic->cwin.candidate_index - uic->cwin.layout_begin,
		       0);
  sprintf(idx, "%d/%d",uic->cwin.candidate_index + 1 , uic->cwin.nr_candidates); 

  gtk_label_set_text( GTK_LABEL(uic->cwin.numlabel), idx);

}

static void
cand_begin_cb(void *ptr, int nr, int index)
{
  IMUIMContext *uic = ptr;
  init_candidate_win(uic);
  uic->cwin.is_active = 1;
  uic->cwin.nr_candidates = nr;
  uic->cwin.candidate_index = index;
  uic->cwin.layout_begin = 0;
  layout_candidate(uic);
  gtk_widget_show(uic->cwin.top_win);
}

static void
cand_update_cb(void *ptr, int index)
{
  IMUIMContext *uic = ptr;
  uic->cwin.candidate_index = index;
  layout_candidate(uic);
}

static void
cand_end_cb(void *ptr)
{
  IMUIMContext *uic = ptr;
  gtk_widget_hide(uic->cwin.top_win);
  uic->cwin.is_active = 0;
}


/*
 * CONSTRUCTOR
 */
GtkIMContext *
im_module_create (const gchar *context_id)
{
  GtkObject *obj = g_object_new(type_im_uim, NULL);
  IMUIMContext *uic = IM_UIM_CONTEXT(obj);
  uic->win = NULL;
  uic->menu = NULL;
  uic->stat_win = NULL;
  uic->pseg = 0;
  uic->nr_psegs = 0;

  uic->cwin.top_win = NULL;
  uic->cwin.clist = NULL;
  uic->cwin.select_count = 0;


  if (!strncmp(context_id, "uim-", 4)) {
    context_id = &context_id[4];
  }
  uic->uc =  uim_create_context(uic, "UTF-8",
				NULL, (char *)context_id,
				im_uim_commit_string);
  uim_set_mode_cb(uic->uc, mode_cb);
  uim_set_preedit_cb(uic->uc, clear_cb,
		     pushback_cb, update_cb);
  uim_set_mode_list_update_cb(uic->uc,
			      update_mode_list_cb);
  uim_set_candidate_cb(uic->uc,
		       cand_begin_cb,
		       cand_update_cb,
		       cand_end_cb);
  
  uic->mode = uim_get_current_mode(uic->uc);
  uic->slave = g_object_new(GTK_TYPE_IM_CONTEXT_SIMPLE,
			    NULL);
  g_signal_connect(G_OBJECT(uic->slave), "commit",
		   G_CALLBACK(im_uim_commit_cb), uic);
  update_status_menu(uic);
  /**/
  uic->next = context_list.next;
  uic->prev = (IMUIMContext *)&context_list;
  context_list.next->prev = uic;
  context_list.next = uic;
  return GTK_IM_CONTEXT(uic);
}

void
im_module_list(GtkIMContextInfo ***contexts,
	       int *n_contexts)
{
  uim_context uc;
  int nr, i;
  static GtkIMContextInfo **info_list;

  uim_init();
  uc = uim_create_context(NULL, "UTF-8",
			  NULL, NULL, NULL);
  nr = uim_get_nr_im(uc);

  info_list = malloc(sizeof(GtkIMContextInfo*)*nr);
  for (i = 0; i < nr; i++) {
    char *name = uim_get_im_name(uc, i);
    char *s = alloca(strlen(name) + 5);
    sprintf(s, "uim-%s", name);
    free(name);
    info_list[i] = malloc(sizeof(GtkIMContextInfo));
    info_list[i]->context_id = strdup(s);
    info_list[i]->context_name = strdup(s);
    info_list[i]->domain = "gtk+";
    info_list[i]->domain_dirname = "";
    info_list[i]->default_locales = uim_get_im_language(uc, i);
  }
  uim_release_context(uc);
  *contexts = info_list;
  *n_contexts = nr;
}

void
im_module_init(GTypeModule *type_module)
{
  uim_init();
  context_list.next = (IMUIMContext *)&context_list;
  context_list.prev = (IMUIMContext *)&context_list;
  type_im_uim =
    g_type_module_register_type(type_module,
				GTK_TYPE_IM_CONTEXT,
				"GtkIMContextUIM",
				&class_info, 0);

  
}

void
im_module_exit()
{
}
