<?php
/**
 * JoinPointCreator.class
 *
 * @package aowp.aspect.joinpoint
 */
class AOWP_JoinPointCreator {

	/**
	 * ASTからジョインポイントを作ります。
	 *
	 * @param AOWP_PHPElement $ast PHPの構文要素。AOWP_PHPElementのサブクラス。
	 * @return AOWP_JoinPoint or null
	 */
	public static function createJoinPoint($ast, $fileName) {
		switch (get_class($ast)) {
			case 'AOWP_PHPRootElement':
				return new AOWP_ScriptExecutionJoinPoint($ast, $fileName);
			case 'AOWP_PHPFunctionElement';
				return new AOWP_FunctionExecutionJoinPoint($ast);
			case 'AOWP_PHPMethodElement':
				return new AOWP_MethodExecutionJoinPoint($ast);
			case 'AOWP_PHPFunctionCallElement':
				return new AOWP_FunctionCallJoinPoint($ast);
			case 'AOWP_PHPFunctionCallWithVariableElement':
				return new AOWP_FunctionCallJoinPoint($ast);
			case 'AOWP_PHPObjectPropertyElement':
				if (self::isMethodCallJoinPoint($ast)) {
					return new AOWP_MethodCallJoinPoint($ast);
				}
				else if (self::isFieldSetJoinPoint($ast)) {
					return new AOWP_FieldSetJoinPoint($ast); 
				}
				else {
					return new AOWP_FieldGetJoinPoint($ast);
				}
			case 'AOWP_PHPFileIncludeStatementElement':
				return new AOWP_ScriptCallJoinPoint($ast);
			case 'AOWP_PHPNewExprElement':
				return new AOWP_ConstructorCallJoinPoint($ast);
			case 'AOWP_PHPReferenceVariableElement': // Array
				if (self::is_ArraySetJoinPoint($ast)) {
					/**
					 * Example :
					 * follows array $peach isnot set join point.
					 * 
					 * 		$momo[$peach['melon']] = 0;
					 * 
					 * But array $momo is set join point.
					 */
	//				print_r($ast->getParent());
					return new AOWP_ArraySetJoinPoint($ast);
				}
				else {
					return new AOWP_ArrayGetJoinPoint($ast);
				}
			case 'AOWP_PHPVariableElement':
				/**
				 * Not array reference variable, and not method call's instance.
				 *
				 * Example :
				 * follows $val is NOT global variable join point.
				 * 		$val[] = 'hoge';
				 * !($ast->getParent() instanceof AOWP_PHPReferenceVariableElement)
				 *
				 * Example :
				 * follows $val is NOT global variable join point.
				 * 		$val->method();
				 * !($ast->getParent() instanceof AOWP_PHPObjectOperatorElement)
				 * 
				 * Example :
				 * follows $val is NOT global variable join point.
				 * 		$val();
				 * !($ast->getParent() instanceof AOWP_PHPFunctionCallWithVariableElement)
				 * 
				 * Example :
				 * follows $val is NOT global variable join point.
				 * 		$res = Class::$val();
				 * && !($ast->getParent() instanceof AOWP_PHPStaticMethodCallWithVariableElement)
				 */ 
				if ( !($ast->getParent() instanceof AOWP_PHPReferenceVariableElement)
					&& !($ast->getParent() instanceof AOWP_PHPObjectOperatorElement )
					&& !($ast->getParent() instanceof AOWP_PHPFunctionCallWithVariableElement)
					&& !($ast->getParent() instanceof AOWP_PHPStaticMethodCallWithVariableElement)
					&& self::isDeclaredGlobal($ast)) {
	
					if (self::isGlobalVariableSet($ast)) { 
						return new AOWP_GlobalVariableSetJoinPoint($ast);
					}
					else {
						return new AOWP_GlobalVariableGetJoinPoint($ast);
					}	
				}
				else {
					return null; // Array reference variable or local variable.	 
				}
			// unpopulated
			case 'AOWP_PHPStaticMethodCallElement':
//				print_r($ast->getParent());
//				return new AOWP_ClassConstVariableRefJoinPoint($ast);
				return null;
			// unpopulated
			case 'AOWP_PHPClassConstVariableRefElement':
//				print_r($ast->getParent());
				/**
				 * Example :
				 * follows method IS class const variable ref join point.
				 * 
				 * 		$res = Hoge::method()
				 */
//				return new AOWP_ClassConstVariableRefJoinPoint($ast);
				return null;
			// unpopulated
			case 'AOWP_PHPStaticMethodCallWithVariableElement';
//				print_r($ast->getParent());
				/**
				 * Example :
				 * follows $method IS static method call with variable element.
				 * 
				 * 		$res = Hoge::$method();
				 * 
				 */
//				return new AOWP_StaticMethodCallWithVariableJoinPoint($ast);
				return null;
			default:
				return null;
		}
	}
	
	/**
	 * Returns if the ast element is global variable set join point.
	 */
	public static function isGlobalVariableSet (AOWP_PHPVariableElement $ast)
	{
		/**
		 * AOWP_PHPEqualExprElement has three parameters, variable, operatorName and expr.
		 * If target elem appears in left expr, then the parameter of variable has it.
		 */
		return ($ast->getParent() instanceof AOWP_PHPEqualExprElement )
				&& ($ast->getParent()->variable === $ast);
	}
	
	/**
	 * Returns if the ast element is method call join point.
	 */
	private static function isMethodCallJoinPoint (AOWP_PHPObjectPropertyElement $ast)
	{
		return AOWP_ASTUtility::isArrayElement($ast->arguments);
	}
	
	/**
	 * Returns if the ast element is field set join point.
	 */
	public static function isFieldSetJoinPoint (AOWP_PHPObjectPropertyElement $ast)
	{
		/**
		 * Example :
		 * follows field1 is field set join point.
		 * 		$obj->field1 = $val;
		 * 
		 * follows field1 isn't field set join poin.
		 * 		list(, $obj->field1) = $val;
		 */
		return ($ast->getParent()->getParent() instanceof AOWP_PHPEqualExprElement)
				&& ($ast->getParent() instanceof AOWP_PHPObjectOperatorElement)
				&& ($ast->getParent()->objectProperties[count($ast->getParent()->objectProperties) -1 ] === $ast);
	}
	
	/**
	 * Returns if the ast element is array set join point.
	 */
	public static function is_ArraySetJoinPoint (AOWP_PHPReferenceVariableElement $ast)
	{
		return ($ast->getParent() instanceof AOWP_PHPEqualExprElement)
				&& ($ast->getParent()->variable === $ast);
	}
	
	private static function isDeclaredGlobal ($ast)
	{
		$pos = AOWP_ASTConverter::getInsertableSpace($ast);
		$val = self::_getVariableName($ast);
		
		// Check if super global variable.
		if (
				$val==_POST ||
				$val==_GET ||
				$val==_FILES ||
				$val==_SERVER ||
				$val==_ENV ||
				$val==_COOKIE ||
				$val==_SESSION ||
				$val==_REQUEST 
			)
			return true;
			
		// Check if declared at global area.
		if ($pos->getParent() instanceof AOWP_PHPRootElement)
		{
			return true;
		}
		
		// Check if declared as global variable.
		if (AOWP_ASTUtility::isArrayElement($pos->getParent()->innerStatements))
		{
			for ($i=0; $i<count($pos->getParent()->innerStatements); $i++)
			{
				if($pos->getParent()->innerStatements[$i] instanceof AOWP_PHPDefineVariableStatementElement)
				{
					if ($pos->getParent()->innerStatements[$i]->type=='global')
					{
						$variables = $pos->getParent()->innerStatements[$i]->variables;
						for ($j=0; $j<count($variables); $j++)
						{
							$valName = self::_getVariableName($variables[$j]);
							if ($valName==$val)
							{
								return true;
							}
						}
					}
				}
			}
		}
		return false;
	}
	
	private static function _getVariableName ($ast)
	{
		return preg_replace('/^(\$)/', '', AOWP_TemplateEngine::scriptToSource($ast));
	}
}

?>