<?php
/**
 * Utility methods for converting AOWP_AST.
 *
 * @package aowp.aspect.joinpoint.util
 */
class AOWP_ASTConverter {

	/**
	 * Not permitted to create instance.
	 */
	private function __construct ()
	{
	}

	/**
	 * Returns new statement.
	 *
	 * @param String $statement statement
	 * @return AOWP_PHPStatementElement new statement
	 */
	public static function createStatement ($statement)
	{
		$elem =  self::exchangePartialSourceToASTElement($statement);
		return self::extractStatement($elem);
	}

	/**
	 * Returns new statment.
	 */
	public static function createStatementWithUnusedVariable ($counter, $rightExpr=null)
	{
		if ( is_string($rightExpr)) {
			return self::createStatement(
				'${AOWP_UnusedVariableCreator::getUnusedVariableNameAt('.$counter.')} = '.$rightExpr.';');
		} else {
			$statement = self::createStatement(
				'${AOWP_UnusedVariableCreator::getUnusedVariableNameAt('.$counter.')} = $hoge;');
			self::setRightExpr($statement, $rightExpr);
			return $statement;
		}
	}

	private static function exchangePartialSourceToASTElement($source)
	{
		$source = '<?php ' . $source . ' ?>';
		return PHP_Parser::parse($source);
	}

	private static function extractStatement($ASTElement)
	{
		$statement = $ASTElement->globals[1];
		return $statement;
	}

	/**
	 * Returns statements.
	 *
	 * @param AOWP_PHPElement $elem element
	 * @return AOWP_PHPStatementElement statement element
	 */
	public static function getStatements ($elem)
	{
		if (($statementsType=self::getStatementsType($elem)) == null) {
			return null;
		}
		return $elem->{$statementsType};
	}

	/**
	 * Set statements.
	 */
	public static function setStatements ($elem, $statements)
	{
		if (($statementsType=self::getStatementsType($elem)) == null) {
			return null;
		}
		$elem->{$statementsType} = $statements;
	}

	/**
	 * Returns first element.
	 *
	 * @param AOWP_PHPElement $elem
	 * @return AOWP_PHPElement first element
	 */
	public static function getFirstStatement ($elem)
	{
		if (($statementsType=self::getStatementsType($elem)) == null) {
			return null;
		}
		return $elem->{$statementsType}[0];
	}

	/**
	 * Returns last element.
	 *
	 * @param AOWP_PHPElement $elem
	 * @return AOWP_PHPElement last element
	 */
	public static function getLastStatement ($elem)
	{
		if (($statementsType=self::getStatementsType($elem)) == null) {
			return null;
		}
		return $elem->{$statementsType}[count($elem->{$statementsType})-1];
	}

	/**
	 * Returns element at.
	 *
	 * @param AOWP_PHPElement $elem
	 * @param int $idx
	 * @return AOWP_PHPElement element
	 */
	public static function getStatementAt ($elem, $idx)
	{
		if (($statementsType=self::getStatementsType($elem)) == null) {
			return null;
		}
		return $elem->{$statementsType}[$idx];
	}

	/**
	 * Set target at.
	 */
	public static function setStatementAt ($elem, $target, $idx)
	{
		if (($statementsType=self::getStatementsType($elem)) == null) {
			return null;
		}
		if (!self::isValidPosition($elem, $idx)) {
			throw new RuntimeException('Set statement at failed. Invalid insert position.');
		}
		$elem->{$statementsType}[$idx] = $target;
		self::updateParentInfo($elem);
	}

	/**
	 * Inserts new elem.
	 *
	 * @param AOWP_PHPElement $elem This element must have statements as property.
	 * @param AOWP_PHPElement $target
	 * @param int $idx insert position.
	 */
	public static function insertStatementAt ($elem, $target, $idx)
	{
		if (($statementsType=self::getStatementsType($elem)) == null) {
			return null;
		}
		if (!self::isValidPosition($elem, $idx)) {
			throw new RuntimeException('Insert failed. Invalid insert position.');
		}
		$elem->{$statementsType} = self::array_insert($elem->{$statementsType}, $target, $idx);
		self::updateParentInfo($elem);
	}

	public static function array_insert ($originalArray, $targetArray, $idx)
	{
		if(!is_array($targetArray)) $targetArray = array($targetArray);
		return $originalArray = array_merge(
		array_slice($originalArray, 0, $idx), $targetArray,  array_slice($originalArray, $idx) );
	}

	/**
	 *
	 */
	public static function isValidPosition ($elem, $idx)
	{
//		AOWP_Logger::logging('[' . get_class($elem) . '] idx: ' . $idx);
		if ( AOWP_ASTConverter::isNO_ARRAY($idx) ) {
			return false;
		}
//		AOWP_Logger::logging('[' . get_class($elem) . '] statement type: ' . AOWP_ASTConverter::getStatementsType($elem));
		if (($statementsType = AOWP_ASTConverter::getStatementsType($elem)) == null) {
			return false;
		}
//		AOWP_Logger::logging('[' . get_class($elem) . '] statements type count: ' . count($elem->$statementsType) . '. idx: ' . $idx);
		$ct = count($elem->{$statementsType});
		return $ct >= $idx;
	}

	/**
	 * Returns if the idx is NO_ARRAY.
	 */
	public static function isNO_ARRAY ($idx)
	{
		return strcmp($idx, 'NO_ARRAY') == 0;
	}

	/**
	 * Append new element..
	 *
	 * @param AOWP_PHPElement $elem This element must have statements as property.
	 * @param AOWP_PHPElement $target
	 */
	public static function appendStatement ($elem, $target)
	{
		if (($statementsType=self::getStatementsType($elem)) == null) {
			return null;
		}
		$elem->{$statementsType}[] = $target;
		self::updateParentInfo($elem);
	}

	/**
	 * Shift statements.
	 */
	public static function shift ($elem)
	{
		if (($statementsType=self::getStatementsType($elem)) == null) {
			return null;
		}
		return array_shift($elem->{$statementsType});
	}

	/**
	 * Returns the position which the taget inserted at.
	 */
	public static function getAdviceInsertedPosition ($pos, $targetAdvice, $statementType = innerStatements)
	{
		if ( AOWP_ASTUtility::isArrayElement($pos->getParent()->{$statementType}))
		;
		else if ( AOWP_ASTUtility::isArrayElement($pos->getParent()->statements) )
		$statementType = statements;
		else
		throw new RuntimeException('Get advice inserted position failed.');

		$MAX = 50;
		for ($i=0; $i<$MAX ;$i++)
		{
			for ($j=0; $j<count($targetAdvice); $j++)
			{
				if (get_class($pos->getParent()->{$statementType}[$i])==$targetAdvice[$j])
				{
					return $i-1;
				}
			}
		}

		throw new RuntimeException('Get advice inserted position failed. Target Element Not found.');
	}


	/**
	 * Replace original element which appears in a statements for the new statement.
	 *
	 * cf
	 * Target element is func().
	 *
	 * 		$res = 1 + 2 + func() + 4;
	 *
	 * Then converts above statement as follows.
	 *
	 * 		$res = 1 + 2 + ${AOWP_UnusedVariableCreator::getUnusedVariableNameAt(?)} + 4;
	 *
	 * Note:
	 * This method only can use when target ast doesn't have properties.
	 * If target ast has properties, use ObjectProperty::replacePropertiesWithUnusedVariable().
	 */
	public static function replaceOriginalElementFor ($ast, $statement, $callType=AOWP_PHPFunctionCallElement)
	{
		$pos = $ast->getParent();

		foreach ($pos as $key => $value)
		{
			if (get_class($pos->{$key})==$callType)
			{
				$temp = $pos->{$key}; // save
				$pos->{$key} = $statement->expr->variable;
				return $temp;
			}
		}

		for ($i=0; $i<count($pos->exprs); $i++)
		{
			if( get_class($pos->exprs[$i])==$callType )
			{
				$pos->$key = $statement->expr->variable;
				return;
			}
		}

		foreach ($ast->expr as $key => $value)
		{
			if( get_class($ast->expr->$key)==$callType )
			{
				$ast->expr->$key = $statement->expr->variable;
				return;
			}
		}

		throw new RuntimeException('Replace original element failed.');
	}

	/**
	 * Returns if the ast appears at left expr.
	 *
	 * cf.
	 * If the $ast appears as follows, then return true.
	 * $ast = 'hoge';
	 *
	 * Note:
	 * If the statement which $ast appears isn't AOWP_PHPEqualExprElement then this function returns false.
	 * cf.
	 * 		echo $ast;
	 */
	//	public static function isDeclaredAtLeftExpr ($ast)
	//	{
	//		for ( $pos=$ast; !($ast instanceof AOWP_PHPRootElement); $pos=$pos->getParent())
	//		{
	//
	//			if ( AOWP_ASTUtility::isNullElement($pos) )
	//				return false;
	//
	//			if ( AOWP_ASTUtility::isASTElement($pos->variable)
	//				&& ($pos instanceof AOWP_PHPEqualExprElement)
	//			)
	//			{
	//				$flattened_innerStatements = AOWP_ASTWalker::getASTArrayForWalk($pos->variable);
	//				foreach ($flattened_innerStatements as $line)
	//				{
	//					// Check if AOWP_PHPEqualExprElement.
	//					if ( ($line===$ast)&&($pos->operatorName=='=') )
	//					{
	//						return true;
	//					}
	//				}
	//			}
	//		}
	//		return false;
	//	}

	/**
	 * Returns expr of elem.
	 *
	 * @param AOWP_PHPElement $elem
	 * @return AOWP_PHPElement expr
	 */
	public static function getExpr ($elem)
	{
		return $elem->expr;
	}

	/**
	 * Returns left expr of equal statement.
	 */
	public static function getLeftExpr ($elem)
	{
		if ( !($elem->expr instanceof AOWP_PHPEqualExprElement) ) {
			throw new RuntimeException ('Get left expr failed. First args must be AOWP_PHPEqualExprElement');
		}
		return $elem->expr->variable;
	}

	/**
	 * Returns right expr of equal statement.
	 */
	public static function getRightExpr ($elem)
	{
		if ( !($elem->expr instanceof AOWP_PHPEqualExprElement) ) {
			throw new RuntimeException ('Get right expr failed. First args must be AOWP_PHPEqualExprElement');
		}
		return $statement->expr->expr;
	}

	/**
	 * Set left expr to equal statement.
	 */
	public static function setLeftExpr ($equalExpr, $leftExpr)
	{
		if ( !($equalExpr->expr instanceof AOWP_PHPEqualExprElement) ) {
			throw new RuntimeException ('Set left expr failed. First args must be AOWP_PHPEqualExprElement');
		}
		$equalExpr->expr->variable = $leftExpr;
	}

	/**
	 * Set right expr to equal statement.
	 */
	public static function setRightExpr ($equalExpr, $rightExpr)
	{
		if ( !($equalExpr->expr instanceof AOWP_PHPEqualExprElement) ) {
			throw new RuntimeException ('Set right expr failed. First args must be AOWP_PHPEqualExprElement');
		}
		$equalExpr->expr->expr = $rightExpr;
	}

	/**
	 * Exchange right expr and left expr.
	 */
	public static function exchangeLeftExprAndRightExpr ($statement)
	{
		if ( $statement->expr instanceof AOWP_PHPEqualExprElement )
		{
			$temp = $statement->expr->variable;                  // LeftExpr
			$statement->expr->variable = $statement->expr->expr; // RightExpr
			$statement->expr->expr = $temp;
		} else {
			throw new RuntimeException('Exchange Right expr and left failed.');
		}
	}

	/**
	 * Returns properties line as string.
	 *
	 * Example :
	 * Target object property is method1().
	 *
	 * 		$obj->field1->field2->method1()->method2()
	 *
	 * then returns,
	 *
	 * 		$obj->field1->field2->method1()
	 *
	 * as String.
	 *
	 * @param AOWP_PHPObjectPropertyElement $propLine properties line
	 * @return String properties line
	 */
	public static function getPropertiesLine (AOWP_PHPObjectPropertyElement $propLine)
	{
		$line = self::getPrePropertiesLine($propLine);
		$line.='->';
		$line.= AOWP_TemplateEngine::scriptToSource($propLine);
		return $line;
	}

	/**
	 * Return pre properties line as string.
	 *
	 * Example :
	 * Target object property is method1().
	 *
	 * 		$obj->field1->field2->method1()->method2()
	 *
	 * then returns,
	 *
	 * 		$obj->field1->field2
	 *
	 * as String. This strings represent an instance of the method1().
	 *
	 * @param AOWP_PHPObjectPropertyElement $propLine properties line
	 * @return String properties line
	 */
	public static function getPrePropertiesLine (AOWP_PHPObjectPropertyElement $objProp)
	{
		$o = $objProp->getParent();

		if (!AOWP_ASTUtility::isNullElement($o->expr->variableName)) {
			$line = $o->expr->variableName.'->';
		} else {
			$arguments = $o->expr;
		}

		// AOWP_PHPCompoundVariableElement.
		if (!AOWP_ASTUtility::isNullElement($expr = $o->expr))
		{
			$line = AOWP_TemplateEngine::scriptToSource($expr).'->';
			$arguments = $o->objectProperties;
		}

		$arguments = $o->objectProperties;
		for ( $i=0; $arguments[$i]!=$objProp && $arguments[$i]!=null; $i++)  // && $arguments[$i]!=null?
		{
			$line.=AOWP_TemplateEngine::scriptToSource($arguments[$i]).'->';
		}
		$line = preg_replace('/->$/', '', $line);

		return $line;
	}

	/**
	 * Replace target property with unused variable.
	 *
	 * Example1 :
	 * Target object property is field2
	 *
	 * 		$tmp = $val->field1->method1()->field2->method2();
	 *
	 * Converts as follows.
	 *
	 * 		$tmp = ${AOWP_UnusedVariableCreator::getUnusedVariableNameAt(0)}->method2();
	 *
	 *
	 * Example2 :
	 * Target object property is method2
	 *
	 * 		$tmp = $val->field1->method1()->field2->method2();
	 *
	 * Converts as follows.
	 *
	 * 		$tmp = ${AOWP_UnusedVariableCreator::getUnusedVariableNameAt(0)};
	 *
	 *
	 * @param AOWP_PHPObjectPropertyElement $elem prop
	 * @param AOWP_PHPStatementElement $statement statement
	 */
	public static function replacePropertiesWith (AOWP_PHPObjectPropertyElement &$elem, $statement)
	{
		if (self::_hasMoreThanOneProperties($elem))
		{
			//			echo 'removed all prop<br/>';
			// cf echo 'test'.$fuite->melon->getName('I ', 'like ');
			//          LeftExpr Operator RightExpr
			$tmp = $elem->getParent()->getParent();
			foreach ($tmp as $key => $value)
			{
				if ($value==$elem->getParent())
				{
					//					echo 'find prop0<br/>';
					$tmp->{$key} = $statement->expr->variable;
					return;
				}
			}
				
			// cf echo $fuite->$piyo()->getName('This ', 'is a ');
			//         exprs
			$tmp = $elem->getParent()->getParent();
			for ($i=0; $i<count($tmp->exprs); $i++)
			{
				if ($tmp->exprs[$i]==$elem->getParent())
				{
					//					echo 'find prop1<br/>';
					$tmp->exprs[$i] = $statement->expr->variable;
					return;
				}
			}
				
			// follows wrong
			//			$tmp = $elem->getParent()->getParent()->exprs;
			//			for ($i=0; $i<count($tmp); $i++)
			//			{
			//				if ($tmp[$i]==$elem->getParent())
			//				{
			//					echo 'find prop1<br/>';
			//					echo get_class($tmp[$i]);
			//					print_r($tmp[$i]);
			//					$tmp[$i] = $statement->expr->variable;
			//					return;
			//				}
			//			}
				
			throw new RuntimeException('Replace properties failed.');
				
			} else {
				self::removePropertiesTo($elem);
				$elem->getParent()->expr = self::getLeftExpr($statement);
			}
		}

		private static function _hasMoreThanOneProperties ($elem)
		{
			$idx = self::getTargetPropertyIndex($elem);
			return (($idx+1)==count($elem->getParent()->objectProperties));
		}

		/**
		 * Returns an index where the target property appears.
		 *
		 * Example :
		 * Target property is field2
		 *
		 * 		$tmp = $val->field1->method1()->field2->method2();
		 *
		 * then return 2.
		 * 		Index of field1 is 0.
		 * 		Index of method1() is 1.
		 *
		 * @param AOWP_PHPObjectPropertyElement $prop prop
		 */
		public static function getTargetPropertyIndex (AOWP_PHPObjectPropertyElement $prop)
		{
			for ($i=0; $i<count($prop->getParent()->objectProperties); $i++)
			{
				if ($prop->getParent()->objectProperties[$i]==$prop)
				{
					return $i;
				}
			}

			return -1;
		}

		/**
		 * Returns object property at.
		 */
		public static function getPropertyAt (AOWP_PHPObjectPropertyElement $prop, $idx)
		{
			return $prop->getParent()->objectProperties[$idx];
		}

		/**
		 * Remove object propertyies from head to target property.
		 *
		 * Example1 :
		 * Target property is field2
		 *
		 * 		$tmp = $val->field1->method1()->field2->method2();
		 *
		 * Converts as follows.
		 *
		 * 		$tmp = $val->method2();
		 *
		 *
		 * Example2 :
		 * Target property is method2
		 *
		 * 		$tmp = $val->field1->method1()->field2->method2();
		 *
		 * Converts as follows.
		 *
		 * 		$tmp = $val->;
		 *
		 * This method simply removes object properties.
		 *
		 * @param AOWP_PHPObjectPropertyElement $prop prop
		 */
		public static function removePropertiesTo($prop)
		{
			foreach ($prop->getParent()->objectProperties as $objProp)
			{
				$ret = array_shift($prop->getParent()->objectProperties);
				if ($ret==$prop)
				{
					break;
				}
			}
		}

		/**
		 * Replace pre properties for an unused variable.
		 *
		 * Example :
		 * Target is method1.
		 *
		 * 		$val->field1->field2->method1();
		 *
		 * Converts as follows.
		 *
		 * 		${AOWP_UnusedVariableCreator::getUnusedVariableNameAt(0)}->method1();
		 */
		public static function replacePrePropertiesWith (AOWP_PHPObjectPropertyElement $elem, $statement)
		{
			$idx = self::getTargetPropertyIndex($elem);
			for ($i=0; $i<$idx; $i++)
			{
				array_shift($elem->getParent()->objectProperties);
			}

			$elem->getParent()->expr = self::getLeftExpr($statement);
		}

		/**
		 * Return insertable space, statements, classStatements or innerStatements.
		 * These statements are an array.
		 *
		 * @param AOWP_PHPElement $element element
		 * @return Array insertable space
		 */
		public static function getInsertableSpace (AOWP_PHPElement $element)
		{
			if ($element instanceof AOWP_PHPRootElement){
				return $element; // root->statements
			}

			// look up insertable space.
			$insertableSpaceElement = $element;
			while (AOWP_ASTConverter::getStatementsType($insertableSpaceElement->getParent()) === null) {
				$insertableSpaceElement = $insertableSpaceElement->getParent();
			}

			// get statement or statements type.
			$parentInsertableSpaceElement = $insertableSpaceElement->getParent();
			$parentStatmentType = AOWP_ASTConverter::getStatementsType($parentInsertableSpaceElement);

			// insertable space is statements, classStatements or innerStatements, that is an array.
			if (($parentStatmentType == 'statements') || ($parentStatmentType == 'classStatements')) {
				return $insertableSpaceElement;
			}
			else if ($parentStatmentType == 'innerStatements') {
				// look up in for exprs.
				$parentInsertableSpaceElement = $insertableSpaceElement->getParent();
				if ($parentInsertableSpaceElement instanceof AOWP_PHPForStatementElement) {
					if (AOWP_ASTConverter::_isForLoopExpr($parentInsertableSpaceElement, $element)) {
						return $parentInsertableSpaceElement;
					}
					else {
						return $insertableSpaceElement;
					}
				}
				else if ($parentInsertableSpaceElement instanceof AOWP_PHPIfStatementElement) {
					if (self::_isIfExpr($parentInsertableSpaceElement, $element)) {
						return $parentInsertableSpaceElement;
					}
					else {
						return $insertableSpaceElement;
					}
				}
				else {
					return $insertableSpaceElement;
				}
			}
			else {
				throw new RuntimeException('Get insertable Space failed.');
			}
		}

		private static function _isForLoopExpr ($forElement, $element) {
			// initial Exprs
			for ($i = 0; $i < count($forElement->initialExprs); $i++) {
				$flattened_innerStatements = AOWP_ASTWalker::getASTArrayForWalk($forElement->initialExprs[$i]);
				foreach ($flattened_innerStatements as $line) {
					if ($line === $element) {
						return true;
					}
				}
			}
			// conditions
			for ($i = 0; $i < count($forElement->conditions); $i++) {
				$flattened_innerStatements = AOWP_ASTWalker::getASTArrayForWalk($forElement->conditions[$i]);
				foreach ($flattened_innerStatements as $line) {
					if ($line === $element) {
						return true;
					}
				}
			}
			// updateExprs
			for ($i = 0; $i < count($forElement->updateExprs); $i++) {
				$flattened_innerStatements = AOWP_ASTWalker::getASTArrayForWalk($forElement->updateExprs[$i]);
				foreach ($flattened_innerStatements as $line) {
					if ($line === $element) {
						return true;
					}
				}
			}
			return false;
		}

		private static function _isIfExpr ($ifElement, $element) {
			$flattened_innerStatements = AOWP_ASTWalker::getASTArrayForWalk($ifElement->condition);
			foreach ($flattened_innerStatements as $lineStatement) {
				if($lineStatement === $element) {
					return true;
				}
			}
			return false;
		}

		/**
		 * Returns elements as array which appears within flattened target elem.
		 *
		 * @param AOWP_PHPElement $elem
		 * @param String $className
		 * @return Array elements
		 */
		public static function getElementsByClassName ($elem, $className)
		{
			$flattened_statements = AOWP_ASTWalker::getASTArrayForWalk($elem);
			foreach ($flattened_statements as $line)
			{
				if ($line instanceof $className) {
					$elements[] = $line;
				}
			}
			return $elements;
		}

		/**
		 * Reassigns new index, and set new Parent to every statements.
		 *
		 * @param AOWP_PHPElement $elem
		 */
		public static function updateParentInfo($elem)
		{
			$statementsType = self::getStatementsType($elem);
			for ($i=0; $i<count($elem->{$statementsType}); $i++)
			{
				if ( AOWP_ASTUtility::isASTElement($elem->{$statementsType}[$i]) ) {
					$elem->{$statementsType}[$i]->setParentInfo($elem, $statementsType, $i);
				} else {
					continue; // skip non object element.
				}
			}
		}

		/**
		 * Returns statements type.
		 *
		 * @param AOWP_PHPElement
		 * @return String statements type
		 */
		public static function getStatementsType ($elem)
		{
			if (AOWP_ASTUtility::isArrayElement($elem->statements)) {
				return 'statements';
			}
			else if (AOWP_ASTUtility::isASTElement($elem->statements)) {
				$elem->statements = array($elem->statements);
				$elem->statements[0]->setParentInfo($elem, statements, 0);
				return 'statements';
			}
			else if (AOWP_ASTUtility::isArrayElement($elem->classStatements)) {
				return 'classStatements';
			}
			else if (AOWP_ASTUtility::isASTElement($elem->classStatements)) {
				$elem->classStatements = array($elem->classStatements);
				$elem->classStatements[0]->setParentInfo($elem, classStatements, 0);
				return 'classStatements';
			}
			else if (AOWP_ASTUtility::isArrayElement($elem->innerStatements)) {
				return 'innerStatements';
			}
			else if (AOWP_ASTUtility::isASTElement($elem->innerStatements)) {
				$elem->innerStatements = array($elem->innerStatements);
				$elem->innerStatements[0]->setParentInfo($elem, innerStatements, 0);
				return 'innerStatements';
			}
			else {
				return null;
			}
		}
}
?>
