﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Xml.Serialization;
using System.Text.RegularExpressions;
using NaGet.Packages;

namespace NaGet.Packages.Install
{	
	public class InstalledPackage : Package
	{
		/// <summary>
		/// アンインストール情報
		/// </summary>
		public UninstallInformation UninstallInfo;
		
		/// <summary>
		/// インストール先フォルダを返す。
		/// <code>UninstallInfo.InstallLocation</code>が空ならば、
		/// 解決を試みる。
		/// </summary>
		/// <returns>インストール先フォルダ。見つからなかったら<code>null</code></returns>
		public string discoverInstalledLocation()
		{
			if (string.IsNullOrEmpty(UninstallInfo.InstallLocation)) {
				// ○○/名前を探す
				string[] suggestions = new string[3];
				suggestions[0] = Environment.GetEnvironmentVariable("ProgramFiles");
				if (!string.IsNullOrEmpty(Environment.GetEnvironmentVariable("ProgramFiles(x86)"))) {
					suggestions[1] = Environment.GetEnvironmentVariable("ProgramFiles(x86)");
				}
				suggestions[2] = Path.GetPathRoot(suggestions[0]);

				foreach (string suggest in suggestions) {
					if (!string.IsNullOrEmpty(suggest)) {
						string path = Path.Combine(suggest, this.Name);
						if (Directory.Exists(path)) return path;
					}
				}
				
				return null; // when not found
			} else {
				return UninstallInfo.InstallLocation;
			}
		}
		
		public static InstalledPackage PackageConverter(Package basePkg) 
		{
			InstalledPackage pkg = new InstalledPackage();
			
			NaGet.Utils.FieldCopy(basePkg, ref pkg);
			pkg.UninstallInfo = UninstallInformation.NewInstance(basePkg);
			
			return pkg;
		}
		
		public static InstalledPackage PackageConverter(Package basePkg, UninstallInformation info)
		{
			InstalledPackage instPkg = new InstalledPackage();
			
			NaGet.Utils.FieldCopy(basePkg, ref instPkg);
			instPkg.UninstallInfo = info;
			
			Match match = Regex.Match(info.DisplayName, basePkg.UninstallerKey);
			if (! match.Success) {
				throw new ArgumentException(string.Format("{0}({1}) does not matched for {2}.", basePkg.Name, basePkg.Version, info.DisplayName));
			} else if (match.Groups[1].Success) { // DisplayNameの方のバージョン表記を優先
				instPkg.Version = match.Groups[1].Value;
			} else if (! string.IsNullOrEmpty(info.DisplayVersion) ) {
				instPkg.Version = info.DisplayVersion;
			} else {
				instPkg.Version = string.Empty;
			}
			
			return instPkg;
		}
	}
}
