﻿using System;
using System.Diagnostics;
using System.Text.RegularExpressions;
using System.Collections;
using System.CodeDom.Compiler;
using System.IO;

namespace NaGet.Packages.Install
{
	/// <summary>
	/// Description of Uninstallation.
	/// </summary>
	public class Uninstallation
	{
		/// <summary>
		/// アンインストールするパッケージ
		/// </summary>
		public InstalledPackage UninstalledPackage;
		
		/// <summary>
		/// 外部アプリのエラー出力の受信ハンドラ
		/// </summary>
		public event EventHandler<NaGet.Utils.AnyDataEventArgs<string>> ErrorDataReceived;
		
		/// <summary>
		/// 外部アプリの標準出力の受信ハンドラ
		/// </summary>
		public event EventHandler<NaGet.Utils.AnyDataEventArgs<string>> OutputDataReceived;
		
		/// <summary>
		/// サイレントアンインストールするか否か
		/// </summary>
		public bool Silent = false;
		
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="package">アンインストールするパッケージ</param>
		public Uninstallation(InstalledPackage package)
		{
			UninstalledPackage = package;
		}
		
		/// <summary>
		/// インストールされた状態か否か
		/// </summary>
		public bool Installed
		{
			get {
				if (UninstalledPackage.Type == InstallerType.ARCHIVE) {
					return Directory.Exists(UninstalledPackage.UninstallInfo.InstallLocation);
				} else {
					foreach (UninstallInformation info in RegistriedUninstallers.Uninstallers) {
						if (! string.IsNullOrEmpty(UninstalledPackage.UninstallerKey)) {
							Match match = Regex.Match(info.DisplayName, UninstalledPackage.UninstallerKey);
						
							if (match.Success) {
								return true;
							}
						}
					}
				}
				return false;
			}
		}
		
		/// <summary>
		/// アンインストーラ等を起動してアンインストール作業を行う
		/// </summary>
		/// <returns>アンインストーラの終了コード</returns>
		public int Uninstall()
		{
			if (! Installed) {
				throw new ApplicationException("Program not found, may be already uninstalled");
			}
			
			int exitValue = 0;
			string uninstallString = Silent? UninstalledPackage.UninstallInfo.QuietUninstallString : UninstalledPackage.UninstallInfo.UninstallString;
			if (string.IsNullOrEmpty(uninstallString)) {
				throw new ApplicationException(string.Format("Could not found {0}install script", Silent? "silent " : ""));
			}
			
			if (UninstalledPackage.UninstallInfo.WindowsInstaller &&
			    Regex.Match(uninstallString.Substring("MsiExec.exe /I".Length),
			                @"^\{[0-9A-F]{8}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{4}-[0-9A-F]{12}\}$").Success) {
				string guid = uninstallString.Substring("MsiExec.exe /I".Length);
				using (Process hProcess = NaGet.Utils.ProcessStartWithOutputCapture(
					new ProcessStartInfo("msiexec", string.Format("/X{0}", guid)),
					NaGet.Utils.ConvertToDataReceivedEventHandler(OutputDataReceived),
					NaGet.Utils.ConvertToDataReceivedEventHandler(ErrorDataReceived)) ) {
					
					hProcess.WaitForExit();
					
					exitValue = hProcess.ExitCode;
				}
			} else if (File.Exists(uninstallString)) {
				// 単独のファイルの場合
				using (Process hProcess = NaGet.Utils.ProcessStartWithOutputCapture(
					new ProcessStartInfo(uninstallString),
					NaGet.Utils.ConvertToDataReceivedEventHandler(OutputDataReceived),
					NaGet.Utils.ConvertToDataReceivedEventHandler(ErrorDataReceived)) ) {
					
					hProcess.WaitForExit();
					
					exitValue = hProcess.ExitCode;
				}
			} else {
				ProcessStartInfo procInfo = new ProcessStartInfo(null, uninstallString);
				procInfo.UseShellExecute = false;
				if (UninstalledPackage.Type == InstallerType.ARCHIVE) {
					procInfo.CreateNoWindow = true;
				}
				using (NaGet.InteropServices.CreateProcessCaller p = new NaGet.InteropServices.CreateProcessCaller(procInfo)) {
					p.WaitForExit();
					
					exitValue = p.ExitCode;
				}
			}
			
			return exitValue;
		}
		
		public override string ToString()
		{
			return string.Format("{0}({1})", UninstalledPackage.Name, UninstalledPackage.Version);
		}
	}
}
