﻿using System;
using System.IO;
using System.Collections.Generic;
using NaGet.Packages;
using NaGet.Packages.Install;

namespace NaGet.Packages
{
	/// <summary>
	/// パッケージリスト達をひとまとめにして管理する
	/// </summary>
	public class PackageListsManager
	{
		internal PackageList<Package> availablePkgList;
		internal PackageList<InstalledPackage> installedPkgList;
		internal PackageList<InstalledPackage> systemInstalledPkgList;
		
		/// <summary>
		/// ファイルリストから読み込まれたパッケージのリスト
		/// </summary>
		public PackageList<Package> AvailablePkgList {
			get { return availablePkgList; }
		}
		
		/// <summary>
		/// 本ソフトウェアを介してインストールされたパッケージのリスト
		/// </summary>
		public PackageList<InstalledPackage> InstalledPkgList {
			get { return installedPkgList; }
		}
		
		/// <summary>
		/// システムから検出されたパッケージのリスト
		/// </summary>
		public PackageList<InstalledPackage> SystemInstalledPkgList {
			get { return systemInstalledPkgList; }
		}
		
		/// <summary>
		/// インストールログ
		/// </summary>
		private List<InstallationLog> systemInstalledLogList;
		
		/// <summary>
		/// 初期化をする。ファイル群を読み込んで更新がなされる。
		/// </summary>
		public PackageListsManager()
		{
			LoadPackageLists();
		}
		
		#region ファイル読み書き
		
		/// <summary>
		/// ファイルを読み込んで更新する
		/// </summary>
		public void LoadPackageLists()
		{
			try {
				availablePkgList = NaGet.Utils.GetDeserializedObject<PackageList<Package>>(NaGet.Env.PackageListFile);
			} catch (FileNotFoundException) {
				availablePkgList = new PackageList<Package>();
			}
			
			try {
				installedPkgList = NaGet.Utils.GetDeserializedObject<PackageList<InstalledPackage>>(NaGet.Env.ArchiveInstalledPackageListFile);
			} catch (FileNotFoundException) {
				installedPkgList = new PackageList<InstalledPackage>();
			}
			
			try {
				systemInstalledPkgList = NaGet.Utils.GetDeserializedObject<PackageList<InstalledPackage>>(NaGet.Env.SystemInstalledPackageListFile);
			} catch (FileNotFoundException) {
				systemInstalledPkgList = new PackageList<InstalledPackage>();
			}
			
			
			systemInstalledLogList = new List<InstallationLog>();
			try {
				systemInstalledLogList.AddRange(
					NaGet.Utils.GetDeserializedObject<InstallationLog[]>(NaGet.Env.SystemInstalledPackageLogFile)
				);
			} catch (Exception) {
				// do nothing
			}
		}
		
		/// <summary>
		/// ファイルリストから読み込まれたパッケージのリスト
		/// をファイルに書き込む。
		/// </summary>
		public void SaveAvailablePackageList()
		{
			NaGet.Utils.PutSerializeObject(NaGet.Env.PackageListFile, availablePkgList);
		}
		/// <summary>
		/// 本ソフトウェアを介してインストールされたパッケージのリスト
		/// をファイルに書き込む。
		/// </summary>
		public void SaveInstalledPackageList()
		{
			NaGet.Utils.PutSerializeObject(NaGet.Env.ArchiveInstalledPackageListFile, installedPkgList);
		}
		/// <summary>
		/// システムから検出されたパッケージのリスト
		/// をファイルに書き込む。
		/// </summary>
		public void SaveSystemInstalledPackageList()
		{
			NaGet.Utils.PutSerializeObject(NaGet.Env.SystemInstalledPackageListFile, systemInstalledPkgList);
		}
		/// <summary>
		/// インストールログ
		/// をファイルに書き込む。
		/// </summary>
		public void SaveSystemInstalledLogList()
		{
			NaGet.Utils.PutSerializeObject(NaGet.Env.SystemInstalledPackageLogFile, systemInstalledLogList);
		}
		
		/// <summary>
		/// 本オブジェクトの保有しているパッケージ情報の全てをファイルに書き出す。
		/// </summary>
		public void CommitToFile()
		{
			SaveAvailablePackageList();
			SaveInstalledPackageList();
			SaveSystemInstalledPackageList();
			
			SaveSystemInstalledLogList();
		}
		
		#endregion
		
		#region パッケージインストール検知関連
		
		/// <summary>
		/// 本ソフトウェアを介してインストールされたパッケージを検出する。
		/// 具体的には<see cref="NaGet.Env.ArchiveProgramFiles" />
		/// で指定されたフォルダの下を見る。
		/// </summary>
		public void DetectInstalledPkgs()
		{
			PackageList<InstalledPackage> pkgList = new PackageList<InstalledPackage>();
			if (Directory.Exists(NaGet.Env.ArchiveProgramFiles)) {
				foreach (string path in NaGet.Utils.ExtendWildcardFile(NaGet.Env.ArchiveProgramFiles, Path.Combine("*", ".applistation.package.xml"))) {
					pkgList.AddPackage(NaGet.Utils.GetDeserializedObject<InstalledPackage>(path));
				}
				this.installedPkgList = pkgList;
			}
		}
		
		/// <summary>
		/// レジストリから、コンピュータにインストールされたパッケージを検出し、
		/// インストールトリストに登録(更新)をする。このとき、
		/// インストールのログを見て可能な限りパッケージ情報を補完する。
		/// </summary>
		public void DetectSystemInstalledPkgs()
		{
			List<InstalledPackage> installedPkgList = new List<InstalledPackage>();
			installedPkgList.AddRange(RegistriedUninstallers.DetectInstalledPackages(availablePkgList));
			
			// インストールのログを見て可能な限りパッケージ情報を補完する。
			for (int i = 0; i < installedPkgList.Count; i++) {
				InstalledPackage pkg = installedPkgList[i];				
				int count = installedPkgList.FindAll(PackageList<InstalledPackage>.GetPredicateForPackageName(pkg.Name)).Count;

				foreach (InstallationLog log in systemInstalledLogList.FindAll(createPackageNamePredicator(pkg))) {
					// 1. バージョンがアンインストール情報から取得できなかったときは無条件で登録
					// 2. 同一名パッケージがない場合は無条件で登録
					// 3. 複数個同一パッケージがインストールされている場合は、バージョン比較して登録。
					if (string.IsNullOrEmpty(pkg.Version) ||
					    count < 2 ||
					    log.Package.Version == pkg.Version) {
						
						// 補完したパッケージ情報への差し替え
						UninstallInformation pkgUninstInfo = pkg.UninstallInfo;
						NaGet.Utils.FieldCopy((Package) log.Package, ref pkg);
						pkg.UninstallInfo = pkgUninstInfo;
						if (pkg.UninstallInfo.InstallDate == null) { // インストール日時の補完
							pkg.UninstallInfo.InstallDate = log.Date;
						}
						
						break;
					}
				}
			}
			this.systemInstalledPkgList.Packages = installedPkgList.ToArray();
		}
		
		#endregion
		
		#region パッケージ便利メソッド
		
		/// <summary>
		/// 全て(すなわち、本ソフトウェアを介してインストールされたものと、
		/// システムにインストールされたもの)のインストールされたパッケージを返す便利メソッド
		/// </summary>
		/// <returns>全てのインストールされたパッケージのイテレータ</returns>
		public IEnumerable<InstalledPackage> GetAllInstalledPackages()
		{
			return NaGet.Utils.MergeEnumerable<InstalledPackage>(
				installedPkgList.GetEnumerator(),
				systemInstalledPkgList.GetEnumerator()
			);
		}
		
		/// <summary>
		/// 全てのパッケージをイテレータで返す便利メソッド。
		/// </summary>
		/// <returns>パッケージのイテレータ</returns>
		public IEnumerable<Package> GetAllPackages()
		{
			return NaGet.Utils.MergeEnumerable<Package>(
				availablePkgList.GetEnumerator(),
				(IEnumerator<Package>) GetAllInstalledPackages().GetEnumerator()
			);
		}
		
		#endregion
		
		#region インストールログ関連
		
		/// <summary>
		/// 指定パッケージ名に対応するログを判定するPredicate
		/// </summary>
		/// <param name="pkg">判定する指定パッケージ</param>
		/// <returns>Predicate</returns>
		private Predicate<InstallationLog> createPackageNamePredicator(Package pkg)
		{
			return delegate(InstallationLog log) {
				return log.Package.Name == pkg.Name;
			};
		}
		
		/// <summary>
		/// インストールログにインストール情報を追加する。
		/// ファイルへの書き込みは行われない。
		/// </summary>
		/// <remarks>過去の同一パッケージ名のログは消去される</remarks>
		/// <param name="inst">書き込むインストール情報</param>
		public void WriteInstallationLog(Installation inst)
		{
			// 重複は削除
			systemInstalledLogList.RemoveAll(createPackageNamePredicator(inst.InstalledPackage));
			
			InstallationLog newLog = new InstallationLog();
			newLog.Date = DateTime.Now;
			newLog.Package = inst.InstalledPackage;
			
			systemInstalledLogList.Add(newLog);
		}
		
		#endregion
	}
}
