﻿using System;
using System.IO;
using System.Net;

namespace NaGet
{
	/// <summary>
	/// 環境変数などを取り扱うクラス
	/// </summary>
	public sealed class Env
	{		
		/// <summary>
		/// 呼び出し禁止
		/// </summary>
		private Env()
		{
		}
		
		#region pref.xml関連
		
		/// <summary>
		/// 設定を格納
		/// </summary>
		private static NaGetLibPref pref;
		
		/// <summary>
		/// 設定（のクローンオブジェクト）を返す。
		/// </summary>
		public static NaGetLibPref Pref {
			get {
				NaGetLibPref prefCopied = new NaGetLibPref();
				
				if (pref == null) {
					LoadPref();
				}
				NaGet.Utils.FieldCopy(pref, ref prefCopied);
				
				return prefCopied;
			}
		}
		
		/// <summary>
		/// 設定ファイルのある場所を取得する
		/// </summary>
		public static string PrefPath {
			get {
				return Path.Combine(Environment.CurrentDirectory, "pref.xml");
			}
		}
		
		/// <summary>
		/// 設定ファイルを読み込む。または再読み込みする。
		/// </summary>
		public static void LoadPref()
		{
			string path = PrefPath;
			try {
				if (! File.Exists(path)) throw new FileNotFoundException(string.Empty, path);
				pref = NaGet.Utils.GetDeserializedObject<NaGetLibPref>(path);
			} catch(Exception) {
				pref = new NaGetLibPref();
			}
		}
		
		
		#endregion

		#region ファイルパス
		
		/// <summary>		
		/// ファイルリスト提供サーバのリストファイル	
		/// </summary>
		[Obsolete]
		public static readonly string ProviderListFile = "provider.list.txt";
		
		/// <summary>
		/// ファイルリスト提供レポジトリリストのファイル
		/// </summary>
		public static readonly string RepositoriesListFile = "repos.list.xml";

		/// <summary>
		/// ファイルリストから読み込まれたパッケージのリストファイル
		/// </summary>
		public static readonly string PackageListFile = "packages.list.xml";

		/// <summary>
		/// 本ソフトウェアを介してインストールされたパッケージのリストファイル
		/// </summary>
		public static readonly string ArchiveInstalledPackageListFile = "packages.envinstalled.xml";

		/// <summary>
		/// システムから検出されたパッケージのリストファイル
		/// </summary>
		public static readonly string SystemInstalledPackageListFile = "packages.sysinstalled.xml";
		
		/// <summary>
		/// インストールログファイル
		/// </summary>
		public static readonly string SystemInstalledPackageLogFile = "packages.sysinstalled.log.xml";
		
		/// <summary>
		/// アプリケーションデータを保存するフォルダのパス
		/// </summary>
		public static string AppDataFolderPath
		{
			get {
				if (pref == null) LoadPref();
				
				if (! string.IsNullOrEmpty(pref.AppDataFolder)) {
					return pref.AppDataFolder;
				}
				
				return Environment.CurrentDirectory;
			}
		}

		/// <summary>
		/// インストーラの一時置き場の親ディレクトリ
		/// </summary>
		public static string ArchiveFolderPath
		{
			get { return Path.Combine(AppDataFolderPath, "Cache"); }
		}
		
		/// <summary>
		/// アーカイバ方式のパッケージのインストール先フォルダ
		/// </summary>
		public static string ArchiveProgramFiles {
			get {
				return Path.Combine(AppDataFolderPath, "progs");
			}
		}
		
		/// <summary>
		/// アーカイバ方式のパッケージのプログラムグループフォルダ
		/// </summary>
		public static string ArchiveProgramGroup {
			get {
				return Path.Combine(AppDataFolderPath, "programs");
			}
		}

		/// <summary>
		/// アーカイバ方式のパッケージのSystem32のフォルダ
		/// </summary>
		public static string ArchiveSystem32 {
			get {
				return Path.Combine(ArchiveProgramFiles, ".system32");
			}
		}
		
		#endregion
		
		#region ネットワーク
		
		/// <summary>
		/// ウェブアクセスのためのプロキシを取得する。
		/// </summary>
		/// <returns>プロキシ</returns>
		public static IWebProxy WebProxy
		{
			get {
				if (pref == null) LoadPref();
				
				if (string.IsNullOrEmpty(pref.ProxyAddress)) {
					// 設定されていないときはシステムのデフォルトを使う
					return WebRequest.GetSystemWebProxy();
				} else if (pref.ProxyAddress == "-") {
					// "-"のとき直接接続
					return null;
				} else {
					// host:portが設定されているならば、それをもとに設定
					return new WebProxy(pref.ProxyAddress);
				}
			}
		}
		
		#endregion
	}
}
