﻿using System;
using System.Drawing;
using System.Windows.Forms;
using System.Text;
using System.IO;
using NaGet.Packages.Install;
using NaGet.Packages;

namespace AppliStation
{
	/// <summary>
	/// Description of PackageInfoForm.
	/// </summary>
	public partial class PackageInfoForm : Form
	{
		private Package pkg;
		
		public PackageInfoForm()
		{
			InitializeComponent();
		}
		
		/// <summary>
		/// 現在選択中(=表示中)のパッケージ
		/// </summary>
		public Package SelectedObject {
			get { return pkg; }
			set {
				pkg = value;
				updateSelectedObject();
			}
		}
		
		private void updateSelectedObject()
		{
			bool isInstalledPackage = pkg is InstalledPackage;
			
			iconLabel.Image = null;
			iconLabel.Size = Size.Empty;
			if (isInstalledPackage) {
				Icon icon = AppliStation.Util.GUIUtils.GetIconForPackage((InstalledPackage) pkg);
				if (icon != null) {
					iconLabel.Image = icon.ToBitmap();
					iconLabel.Size = icon.Size;
				}
			}
			nameLabel.Text = string.Format("{0} ({1})", pkg.Name, pkg.Version);
			nameLabel.Font = new Font(SystemFonts.CaptionFont.FontFamily, SystemFonts.CaptionFont.Size*1.25f);
			
			summaryTextBox.Text = pkg.Summary;
			
			if (isInstalledPackage) {
				UninstallInformation uninstInfo = ((InstalledPackage) pkg).UninstallInfo;
				
				string instPath = null;
				if ((pkg.Type == InstallerType.ARCHIVE)
				    ||(pkg.Type == InstallerType.ITSELF)) {
					instPath = Path.Combine(NaGet.Env.ArchiveProgramFiles, pkg.Name);
				} else if (Directory.Exists(uninstInfo.InstallLocation)) {
					instPath = uninstInfo.InstallLocation;
				}
				
				statusLinkLabel.Links.Clear();
				if (string.IsNullOrEmpty(instPath)) {
					statusLinkLabel.Text = "情報が提供されていません";
					statusLinkLabel.Enabled = false;
				} else {
					statusLinkLabel.Text = instPath;
					statusLinkLabel.Enabled = true;
					
					addLinkToLinkLabel(statusLinkLabel, uninstInfo.InstallLocation, uninstInfo.InstallLocation);
				}
			}
			urlLinkLabel.Links.Clear();
			if (pkg.Url != null) {
				urlLinkLabel.Text = pkg.Url.Href;
				urlLinkLabel.Links.Add(0, pkg.Url.Href.Length, pkg.Url.Href);
				urlLinkLabel.Enabled = true;
				urlLabel.Enabled = true;
			} else {
				urlLinkLabel.Text = string.Empty;
				urlLinkLabel.Enabled = false;
				urlLabel.Enabled = false;
			}
			tagsTextBox.Text = pkg.Tags;
			setLinkLabelFromLicense(licenseLinkLabel, pkg.License, (pkg.Url!=null)?pkg.Url.Href:string.Empty);
			setTypeValueLabel(typeValueLabel, pkg.Type);
			if (! string.IsNullOrEmpty(pkg.PackageListName)) {
				packageListNameValueLabel.Text = pkg.PackageListName;
				packageListNameValueLabel.Enabled = true;
			} else {
				packageListNameValueLabel.Enabled = false;
			}
			
			tabControl1.TabPages.Clear();
			if (pkg is InstalledPackage && string.IsNullOrEmpty(pkg.Version)) {
				tabControl1.Visible = false;
			} else {
				tabControl1.Visible = true;
				foreach (Installer inst in pkg.Installer) {
					TabPage page = new TabPage(string.Format("インストーラ その{0}", tabControl1.TabCount+1));
					page.Dock = DockStyle.Fill;
					
					InstallerInfoForm instForm = new InstallerInfoForm();
					instForm.SelectedObject = inst;
					instForm.Dock = DockStyle.Fill;
					page.Controls.Add(instForm);
					
					tabControl1.TabPages.Add(page);
				}
			}
			
			this.Size = new Size(this.Size.Width, (tabControl1.TabCount>0)? 380 : 250);
		}
		
		/// <summary>
		/// インストーラの種類についてラベルにテキストを設定する
		/// </summary>
		/// <param name="typeValueLabel">設定先のラベル</param>
		/// <param name="instType">インストーラの種類</param>
		private static void setLinkLabelFromLicense(LinkLabel linklabel, string license, string publicSiteUrl)
		{
			linklabel.Enabled = true;
			switch (license ?? string.Empty) {
				case "":
					if (string.IsNullOrEmpty(publicSiteUrl)) {
						linklabel.Text = "フリーソフトウェア (無償)";
						linklabel.Enabled = false;
					} else {
						linklabel.Text = "フリーソフトウェア (詳細は公式サイトなどで確認してください)";
						addLinkToLinkLabel(linklabel, "公式サイト", publicSiteUrl);
					}
					break;
				#region オープンソースかつコピーレフト
				case "GPL":
					linklabel.Text = "GPL (オープンソース)";
					addLinkToLinkLabel(linklabel, "GPL", @"http://www.opensource.jp/gpl/gpl.ja.html");
					addLinkToLinkLabel(linklabel, "オープンソース", @"http://creativecommons.org/licenses/GPL/2.0/deed.ja");
					break;
				case "GPLv2":
				case "GPLv2+":
					linklabel.Text = "GPLv2 (オープンソース)";
					addLinkToLinkLabel(linklabel, "GPLv2", @"http://www.gnu.org/licenses/gpl-2.0.txt");
					break;
				case "GPLv3":
				case "GPLv3+":
					linklabel.Text = "GPLv3 (オープンソース)";
					addLinkToLinkLabel(linklabel, "GPLv3", @"http://www.gnu.org/licenses/gpl-3.0.txt");
					break;
				case "LGPL":
					linklabel.Text = "LGPL (オープンソース)";
					addLinkToLinkLabel(linklabel, "LGPL", @"http://www.opensource.jp/lesser/lgpl.ja.html");
					addLinkToLinkLabel(linklabel, "オープンソース", @"http://creativecommons.org/licenses/LGPL/2.1/deed.ja");
					break;
				case "LGPLv2":
				case "LGPLv2+":
					linklabel.Text = "LGPLv2 (オープンソース)";
					addLinkToLinkLabel(linklabel, "LGPLv2", @"http://www.gnu.org/licenses/lgpl-2.1.txt");
					break;
				case "LGPLv3":
				case "LGPLv3+":
					linklabel.Text = "LGPLv3 (オープンソース)";
					addLinkToLinkLabel(linklabel, "LGPLv3", @"http://www.gnu.org/licenses/lgpl-3.0.txt");
					break;
				case "AGPLv3":
				case "AGPLv3+":
					linklabel.Text = "AGPLv3 (オープンソース)";
					addLinkToLinkLabel(linklabel, "AGPLv3", @"http://www.gnu.org/licenses/agpl-3.0.txt");
					break;
				case "MPL":
					linklabel.Text = "MPL (オープンソース)";
					addLinkToLinkLabel(linklabel, "MPL", @"http://www.mozilla.org/MPL/MPL-1.1.html");
					break;
				case "CPL":
					linklabel.Text = "CPL (オープンソース)";
					linklabel.Links.Add(0, linklabel.Text.Length, @"http://sourceforge.jp/projects/opensource/wiki/licenses%2FCommon_Public_License");
					break;
				case "Artistic":
					linklabel.Text = "Artisticライセンス (オープンソース)";
					addLinkToLinkLabel(linklabel, "Artisticライセンス", @"http://www.perlfoundation.org/artistic_license_2_0");
					break;
				#endregion
				#region オープンソースだが、コピーレフトでない著名なライセンス
				case "MIT":
					linklabel.Text = "MITライセンス (オープンソース)";
					addLinkToLinkLabel(linklabel, "MITライセンス", @"http://www.opensource.org/licenses/mit-license.php");
					break;
				case "BSD":
					linklabel.Text = "修正済BSDライセンス (オープンソース)";
					addLinkToLinkLabel(linklabel, "BSDライセンス", @"http://www.opensource.org/licenses/bsd-license.php");
					break;
				case "Apache":
				case "ASLv2":
				case "ASLv2+":
					linklabel.Text = "Apacheライセンス (オープンソース)";
					addLinkToLinkLabel(linklabel, "Apacheライセンス", @"http://www.apache.org/licenses/LICENSE-2.0.txt");
					break;
				case "zlib":
					linklabel.Text = "zlibライセンス";
					linklabel.Links.Add(0, linklabel.Text.Length, @"http://opentechpress.jp/docs/licenses/zlib-license.shtml");
					break;
				#endregion
				case "WTFPL":
				case "WTFPLv2":
					linklabel.Text = "WTFPL";
					linklabel.Links.Add(0, linklabel.Text.Length, @"http://sam.zoy.org/wtfpl/");
					break;
				case "NYSL":
					linklabel.Text = "煮るなり焼くなり好きにしろライセンス";
					linklabel.Links.Add(0, linklabel.Text.Length, @"http://www.kmonos.net/nysl/NYSL_withfaq.TXT");
					break;
				case "Public Domain":
					linklabel.Text = "パブリックドメイン (著作権帰属なし)";
					linklabel.Links.Add(0, linklabel.Text.Length, @"http://e-words.jp/w/E38391E38396E383AAE38383E382AFE38389E383A1E382A4E383B3.html");
					break;
				default:
					if (license.StartsWith("http://")) {
						linklabel.Text = license;
						linklabel.Links.Add(0, license.Length, license);
					} else {
						linklabel.Text = license;
						linklabel.Links.Add(0, 0, string.Empty);
					}
					break;
			}
			
		}
		
		/// <summary>
		/// インストーラの種類についてラベルにテキストを設定する
		/// </summary>
		/// <param name="typeValueLabel">設定先のラベル</param>
		/// <param name="instType">インストーラの種類</param>
		private static void setTypeValueLabel(Label typeValueLabel, InstallerType instType)
		{
			switch (instType) {
				case InstallerType.ARCHIVE:
					typeValueLabel.Text = "自己解凍書庫または書庫";
					break;
				case InstallerType.EXEC_INSTALLER:
					typeValueLabel.Text = "インストーラ形式";
					break;
				case InstallerType.MSI_PACKAGE:
					typeValueLabel.Text = "MSIファイル形式";
					break;
				case InstallerType.ITSELF:
					typeValueLabel.Text = "単体の実行ファイル";
					break;
				default:
					typeValueLabel.Text = "不明なインストール形式(インストール不能)";
					break;
			}
		}
		
		/// <summary>
		/// リンクラベルのリンク挿入便利メソッド
		/// </summary>
		/// <param name="linkLabel">対象リンクラベル</param>
		/// <param name="word">リンク対象文字列</param>
		/// <param name="link">リンクオブジェクト</param>
		private static void addLinkToLinkLabel(LinkLabel linkLabel, string word, string link)
		{
			int pos = linkLabel.Text.IndexOf(word, 0);
			if (pos >= 0) {
				linkLabel.Links.Add(pos, word.Length, link);
			} else throw new ArgumentException(string.Format("label text \"{0}\" does not contain \"{1}\"!", linkLabel.Text, word));
		}
		
		private void linkLabelLinkClicked(object sender, LinkLabelLinkClickedEventArgs e)
		{
			string linkData = e.Link.LinkData.ToString();
			
			if (Directory.Exists(linkData)) {
				e.Link.Visited = true;
				System.Diagnostics.Process.Start(e.Link.LinkData.ToString());
			} else if (System.Text.RegularExpressions.Regex.IsMatch(linkData, @"^(https?|file)://")) {
				e.Link.Visited = true;
				try {
					System.Diagnostics.Process.Start(e.Link.LinkData.ToString());
				} catch (System.ComponentModel.Win32Exception) {
					MessageBox.Show(string.Format("{0}を開くのに失敗しました。", e.Link.LinkData), "ブラウザ起動エラー", MessageBoxButtons.OK, MessageBoxIcon.Error);
				}
			}
		}
	}
}
