﻿using System;
using System.IO;
using System.Diagnostics;
using NaGet.Net;
using NaGet.SubCommands;
using System.Xml.Serialization;

namespace NaGet.Packages.Install
{

    /// <summary>
    /// ダウンロード・インストール処理をカプセル化するクラス
    /// </summary>
    public class Installation
	{
		private Package installedPackage;
		
		/// <summary>
		/// インストールするパッケージ
		/// </summary>
		public Package InstalledPackage {
			get { return installedPackage; }
			set {
				installedPackage = value;
				
				installerFile = getArchiveFilePath();
				installerIndex = GetPreferInstallerIndex(value);
			}
		}

		/// <summary>
		/// (保存される)インストーラのファイルのパス
		/// </summary>
		private string installerFile;

		/// <summary>
		/// インストールが完了されたか否かのフラグ
		/// </summary>
		private bool installed = false;
		
		/// <summary>
		/// サイレントインストールを行うか否か
		/// </summary>
		private bool silent = false;
	  
		/// <summary>
		/// 起動するインストーラのインデックス番号
		/// </summary>
		protected int installerIndex = 0;
		
		/// <summary>
		/// 外部アプリのエラー出力の受信ハンドラ
		/// </summary>
		public event EventHandler<NaGet.Utils.AnyDataEventArgs<string>> ErrorDataReceived;
		
		/// <summary>
		/// 外部アプリの標準出力の受信ハンドラ
		/// </summary>
		public event EventHandler<NaGet.Utils.AnyDataEventArgs<string>> OutputDataReceived;
		
		/// <summary>
		/// コンストラクタ
		/// </summary>
		public Installation()
		{
		}
		
		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="package">インストールするパッケージ</param>
		public Installation(Package package)
		{
			InstalledPackage = package;
		}
		
		/// <summary>
		/// インストーラファイルを取得する
		/// </summary>
		public string InstallerFile {
			get { return installerFile; }
		}
		
		
		#region インストーラ状態チェック関連
		
		/// <summary>
		/// インストール可能か否か
		/// </summary>
		public bool IsInstallablePackage()
		{
			return installerIndex >= 0;
		}
		
		/// <summary>
		/// ダウンロード済みであるか否か
		/// </summary>
		public bool Downloaded {
			get {
				return File.Exists(installerFile) && ((File.GetAttributes(installerFile) & FileAttributes.Hidden) != FileAttributes.Hidden);
			}
		}
		
		/// <summary>
		/// インストール作業済みであるか否か。実際にインストールされたかどうかではありません。
		/// </summary>
		public bool Installed {
			get { return installed; }
		}
		
		/// <summary>
		/// インストーラの処理が成功してインストールされたプログラムが確認できるか否か。
		/// </summary>
		public bool InstallSuccessed {
			get {
				switch (InstalledPackage.Type) {
					case InstallerType.ARCHIVE:
					case InstallerType.ITSELF:
						// アーカイブインストーラおよび配布exeそれ自身が実行ファイルのとき、
						// （AppliStationの作る）プログラムフォルダの存在のみで確認を行う。
						return Directory.Exists(Path.Combine(NaGet.Env.ArchiveProgramFiles, InstalledPackage.Name));
					case InstallerType.EXEC_INSTALLER:
					case InstallerType.MSI_PACKAGE:
						return RegistriedUninstallers.GetInstalledPackageFor(InstalledPackage) != null;
					default:
						return false;
				}
			}
		}
		
		/// <summary>
		/// サイレントインストールを行うかのフラグ。
		/// </summary>
		public bool Silent {
			get {
				return (SupportsSilentOnly)? true :
					(IsSupportsSilent)? silent :
					false;
			}
			set { silent = value; }
		}
		
		/// <summary>
		/// サイレントインストールをサポートしているか否か
		/// </summary>
		public bool IsSupportsSilent {
			get {
				switch (InstalledPackage.Type) {
					case InstallerType.ARCHIVE:
					case InstallerType.ITSELF:
					case InstallerType.MSI_PACKAGE:
						return true;
					case InstallerType.EXEC_INSTALLER:
						return ! string.IsNullOrEmpty(InstalledPackage.SilentInstallArguments);
					default:
						return false;
				}
			}
		}
		
		/// <summary>
		/// サイレントインストールだけをサポートしているか否か
		/// </summary>
		public bool SupportsSilentOnly {
			get {
				return (InstalledPackage.Type == InstallerType.ARCHIVE)
					|| (InstalledPackage.Type == InstallerType.ITSELF);
			}
		}
		
		/// <summary>
		/// 選択されたパッケージは、AppliStationではなくPCへのインストールをするのか否かを返す。
		/// </summary>
		/// <remark>RunAsが必要か否かの判断にしようされる</remark>
		public bool TargetPC {
			get {
				return (InstalledPackage.Type != InstallerType.ARCHIVE)
					&& (InstalledPackage.Type != InstallerType.ITSELF);
			}
		}
		
		#endregion
		
		/// <summary>
		/// ダウンロードを行う。
		/// </summary>
		/// <param name="downloader">ダウンローダオブジェクト</param>
		public void Download(Downloader downloader)
		{
			if (! Installed) {
				string url = InstalledPackage.Installer[installerIndex].Url.Href;
				downloader.Download(url, installerFile);
				
				// サーバ指定のファイル名に変更する
				if (! string.IsNullOrEmpty(downloader.DownloadedFileName)) {
					string newFile = Path.Combine(Path.GetDirectoryName(installerFile), downloader.DownloadedFileName);
					File.Move(installerFile, newFile);
					installerFile = newFile;
				}
				
				// 権限を親フォルダに落とす
				try {
					string targetDir = Path.GetDirectoryName(installerFile);
					NaGet.Utils.SetAccessControlRecursive(targetDir, File.GetAccessControl(Path.GetDirectoryName(targetDir)));
				} catch (Exception) {} // 失敗時は何もしない
			}
		}
		
		/// <summary>
		/// インストーラファイルをスキャンする
		/// </summary>
		/// <remarks>ウイルスのため退避・削除されたときも例外を投げずにあたかも正常かのように動作しえます。</remarks>
		/// <exception cref="ComException">スキャンで意図せぬ動作があったとき</exception>
		/// <exception cref="FileNotFoundException">スキャン後にインストーラファイルが存在しないとき</exception>
		/// <param name="scanner">スキャナーオブジェクト</param>
		public void ScanInstallerFile(DownloadScanner scanner)
		{
			Exception e = null;
			try {
				scanner.Scan(installerFile, InstalledPackage.Installer[installerIndex].Url.Href);
			} catch (System.Runtime.InteropServices.COMException ex) {
				e = ex;
			}
			
			if (! File.Exists(installerFile)) {
				// ファイルが消されているならばFileNotFoundExceptionを出す
				throw new FileNotFoundException(string.Empty, installerFile, e);
			} else if ( e != null ) {
				// ファイルが消されていないが例外が発生していたときは、その例外を投げる
				throw e;
			}
		}
		
		/// <summary>
		/// ハッシュ検証のためのハッシュの種類の数を返す
		/// </summary>
		/// <returns>ハッシュの個数</returns>
		public int GetRegisteredHashCount()
		{
			HashValue[] hashValues = InstalledPackage.Installer[installerIndex].Hash;
			return (hashValues == null)? 0 : hashValues.Length;
		}
		
		/// <summary>
		/// ハッシュ検証を行う
		/// </summary>
		/// <returns>検証にパスしたらtrue、パスしなかったらfalse</returns>
		public bool VerifyHashValues()
		{
			HashValue[] hashValues = InstalledPackage.Installer[installerIndex].Hash;
			if (hashValues != null) {
				foreach (HashValue hash in hashValues) {
					if (! hash.Validate(installerFile)) {
						return false;
					}
				}
			}
			
			return true;
		}
		
		private int invokeInstaller(string installerfile, InstallerType type)
		{
			if (! File.Exists(installerfile)) {
				throw new FileNotFoundException(string.Format("{0} is not found, perhaps failed to download.", installerfile), installerfile);
			}
			
			Process hProcess = null;
			try {
				switch (type) {
				case InstallerType.EXEC_INSTALLER:
					if (Silent) {
						hProcess = Process.Start(installerfile, InstalledPackage.SilentInstallArguments);
					} else {
						hProcess = Process.Start(installerfile);
					}
					
					break;
				case InstallerType.MSI_PACKAGE:
					string msiexecArgs = string.Format("{0} /norestart /i \"{1}\" REBOOT=ReallySuppress",
					                            (Silent)? "/passive" : string.Empty,
					                            installerfile);
					
					hProcess = Process.Start("msiexec", msiexecArgs);
					break;
				case InstallerType.ARCHIVE:
				case InstallerType.ITSELF:
					string argument = string.Format("-i \"{0}\" \"{1}\"", installerfile, this.InstalledPackage.Name);
					hProcess = createExtractArchiveProcess(argument,
					                                       this.OutputDataReceived,
					                                       this.ErrorDataReceived);
					// Note: ARCHIVEかITSELFの判断はarchive-instが行う
					break;
				default:
					throw new NotImplementedException("Not implemented archive installation yet");
				}
				
				if (NaGet.Env.InstallProcessOnBackground) {
					try {
						hProcess.PriorityClass = ProcessPriorityClass.Idle;
					} catch (Exception) {}
				}
				
				hProcess.WaitForExit();
				
				return hProcess.ExitCode;
			} finally {
				if (hProcess != null) {
					hProcess.Close();
				}
			}
		}
		
		/// <summary>
		/// インストーラ等を起動してインストール作業を行う
		/// </summary>
		/// <returns>インストーラの終了コード</returns>
		public int Install()
		{
			string installerFile = this.installerFile;
			string tempDir = Path.Combine(Path.GetTempPath(), Path.GetRandomFileName());
			
			// アーカイブされているなら一旦展開
			if (InstalledPackage.ArchivedInstaller) {
				Directory.CreateDirectory(tempDir);
				
				string argument = string.Format("-t \"{0}\" \"{1}\"", installerFile, tempDir);
				using (Process hProcess = createExtractArchiveProcess(argument,
				                                                      this.OutputDataReceived,
				                                                      this.ErrorDataReceived)) {
					if (NaGet.Env.InstallProcessOnBackground) {
						try {
							hProcess.PriorityClass = ProcessPriorityClass.Idle;
						} catch (Exception) {}
					}
					
					hProcess.WaitForExit();
				
					if (hProcess.ExitCode != 0) {
						throw new ApplicationException("Extract error " + installerFile + " to " + tempDir);
					}
				}
				
				installerFile = seekInstallerFile(tempDir, InstalledPackage.Type);
				if (installerFile == null && Directory.GetDirectories(tempDir).Length == 1) {
					installerFile = seekInstallerFile(Path.Combine(tempDir, Directory.GetDirectories(tempDir)[0]), InstalledPackage.Type);
				}
			}
			
			int exitCode = invokeInstaller(installerFile, InstalledPackage.Type);
			
			installed = true;
			
			return exitCode;
		}
		
		/// <summary>
		/// もっともふさわしいインストーラ番号を返す
		/// </summary>
		/// <returns></returns>
		public static int GetPreferInstallerIndex(Package InstalledPackage)
		{
			if (InstalledPackage.Type == InstallerType.CANNOT_INSTALL) { // インストール不能パッケージ
				return -1;
			}
			
			int best = -1;
			int bestVal = 0;
			
			for (int i = 0; i < InstalledPackage.Installer.Length; i++) {
				Platform platform = InstalledPackage.Installer[i].Platform;
				
				int pts = 0;
				if (platform != null) {
					pts = (platform.IsRunnable())? 10 : 0;
					pts += (platform.IsRunnableArchOnWow64() && platform.IsRunnableOS())? 1 : 0;
				} else { // if (platform == null) {
					pts = 1; // null の場合は動作プラットホーム扱い
				}
				if (pts > bestVal) {
					bestVal = pts;
					best = i;
				}
			}
			
			return best;
		}
		
		/// <summary>
		/// インストーラの保存先パスを生成
		/// </summary>
		private string getArchiveFilePath()
		{
			Package package = this.InstalledPackage;
			
			string folderName = string.Format("{0}({1})", package.Name, package.Version);
			string fileName = NaGet.Utils.Url2filename(package.Installer[0].Url.Href);
			
			string folder = Path.Combine(NaGet.Env.ArchiveFolderPath, folderName);
			
			if (! File.Exists(Path.Combine(folder, fileName))) {
				if (Directory.Exists(folder)) {
					if (! File.Exists(Path.Combine(folder, fileName))) {
						fileName = seekInstallerFile(folder, package.Type) ?? fileName;
					}
				} else {
					Directory.CreateDirectory(folder);
					
					// 権限を親フォルダに合わせる
					try {
						Directory.SetAccessControl(folder, Directory.GetAccessControl(Path.GetDirectoryName(folder)));
					} catch (Exception) {} // 失敗時無視
				}
			}
			
			return Path.Combine(folder, fileName);
		}
		
		/// <summary>
		/// すでにインストールされているパッケージを取得する
		/// </summary>
		/// <param name="installedPkgs">
		/// インストールドリスト
		/// </param>
		/// <returns>
		/// インストールされているパッケージの情報。インストールされたパッケージが見つからないならばnullを返す  
		/// </returns>
		public InstalledPackage GetInstalledPackage(PackageList<InstalledPackage> installedPkgs)
		{
			return installedPkgs.GetPackageForName(InstalledPackage.Name);
		}
		
		/// <summary>
		/// インストーラファイルを探す
		/// </summary>
		/// <param name="basedir">探すフォルダ</param>
		/// <param name="type">インストーラの種類</param>
		/// <returns>探し出されたインストーラファイルのフルパス。存在しないならばnull</returns>
		private static string seekInstallerFile(string basedir, InstallerType type)
		{
			if (! Directory.Exists(basedir)) {
				return null;
			}
			
			System.Collections.Generic.List<string> list = new System.Collections.Generic.List<string>();
			switch (type) {
				case InstallerType.MSI_PACKAGE:
					list.AddRange(Directory.GetFiles(basedir, "*.msi"));
					break;
				case InstallerType.EXEC_INSTALLER:
					list.AddRange(Directory.GetFiles(basedir, "*.exe"));
					break;
				case InstallerType.ARCHIVE:
					list.AddRange(Directory.GetFiles(basedir, "*.zip"));
					list.AddRange(Directory.GetFiles(basedir, "*.lzh"));
					list.AddRange(Directory.GetFiles(basedir, "*.cab"));
					list.AddRange(Directory.GetFiles(basedir, "*.7z"));
					list.AddRange(Directory.GetFiles(basedir, "*.tar*"));
					break;
				case InstallerType.ITSELF:
					list.AddRange(Directory.GetFiles(basedir, "*.exe"));
					break;
				default:
					return null;
			}
			
			// 存在しないファイルを削除
			list.RemoveAll(
				delegate(string file) {
					return ! File.Exists(file);
				}
			);
			
			// "setup"の語が入ったファイルはインストーラとみなし、優先選択
			foreach (string path in list) {
				if (Path.GetFileName(path).ToLower().IndexOf("setup") >= 0) {
					return path;
				}
			}
			
			// それ以外なら一つ目を返す
			return (list.Count > 0)? list[0] : null;
		}
		
		
		/// <summary>
		/// アーカイブファイルの展開・インストールを行う
		/// </summary>
		/// <param name="archiveInstArgs">"archive-inst.exe"への引数</param>
		/// <param name="outputReceived">標準出力用リスナ(null可)</param>
		/// <param name="errorReceived">エラー出力用リスナ(null可)</param>
		/// <returns>実行プロセス</returns>
		private Process createExtractArchiveProcess(string archiveInstArgs,
		                                  EventHandler<NaGet.Utils.AnyDataEventArgs<string>> outputReceived,
		                                  EventHandler<NaGet.Utils.AnyDataEventArgs<string>> errorReceived)
		{
			string archiveInstExe = Path.GetFullPath("archive-inst.exe");
			if (! File.Exists(archiveInstExe)) {
				string errMsg = string.Format("\"{0}\" does not found!");
				throw new ApplicationException(errMsg,
				                               new FileNotFoundException(errMsg, archiveInstExe));
			}
			
			
			
			ProcessStartInfo procInfo = new ProcessStartInfo(archiveInstExe, archiveInstArgs);
			procInfo.UseShellExecute = false;
			procInfo.CreateNoWindow = true;
			procInfo.WorkingDirectory = Environment.CurrentDirectory;
			
			return NaGet.Utils.ProcessStartWithOutputCapture(procInfo, outputReceived, errorReceived);
		}
		
		/// <summary>
		/// ダウンロードしたインストーラファイルを削除する
		/// </summary>
		public virtual void RemoveDownloadedFile()
		{
			if (Downloaded && File.Exists(installerFile)) {
				File.Delete(installerFile);
			}
		}
		
		public override string ToString()
		{
			return string.Format("{0}({1})", InstalledPackage.Name, InstalledPackage.Version);
		}
		
		public static string ToString(Installation[] installations)
		{
			string[] strs = new string[installations.Length];
			for (int i = 0; i < installations.Length; i++) {
				strs[i] = installations[i].ToString();
			}
			return string.Join(" ", strs);
		}
		
		/// <summary>
		/// パッケージ配列をインストール処理配列に変換する便利メソッド
		/// </summary>
		/// <param name="pkgs">パッケージ配列</param>
		/// <returns>変換されたインストール処理配列</returns>
		public static Installation[] ConvertInstallations(Package[] pkgs)
		{
			Installation[] insts = new Installation[pkgs.Length];
			for (int i = 0; i < pkgs.Length; i++) {
				insts[i] = new Installation(pkgs[i]);
			}
			return insts;
		}
	}
}
