﻿using System;
using System.ComponentModel;
using System.Xml.Serialization;
using System.Collections.Generic;

namespace NaGet.Packages
{
	/// <summary>
	/// レポジトリのリストを表現するクラス
	/// </summary>
	public class RepositoriesList
	{
		/// <summary>
		/// レポジトリの配列
		/// </summary>
		public RepositoryInfo[] Repositories = null;
		
		/// <summary>
		/// デフォルトコンストラクタ
		/// </summary>
		public RepositoriesList()
		{
		}
		
		/// <summary>
		/// 有効なレポジトリの配列を返す
		/// </summary>
		[XmlIgnore]
		public RepositoryInfo[] EnabledRepositories {
			get {
				return Array.FindAll(Repositories, delegate(RepositoryInfo repo) {
				                     	return repo.Enabled;
				                     });
			}
		}
		
		/// <summary>
		/// provider.list.txtから変換してRepositoriesListとして返す
		/// </summary>
		/// <param name="path">変換もとのprovider.list.txtのパス</param>
		/// <returns>変換されたRepositoriesList</returns>
		public static RepositoriesList MigrateFromProviderListTxt(string path)
		{
			List<RepositoryInfo> repos = new List<RepositoryInfo>();
			using(System.IO.StreamReader reader = new System.IO.StreamReader(path)) {
				string line;
				while ((line = reader.ReadLine()) != null) {
					RepositoryInfo repo = new RepositoryInfo();
					repo.Enabled = true;
					repo.Url = new LocationEntry(line);
					repo.Name = string.Empty;
					repos.Add(repo);
				}
			}
			
			RepositoriesList repolist = new RepositoriesList();
			repolist.Repositories = repos.ToArray();
			return repolist;
		}
	}
	
	/// <summary>
	/// レポジトリで提供されるソフトリストのフォーマットを返す
	/// </summary>
	public enum RepositoryType : uint
	{
		/// <summary>
		/// AppliStation Native XML Format Version 1.0
		/// </summary>
		[XmlEnum(Name="ASNATIVEv1.0")]
		APPLISTATION_NATIVE_XML_1_0 = 0,
	}
	
	/// <summary>
	/// パッケージリストを提供するレポジトリの参照情報を格納するクラス
	/// </summary>
	public class RepositoryInfo
	{
		/// <summary>
		/// レポジトリの提供するソフトリストの名称
		/// </summary>
		[XmlAttribute]
		[ReadOnly(true)]
		public string Name;
		
		/// <summary>
		/// ソフトリストの形式
		/// </summary>
		[XmlAttribute]
		public RepositoryType Type;
		
		/// <summary>
		/// ソフトリストの取得先URL
		/// </summary>
		public LocationEntry Url;
		
		/// <summary>
		/// ソフトリストを読み取るか否か（無視するか否か）のフラグ。
		/// </summary>
		[XmlAttribute]
		[DefaultValue(true)]
		public bool Enabled = true;
	}
}
