﻿using System.Xml.Serialization;

namespace NaGet.Packages
{
	/// <summary>
	/// インストーラーの種類を表す。
	/// </summary>
	public enum InstallerType
	{
		/// <summary>
		/// インストーラー
		/// </summary>
		[XmlEnum(Name="installer")]
		EXEC_INSTALLER,
		/// <summary>
		/// Microsoft Software Installer
		/// </summary>
		[XmlEnum(Name="msi")]
		MSI_PACKAGE,
		/// <summary>
		/// 自己解凍書庫を含む書庫一般
		/// </summary>
		[XmlEnum(Name="archive")]
		ARCHIVE,
		/// <summary>
		/// それ自身がプログラムそのものである
		/// </summary>
		[XmlEnum(Name="itself")]
		ITSELF,
		/// <summary>
		/// インストールできない単なるデータ(BIOSなど)
		/// </summary>
		[XmlEnum(Name="cannotinstall")]
		CANNOT_INSTALL,
	}
	
	/// <summary>
	/// パッケージ情報を格納するクラス
	/// </summary>
	public class Package
	{
		/// <summary>
		/// パッケージ名
		/// </summary>
		public string Name;

		/// <summary>
		/// バージョン文字列
		/// </summary>
		public string Version;

		/// <summary>
		/// パッケージの概要
		/// </summary>
		public string Summary;
		
		/// <summary>
		/// パッケージの解説
		/// </summary>
		public string Description;

		/// <summary>
		/// 公式サイトのURL
		/// </summary>
		public LocationEntry Url;
		
		/// <summary>
		/// 製作者の名前
		/// </summary>
		public string Author;
		
		/// <summary>
		/// インストーラーのタイプ
		/// </summary>
		public InstallerType Type;
		
		/// <summary>
		/// インストーラーが書庫内に入っているかのフラグ
		/// </summary>
		public bool ArchivedInstaller = false;
		
		/// <summary>
		/// アンインストーラーのレジストリのキー
		/// </summary>
		public string UninstallerKey;
		
		/// <summary>
		/// サイレントインストールのための引数。
		/// インストーラー形式のみ有効で、nullもしくは空文字列の場合は
		/// サイレントインストールができないことをあらわす
		/// </summary>
		public string SilentInstallArguments;
		
		/// <summary>
		/// インストールスクリプト(MSBuildドキュメント)
		/// </summary>
		public string InstallScript;
		
		/// <summary>
		/// アーカイブインストーラーのときのSystem32のコピー
		/// </summary>
		public string System32CopyFiles;
		
		/// <summary>
		/// インストーラーのリソースの配列
		/// </summary>
		[XmlElement]
		public Installer[] Installer;

		/// <summary>
		/// 必要とされるパッケージの配列
		/// </summary>
		[XmlArray(IsNullable=true),XmlArrayItem("Entry")]
		public Entry[] Requires;
		
		/// <summary>
		/// 競合するパッケージの配列
		/// </summary>
		[XmlArray(IsNullable=true),XmlArrayItem("Entry")]
		public Entry[] Conflicts;
		
		/// <summary>
		/// タグ
		/// </summary>
		public string Tags;
		
		/// <summary>
		/// ライセンス
		/// </summary>
		public string License;
		
		/// <summary>
		/// 所属するパッケージリスト名称
		/// </summary>
		public string PackageListName;
	}

	/// <summary>
	/// パッケージ参照を示すクラス
	/// </summary>
	public struct Entry
	{
		/// <summary>
		/// パッケージ名
		/// </summary>
		[XmlAttribute]
		public string Name;
		/// <summary>
		/// パッケージのバージョン比較の式
		/// </summary>
		[XmlAttribute]
		public string Flags;
		/// <summary>
		/// パッケージのバージョン
		/// </summary>
		[XmlAttribute]
		public string Version;
		
		public Entry(string name, string flags, string version)
		{
			Name = name;
			Flags = flags;
			Version = version;
		}
		public bool Match(Package pkg)
		{
			System.Text.RegularExpressions.Regex regex = new System.Text.RegularExpressions.Regex(
				(System.Text.RegularExpressions.Regex.IsMatch(Name, "^/.*/$"))?
				Name.Substring(1, Name.Length-2) :
				string.Format("^{0}$", System.Text.RegularExpressions.Regex.Escape(Name))
			, System.Text.RegularExpressions.RegexOptions.Singleline);
			
			if (regex.IsMatch(pkg.Name)) {
				if (string.IsNullOrEmpty(Version)) {
					return true;
				} else {
					// TODO Flagの内容は現在のところ無視
					return pkg.Version == Version;
				}
			} else {
				return false;
			}
		}
	}

	public class Installer
	{
		/// <summary>
		/// コンストラクタ
		/// </summary>
		public Installer()
		{
		}
		
		public Platform Platform;
		
		/// <summary>
		/// ダウンロードリソースURLの配列
		/// </summary>
		public LocationEntry Url;
		
		/// <summary>
		/// ファイル検証用ハッシュ
		/// </summary>
		[XmlElement]
		public HashValue[] Hash;
	}
	
	/// <summary>
	/// リソースの位置を示すクラス
	/// </summary>
	public sealed class LocationEntry
	{
		/// <summary>
		/// コンストラクタ
		/// </summary>
		public LocationEntry()
		{
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="sHref">リソースのURL</param>
		public LocationEntry(string sHref)
		{
			Href = sHref;
		}

		/// <summary>
		/// リソースのURL
		/// </summary>
		[XmlAttribute]
		public string Href;
	}
}
