package jp.sourceforge.armadillo;

import java.io.*;
import java.text.*;
import java.util.*;
import java.util.zip.*;

/**
 * A[JCut@CWJB
 */
public final class FileExpander {

    private Expander expander;
    private PrintWriter out;
    private boolean verbose;

    /**
     * FileExpander̐B
     * @param src WJt@C
     * @throws IOException o̓G[ꍇ
     */
    public FileExpander(File src) throws IOException {
        this(src, ArchiveType.getType(src));
    }

    /**
     * FileExpander̐B
     * @param src WJt@C
     * @param type Ƀ^Cv
     * @throws IOException o̓G[ꍇ
     */
    public FileExpander(File src, ArchiveType type) throws IOException {
        this.expander = getExpander(src, type);
        this.out = Utilities.getDefaultPrintWriter();
    }

    /**
     * out(o̓Xg[)̐ݒB
     * @param out o̓Xg[
     */
    public void setOut(PrintWriter out) {
        if (out == null) {
            throw new IllegalArgumentException("out is null");
        }
        this.out = out;
    }

    /**
     * verbose̎擾B
     * @return verbose
     */
    public boolean isVerbose() {
        return verbose;
    }

    /**
     * verbose̐ݒB
     * @param verbose verbose
     */
    public void setVerbose(boolean verbose) {
        this.verbose = verbose;
    }

    /**
     * ɓ̃t@CXgo͂B
     * @throws IOException o̓G[ꍇ
     */
    public void list() throws IOException {
        list(new String[0]);
    }

    /**
     * ɓ̃t@CXgo͂B
     * @param entryNames Gg̔z
     * @throws IOException o̓G[ꍇ
     */
    public void list(String[] entryNames) throws IOException {
        boolean all;
        Set nameSet;
        if (entryNames != null && entryNames.length > 0) {
            all = false;
            nameSet = new HashSet(Arrays.asList(entryNames));
        } else {
            all = true;
            nameSet = Collections.EMPTY_SET;
        }
        DateFormat df = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
        NumberFormat nf = new DecimalFormat("###.0");
        while (true) {
            ArchiveEntry entry = expander.nextEntry();
            if (entry.isNull()) {
                break;
            }
            String name = entry.getName();
            if (!all && !nameSet.contains(name)) {
                continue;
            }
            String line = "";
            if (entry.isDirectory()) {
                line += 'D';
                line += padSpace("", (verbose ? 24 : 8), false);
            } else {
                long size = entry.getSize();
                line += padSpace(String.valueOf(size), 8, true);
                if (verbose) {
                    line += ' ';
                    long compressedSize = entry.getCompressedSize();
                    String compSizeString = (compressedSize > 0)
                            ? String.valueOf(compressedSize)
                            : "";
                    line += padSpace(compSizeString, 8, true);
                    line += ' ';
                    if (size > 0 && compressedSize > 0) {
                        float rate = (compressedSize / 1f / size) * 100;
                        line += padSpace(nf.format(rate), 5, true);
                        line += '%';
                    } else {
                        line += "------";
                    }
                }
                line += ' ';
            }
            line += padSpace(entry.getMethod(), 6, false).substring(0, 6);
            line += ' ';
            line += df.format(new Date(entry.getLastModified()));
            line += ' ';
            line += name;
            out.println(line);
            out.flush();
        }
    }

    /**
     * t@CWJB
     * @param dstdir WJfBNg
     * @throws IOException o̓G[ꍇ
     */
    public void expand(File dstdir) throws IOException {
        expand(dstdir, new String[0]);
    }

    /**
     * t@CWJB
     * @param dstdir WJfBNg
     * @param entryNames Gg̔z
     * @throws IOException o̓G[ꍇ
     */
    public void expand(File dstdir, String[] entryNames) throws IOException {
        if (!dstdir.isDirectory()) {
            throw new IllegalArgumentException("not a directory: " + dstdir);
        }
        boolean all;
        Set nameSet;
        if (entryNames != null && entryNames.length > 0) {
            all = false;
            nameSet = new HashSet(Arrays.asList(entryNames));
        } else {
            all = true;
            nameSet = Collections.EMPTY_SET;
        }
        try {
            if (verbose) {
                out.println("begin to extract");
            }
            while (true) {
                ArchiveEntry entry = expander.nextEntry();
                if (entry.isNull()) {
                    break;
                }
                if (entry.isDirectory()) {
                    continue;
                }
                String name = entry.getName();
                if (!all && !nameSet.contains(name)) {
                    continue;
                }
                if (verbose) {
                    out.println("  extracting ... " + name);
                }
                File newfile = new File(dstdir, name);
                if (newfile.isDirectory()) {
                    continue;
                }
                File tmp = Utilities.createTemporaryFile();
                OutputStream fos = new FileOutputStream(newfile);
                try {
                    expander.expand(fos);
                } finally {
                    fos.close();
                }
                //                IOUtilities.move(tmp, newfile, false);
                newfile.setLastModified(entry.getLastModified());
                if (tmp.exists()) {
                    tmp.delete();
                }
            }
            if (verbose) {
                out.println("end to extract");
            }
        } finally {
            expander.close();
        }
    }

    /**
     * Xy[X𖄂߂B
     * 񂪎w肵蒷ꍇ͂̂܂ܕԂB
     * @param s 
     * @param length ߂Ƃ̒
     * @param forward <code>true</code>̏ꍇAXy[X𕶎̑Oɖ߂
     * @return Xy[X߂ꂽ
     */
    private static String padSpace(String s, int length, boolean forward) {
        if (s.length() > length) {
            return s;
        }
        StringBuffer buffer = new StringBuffer("        "); // 8
        while (buffer.length() < length) {
            buffer.append(buffer);
        }
        String space = buffer.substring(s.length(), length);
        return forward ? space + s : s + space;
    }

    /**
     * Expander̎擾B
     * @param src (WJ)Ƀt@C
     * @return Expander
     * @throws IOException o̓G[ꍇ 
     */
    public static Expander getExpander(File src) throws IOException {
        return getExpander(src, ArchiveType.getType(src));
    }

    /**
     * Expander̎擾B
     * @param src (WJ)Ƀt@C
     * @param type Ƀ^Cv
     * @return Expander
     * @throws IOException o̓G[ꍇ 
     */
    public static Expander getExpander(File src, ArchiveType type) throws IOException {
        if (type == ArchiveType.TAR) {
            return new TarExpander(new FileInputStream(src));
        } else if (type == ArchiveType.TGZ) {
            return new TarExpander(new GZIPInputStream(new FileInputStream(src)));
        } else if (type == ArchiveType.ZIP) {
            return new ZipExpander(new FileInputStream(src));
        } else if (type == ArchiveType.LZH) {
            return new LzhExpander(new FileInputStream(src));
        } else {
            throw new IllegalStateException("File: " + src + ", ArchiverType: " + type);
        }
    }

    /**
     * IB
     * @throws IOException o̓G[ꍇ 
     */
    public void close() throws IOException {
        expander.close();
    }

}
