package jp.sourceforge.armadillo;

import java.io.*;
import java.util.zip.*;

import jp.sourceforge.armadillo.io.*;
import jp.sourceforge.armadillo.lzh.*;

/**
 * LZHA[JCoB
 */
public final class LzhArchiver implements Archiver {

    private static final byte HEADER_LEVEL = 1;

    private LzhOutputStream los;
    private boolean includesDirectory;

    /**
     * LzhArchiver̐B
     * @param os OutputStream
     */
    public LzhArchiver(OutputStream os) {
        this.los = new LzhOutputStream(os);
    }

    /**
     * includesDirectory̐ݒB
     * @param includesDirectory includesDirectory
     */
    public void setIncludesDirectory(boolean includesDirectory) {
        this.includesDirectory = includesDirectory;
    }

    /* (overridden)
     * @see jp.sourceforge.armadillo.Archiver#addEntry(jp.sourceforge.armadillo.ArchiveEntry, java.io.File)
     */
    public void addEntry(ArchiveEntry ae, File file) throws IOException {
        InputStream is = new FileInputStream(file);
        try {
            addEntry(ae, is, file.length());
        } finally {
            is.close();
        }
    }

    /* (overridden)
     * @see jp.sourceforge.armadillo.Archiver#addEntry(jp.sourceforge.armadillo.ArchiveEntry, java.io.InputStream, long)
     */
    public void addEntry(ArchiveEntry ae, InputStream is, long length) throws IOException {
        assert (is == null) == (length == 0);
        long size;
        long compressedSize;
        if (ae.isDirectory()) {
            // fBNg
            if (!includesDirectory) {
                return;
            }
            LzhEntry entry = new LzhEntry(ae.getName());
            entry.setMethod(LzhMethod.LHD);
            entry.setLastModified(ae.getLastModified());
            los.putNextEntry(entry);
            los.closeEntry();
            size = 0L;
            compressedSize = 0L;
        } else {
            // t@C
            // calculate size
            Trial trial = new Trial();
            File tmp = Utilities.createTemporaryFile();
            if (is != null) {
                OutputStream os = new FileOutputStream(tmp);
                try {
                    long remaining = length;
                    byte[] bytes = new byte[8192];
                    while (remaining > 0) {
                        int readLength = is.read(bytes);
                        if (readLength < 0) {
                            break;
                        }
                        trial.write(bytes, 0, readLength);
                        os.write(bytes, 0, readLength);
                        remaining -= readLength;
                    }
                } finally {
                    os.close();
                    trial.flush();
                }
            }
            final long inputSize = tmp.length();
            // prepare header
            String method;
            compressedSize = trial.getCompressedSize();
            if (compressedSize >= inputSize) {
                method = LzhMethod.LH0;
                compressedSize = inputSize;
            } else {
                method = LzhMethod.LH5;
            }
            LzhEntry entry = new LzhEntry(ae.getName());
            entry.setHeaderLevel(HEADER_LEVEL);
            entry.setMethod(method);
            entry.setCompressedSize(compressedSize);
            entry.setSize(inputSize);
            entry.setLastModified(ae.getLastModified());
            entry.setCrc(trial.getCRC());
            los.putNextEntry(entry);
            // write
            InputStream input = new FileInputStream(tmp);
            try {
                size = IOUtilities.transferAll(input, los);
                entry.setSize(size);
                los.closeEntry();
            } finally {
                input.close();
            }
            tmp.delete();
        }
        ae.setSize(size);
        ae.setCompressedSize(compressedSize);
        ae.setAdded(true);
    }

    /**
     * TCYZB
     */
    private static final class Trial {

        private LzssOutputStream los;
        private SizeDetectionOutputStream compressedSizeCounter;
        private Checksum crc;

        /**
         * LzhArchiver.Trial̐B
         */
        Trial() {
            this.compressedSizeCounter = new SizeDetectionOutputStream();
            LzhHuffmanEncoder encoder = new LzhHuffmanEncoder(compressedSizeCounter, 3);
            this.los = new LzssOutputStream(encoder, 8192, 256, 3);
            this.crc = CRC16.newInstanceForHeader();
        }

        /**
         * f[^ށB
         * @param bytes oCgf[^
         * @param offset ItZbg
         * @param length 
         * @throws IOException o̓G[ꍇ
         */
        void write(byte[] bytes, int offset, int length) throws IOException {
            los.write(bytes, offset, length);
            crc.update(bytes, offset, length);
        }

        /**
         * o͂tbVB
         * @throws IOException o̓G[ꍇ
         */
        void flush() throws IOException {
            los.flush();
        }

        /**
         * kTCY̎擾
         * @return kTCY
         */
        long getCompressedSize() {
            return compressedSizeCounter.getSize();
        }

        /**
         * CRC̎擾B
         * @return CRC
         */
        short getCRC() {
            return (short)(crc.getValue() & 0xFFFF);
        }

    }

    /* (overridden)
     * @see java.io.Closeable#close()
     */
    public void close() throws IOException {
        los.close();
    }

}
