package jp.sfjp.armadillo.archive.lzh;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.util.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.io.*;
import jp.sfjp.armadillo.time.*;

/**
 * Dump LZH archive header.
 */
public final class DumpLzhHeader extends DumpArchiveHeader {

    private static final TimeT TIME_T = new TimeT();
    private static final FTime FTIME = new FTime();
    private static final FileTime FILETIME = new FileTime();
    private static final int siglen = 5;
    private static final int siglen_m1 = siglen - 1;

    private static final int USHORT_MASK = 0xFFFF;
    private static final int UBYTE_MASK = 0xFF;

    private static final int BUFFER_SIZE = 1024;
    private static final int LEVEL_OFFSET = 20;
    private static final byte PATH_DELIMITER = (byte)0xFF;
    private static final String ERROR_PREFIX = "invalid header: ";

    private final ByteBuffer buffer;

    public DumpLzhHeader() {
        this.buffer = ByteBuffer.allocate(BUFFER_SIZE).order(ByteOrder.LITTLE_ENDIAN);
    }

    @Override
    public void dump(InputStream is, PrintWriter out) throws IOException {
        final int bufferSize = 65536;
        byte[] bytes = new byte[bufferSize];
        long p = 0;
        RewindableInputStream pis = new RewindableInputStream(is, bufferSize);
        while (true) {
            Arrays.fill(bytes, (byte)0);
            final int readSize = pis.read(bytes);
            if (readSize <= 0)
                break;
            final int offset = findHeader(bytes);
            if (offset < 0) {
                if (readSize < siglen)
                    break;
                pis.rewind(siglen_m1);
                p += readSize - siglen_m1;
                continue;
            }
            final int head = offset - 2;
            if (offset == 0 || offset < bufferSize - siglen)
                pis.rewind(readSize - head);
            else
                throw new AssertionError("else");
            p += head;
            printOffset(out, p);
            try {
                p += dumpHeader(pis, out);
            }
            catch (Exception ex) {
                warn(out, "unexpected error: %s", ex);
                p += 7;
            }
        }
        printEnd(out, "LZH", p);
    }

    static int findHeader(byte[] bytes) {
        // pattern = "-l**-"
        for (int i = 0; i < bytes.length - siglen_m1; i++)
            if (bytes[i] == '-' && bytes[i + 1] == 'l' && bytes[i + 4] == '-')
                return i;
        return -1;
    }

    public long dumpHeader(InputStream is, PrintWriter out) throws IOException {
        buffer.clear();
        buffer.limit(LEVEL_OFFSET + 1);
        Channels.newChannel(is).read(buffer);
        int headerLength0 = buffer.position();
        if (headerLength0 == 0
            || (headerLength0 == 1 && buffer.get(0) == 0x00)
            || (headerLength0 <= LEVEL_OFFSET + 1 && buffer.getShort(0) == 0x00))
            return -1;
        if (headerLength0 != LEVEL_OFFSET + 1)
            return -1; // warn: next header length
        final int level = buffer.get(LEVEL_OFFSET);
        buffer.rewind();
        int headerLength;
        switch (level) {
            case 0:
            case 1:
                headerLength = (buffer.get() & UBYTE_MASK) + 2;
                break;
            case 2:
                headerLength = buffer.getShort() & USHORT_MASK;
                break;
            default:
                throw new LzhException("unsupported header level (=" + level + ")");
        }
        if (headerLength == 0)
            return -1;
        assert headerLength >= 0 && headerLength < BUFFER_SIZE : "header length = " + headerLength;
        buffer.limit(headerLength);
        buffer.position(LEVEL_OFFSET + 1);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() != headerLength)
            throw new LzhException(ERROR_PREFIX + "header length = " + headerLength);
        final long readLength;
        switch (level) {
            case 0:
                readLength = dumpLevel0(is, out);
                break;
            case 1:
                readLength = dumpLevel1(is, out);
                break;
            case 2:
                readLength = dumpLevel2(is, out);
                break;
            default:
                throw new IllegalStateException("unexpected state");
        }
        return readLength;
    }

    private long dumpLevel0(InputStream is, PrintWriter out) throws IOException {
        long readLength = 0;
        // Header Level 0
        final byte headerLength; // length of this header
        final byte checksum; // checksum of header (SUM)
        final byte[] method; // compression method
        final int skipSize; // skip size
        final int size; // uncompressed size
        final short mtime; // last modified time (MS-DOS time)
        final short mdate; // last modified date (MS-DOS time)
        final byte attribute; // file attribute
        final byte level; // header level
        final byte nameLength; // length of path name
        final byte[] name; // path name
        final short crc; // checksum of file (CRC16)
        final short extend; // -
        // ---
        buffer.position(0);
        headerLength = buffer.get();
        checksum = buffer.get();
        method = getBytes(5);
        skipSize = buffer.getInt();
        size = buffer.getInt();
        mtime = buffer.getShort();
        mdate = buffer.getShort();
        attribute = buffer.get();
        level = buffer.get();
        nameLength = buffer.get();
        name = getBytes(nameLength);
        crc = buffer.getShort();
        extend = buffer.getShort();
        printHeaderName(out, "Level 0 Header");
        p(out, "headerLength", headerLength);
        p(out, "checksum", checksum);
        p(out, "method", method);
        p(out, "skipSize", skipSize);
        p(out, "size", size);
        p(out, "mtime", mtime);
        p(out, "mdate", mdate);
        p(out, "attribute", attribute);
        p(out, "level", level);
        p(out, "nameLength", nameLength);
        p(out, "name", name);
        p(out, "crc", crc);
        p(out, "extend", extend);
        readLength += buffer.position();
        if (skipSize > 0)
            readLength += is.skip(skipSize);
        return readLength;
    }

    private long dumpLevel1(InputStream is, PrintWriter out) throws IOException {
        long readLength = 0;
        // Header Level 1
        final byte headerLength; // length of this header
        final byte checksum; // checksum of header (SUM)
        final byte[] method; // compression method
        final int skipSize; // skip size
        final int size; // uncompressed size
        final short mtime; // last modified time (MS-DOS time)
        final short mdate; // last modified date (MS-DOS time)
        final byte reserved; // reserved
        final byte level; // header level
        final byte nameLength; // length of path name
        final byte[] name; // path name
        final short crc; // checksum of file (CRC16)
        final byte osIdentifier; // OS identifier which compressed this
        final short extendHeaderSize; //
        // ---
        buffer.position(0);
        headerLength = buffer.get();
        checksum = buffer.get();
        method = getBytes(5);
        skipSize = buffer.getInt();
        size = buffer.getInt();
        mtime = buffer.getShort();
        mdate = buffer.getShort();
        reserved = buffer.get();
        level = buffer.get();
        nameLength = buffer.get();
        name = getBytes(nameLength);
        crc = buffer.getShort();
        osIdentifier = buffer.get();
        extendHeaderSize = buffer.getShort();
        printHeaderName(out, "Level 1 Header");
        p(out, "headerLength", headerLength);
        p(out, "checksum", checksum);
        p(out, "method", method);
        p(out, "skipSize", skipSize);
        p(out, "size", size);
        p(out, "mtime", mtime);
        p(out, "mdate", mdate);
        p(out, "reserved", reserved);
        p(out, "level", level);
        p(out, "nameLength", nameLength);
        p(out, "name", name);
        p(out, "crc", crc & 0xFFFFFFFFL);
        p(out, "osIdentifier", osIdentifier);
        p(out, "extendHeaderSize", extendHeaderSize);
        // p(out, "extendHeader", extendHeader);
        readLength += buffer.position();
        if (skipSize > 0)
            readLength += is.skip(skipSize);
        return readLength;
    }

    private long dumpLevel2(InputStream is, PrintWriter out) throws IOException {
        long readLength = 0;
        // Header Level 2
        final short headerLength; // length of this header
        final byte[] method; // compression method
        final int compressedSize; // compressed size
        final int size; // uncompressed size
        final int mtime; // last modified timestamp (POSIX time)
        final byte reserved; // reserved
        final byte level; // header level
        final short crc; // checksum of file (CRC16)
        final byte osIdentifier; // OS identifier which compressed this
        final short firstExHeaderLength; // length of next (first) extend header
        // ---
        buffer.position(0);
        headerLength = buffer.getShort();
        method = getBytes(5);
        compressedSize = buffer.getInt();
        size = buffer.getInt();
        mtime = buffer.getInt();
        reserved = buffer.get();
        level = buffer.get();
        crc = buffer.getShort();
        osIdentifier = buffer.get();
        firstExHeaderLength = buffer.getShort();
        final String methodName = new String(method);
        ByteArrayOutputStream x01 = new ByteArrayOutputStream();
        ByteArrayOutputStream x02 = new ByteArrayOutputStream();
        List<String> a = new ArrayList<String>();
        int p = 0;
        short nextHeaderLength = firstExHeaderLength;
        while (nextHeaderLength > 0) {
            final byte identifier = buffer.get();
            final int length = nextHeaderLength - 3; // datalength = len - 1(id) - 2(nextlen)
            buffer.mark();
            a.add(String.format("ex: id=%02X, len=%d", identifier, length));
            switch (identifier) {
                case 0x00: // common
                    final long hcrc = buffer.getShort();
                    a.add(String.format("  hCRC=0x%04X", hcrc & USHORT_MASK));
                    try {
                        for (int i = 0, n = length - 2; i < n; i++)
                            a.add(String.format("  [%d]=%02X", i, buffer.get() & UBYTE_MASK));
                    }
                    catch (Exception ex) {
                        warn(out, ">>> %s", ex);
                        return buffer.position();
                    }
                    break;
                case 0x01: // file name
                    x01.write(nextPath(length));
                    break;
                case 0x02: // dir name
                    x02.write(nextPath(length));
                    break;
                case 0x41: // MS Windows timestamp
                    final long wctime = buffer.getLong();
                    final long wmtime = buffer.getLong();
                    final long watime = buffer.getLong();
                    Date wcd = new Date(FILETIME.toMilliseconds(wctime));
                    Date wmd = new Date(FILETIME.toMilliseconds(wmtime));
                    Date wad = new Date(FILETIME.toMilliseconds(watime));
                    a.add(String.format("  Windows ctime = %s (%d)", wcd, wctime));
                    a.add(String.format("  Windows mtime = %s (%d)", wmd, wmtime));
                    a.add(String.format("  Windows atime = %s (%d)", wad, watime));
                    break;
                case 0x42: // MS filesize
                    a.add("  MS compsize = " + buffer.getLong());
                    a.add("  MS filesize = " + buffer.getLong());
                    break;
                case 0x54: // UNIX time_t
                    final long umtime = buffer.getLong();
                    Date umd = new Date(FILETIME.toMilliseconds(umtime));
                    a.add(String.format("  UNIX mtime = %s (%d)", umd, umtime));
                    break;
                default:
                    for (int i = 0; i < length; i++)
                        a.add(String.format("  [%d]=%02X", i, buffer.get() & UBYTE_MASK));
            }
            buffer.reset();
            buffer.position(buffer.position() + length);
            nextHeaderLength = buffer.getShort();
            p = buffer.position();
        }
        final int headerLength0 = p;
        ByteArrayOutputStream nameb = new ByteArrayOutputStream();
        if (x02.size() > 0) {
            nameb.write(x02.toByteArray());
            if (!nameb.toString().endsWith("/"))
                nameb.write('/');
        }
        if (x01.size() > 0)
            nameb.write(x01.toByteArray());
        printHeaderName(out, "Level 2 Header");
        p(out, "name", nameb.toByteArray());
        p(out, "mtime as Date", new Date(TIME_T.toMilliseconds(mtime)));
        p(out, "method", methodName);
        p(out, "headerLength", headerLength);
        p(out, "compressedSize", compressedSize);
        p(out, "size", size);
        p(out, "mtime", mtime);
        p(out, "reserved", reserved);
        p(out, "level", level);
        p(out, "crc", crc);
        p(out, "osIdentifier", (char)osIdentifier);
        p(out, "firstExHdrLen", firstExHeaderLength);
        for (final String x : a)
            p(out, "  ", x);
        if (headerLength != headerLength0)
            warn(out, "header length is expected %d , but was %d%n", headerLength, p);
        readLength += headerLength0;
        if (compressedSize > 0)
            readLength += is.skip(compressedSize);
        return readLength;
    }

    private void p(PrintWriter out, String name, Object value) {
        if (value instanceof Character) {
            final char c = ((Character)value).charValue();
            out.printf("  %-16s [%04X] ('%s')%n", name, (int)c, value);
        }
        else if (value instanceof Number) {
            final long v;
            final int w;
            if (value instanceof Byte) {
                v = ((Byte)value) & 0xFF;
                w = 2;
            }
            else if (value instanceof Short) {
                v = ((Short)value) & 0xFFFF;
                w = 4;
            }
            else if (value instanceof Integer) {
                v = ((Integer)value) & 0xFFFFFFFFL;
                w = 8;
            }
            else {
                v = ((Number)value).longValue();
                w = 1;
            }
            out.printf("  %-16s [%0" + w + "X] (%d)%n", name, value, v);
        }
        else if (value instanceof byte[])
            out.printf("  * %s = %s%n", name, new String((byte[])value));
        else if (value instanceof Date)
            out.printf("  * %s = %s%n", name, value);
        else
            out.printf("  %s %s%n", name, value);
    }

    private byte[] getBytes(int length) {
        byte[] bytes = new byte[length];
        for (int i = 0; i < bytes.length; i++)
            bytes[i] = buffer.get();
        return bytes;
    }

    private byte[] nextPath(int length) {
        byte[] bytes = new byte[length];
        buffer.get(bytes);
        for (int i = 0; i < length; i++)
            if ((bytes[i] & PATH_DELIMITER) == PATH_DELIMITER)
                bytes[i] = '/';
        return bytes;
    }

    static Date toDate(int mdate, int mtime) {
        final int ftime = (mdate << 16) | mtime & 0xFFFF;
        return new Date(FTIME.toMilliseconds(ftime));
    }

}
