package jp.sfjp.armadillo.archive.lzh;

import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.util.*;

/**
 * LZH archive header.
 * This class supports header level 0, 1, 2.
 */
public final class LzhHeader {

    /**
     * <code>Header level 0.</code>
     */
    public static final byte HEADER_LEVEL_0 = 0;
    /**
     * <code>Header level 1.</code>
     */
    public static final byte HEADER_LEVEL_1 = 1;
    /**
     * <code>Header level 2.</code>
     */
    public static final byte HEADER_LEVEL_2 = 2;

    private static final int BUFFER_SIZE = 1024;
    private static final long MAX_DATA_SIZE = 0xFFFFFFFFL;
    private static final long UINT_MASK = 0xFFFFFFFFL;
    private static final int USHORT_MASK = 0xFFFF;
    private static final int UBYTE_MASK = 0xFF;
    private static final int LEVEL_OFFSET = 20;
    private static final byte FILETYPE_FILE = 0x20;
    private static final byte PLATFORM_JAVA = 'J';
    private static final byte PATH_DELIMITER = (byte)0xFF;
    private static final String ISO_8859_1 = "iso-8859-1";
    private static final String ERROR_PREFIX = "invalid header: ";

    private final ByteBuffer buffer;

    public LzhHeader() {
        this.buffer = ByteBuffer.allocate(BUFFER_SIZE).order(ByteOrder.LITTLE_ENDIAN);
    }

    public LzhEntry read(InputStream is) throws IOException {
        buffer.clear();
        buffer.limit(LEVEL_OFFSET + 1);
        Channels.newChannel(is).read(buffer);
        int readLength = buffer.position();
        if (readLength == 0
            || (readLength == 1 && buffer.get(0) == 0x00)
            || (readLength <= LEVEL_OFFSET + 1 && buffer.getShort(0) == 0x00))
            return null;
        if (readLength != LEVEL_OFFSET + 1)
            return null; // warn: next header length
        int level = buffer.get(LEVEL_OFFSET);
        buffer.rewind();
        int headerLength;
        switch (level) {
            case 0:
            case 1:
                headerLength = (buffer.get() & UBYTE_MASK) + 2;
                break;
            case 2:
                headerLength = buffer.getShort() & USHORT_MASK;
                break;
            default:
                throw new LzhException("unsupported header level (=" + level + ")");
        }
        if (headerLength == 0)
            return null;
        assert headerLength >= 0 && headerLength < BUFFER_SIZE : "header length = " + headerLength;
        buffer.limit(headerLength);
        buffer.position(LEVEL_OFFSET + 1);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() != headerLength)
            throw new LzhException(ERROR_PREFIX + "header length = " + headerLength);
        LzhEntry entry;
        switch (level) {
            case 0:
                entry = readLevel0();
                break;
            case 1:
                entry = readLevel1(is);
                break;
            case 2:
                entry = readLevel2();
                break;
            default:
                throw new IllegalStateException("unexpected state");
        }
        switch (level) {
            case 0:
            case 1:
                entry.headerLength += 2;
                break;
            default:
        }
        assert entry.headerLength == headerLength;
        return entry;
    }

    @SuppressWarnings("unused")
    private LzhEntry readLevel0() throws LzhException {
        // Header Level 0
        final byte headerLength; // length of this header
        final byte checksum; // checksum of header (SUM)
        final byte[] method; // compression method
        final int skipSize; // compressed size (skip size)
        final int size; // uncompressed size
        final short mtime; // last modified time (MS-DOS time)
        final short mdate; // last modified date (MS-DOS time)
        final byte attribute; // file attribute
        final byte level; // header level
        final byte nameLength; // length of path name
        final byte[] name; // path name
        final short crc; // checksum of file (CRC16)
        final short extend; // -
        // ---
        buffer.position(0);
        headerLength = buffer.get();
        checksum = buffer.get();
        method = getBytes(5);
        skipSize = buffer.getInt();
        size = buffer.getInt();
        mtime = buffer.getShort();
        mdate = buffer.getShort();
        attribute = buffer.get();
        level = buffer.get();
        nameLength = buffer.get();
        name = getBytes(nameLength);
        crc = buffer.getShort();
        assert level == HEADER_LEVEL_0;
        LzhEntry entry = new LzhEntry(0);
        entry.headerLength = headerLength;
        entry.checksum = checksum & UBYTE_MASK;
        assert (method[0] & 0x7F) == method[0];
        assert (method[1] & 0x7F) == method[1];
        assert (method[2] & 0x7F) == method[2];
        assert (method[3] & 0x7F) == method[3];
        assert (method[4] & 0x7F) == method[4];
        entry.method = new String(method);
        entry.compressedSize = skipSize;
        entry.size = size;
        entry.setFtime(mdate, mtime);
        entry.attribute = attribute;
        entry.setName(name);
        entry.crc = crc;
        entry.calculatedChecksum = calculateChecksum(2, buffer.limit());
        return entry;
    }

    private LzhEntry readLevel1(InputStream is) throws IOException {
        // Header Level 1
        final byte headerLength; // length of this header
        final byte checksum; // checksum of header (SUM)
        final byte[] method; // compression method
        final int skipSize; // skip size
        final int size; // uncompressed size
        final short mtime; // last modified time (MS-DOS time)
        final short mdate; // last modified date (MS-DOS time)
        final byte reserved; // reserved
        final byte level; // header level
        final byte nameLength; // length of path name
        final byte[] name; // path name
        final short crc; // checksum of file (CRC16)
        final byte osIdentifier; // OS identifier which compressed this
        // ---
        buffer.position(0);
        LzhEntry entry = new LzhEntry(1);
        headerLength = buffer.get();
        checksum = buffer.get();
        method = getBytes(5);
        skipSize = buffer.getInt();
        size = buffer.getInt();
        mtime = buffer.getShort();
        mdate = buffer.getShort();
        reserved = buffer.get();
        level = buffer.get();
        nameLength = buffer.get();
        name = getBytes(nameLength);
        crc = buffer.getShort();
        osIdentifier = buffer.get();
        assert level == HEADER_LEVEL_1;
        assert reserved == FILETYPE_FILE;
        entry.headerLength = headerLength;
        entry.checksum = checksum;
        assert (method[0] & 0x7F) == method[0];
        assert (method[1] & 0x7F) == method[1];
        assert (method[2] & 0x7F) == method[2];
        assert (method[3] & 0x7F) == method[3];
        assert (method[4] & 0x7F) == method[4];
        entry.method = new String(method);
        entry.size = size;
        entry.setFtime(mdate, mtime);
        entry.setName(name);
        entry.crc = crc;
        entry.osIdentifier = osIdentifier;
        entry.calculatedChecksum = calculateChecksum(2, buffer.limit());
        int extendedHeaderSize = 0;
        entry.compressedSize = skipSize - extendedHeaderSize;
        return entry;
    }

    private LzhEntry readLevel2() throws IOException {
        // Header Level 2
        final short headerLength; // length of this header
        final byte[] method; // compression method
        final int compressedSize; // compressed size
        final int size; // uncompressed size
        final int mtime; // last modified timestamp (POSIX time)
        final byte reserved; // reserved
        final byte level; // header level
        final short crc; // checksum of file (CRC16)
        final byte osIdentifier; // OS identifier which compressed this
        final short firstExHeaderLength; // length of next (first) extend header
        // ---
        buffer.position(0);
        headerLength = buffer.getShort();
        method = getBytes(5);
        compressedSize = buffer.getInt();
        size = buffer.getInt();
        mtime = buffer.getInt();
        reserved = buffer.get();
        level = buffer.get();
        crc = buffer.getShort();
        osIdentifier = buffer.get();
        firstExHeaderLength = buffer.getShort();
        assert level == HEADER_LEVEL_2;
        assert reserved == FILETYPE_FILE;
        LzhEntry entry = new LzhEntry(2);
        entry.headerLength = headerLength;
        assert (method[0] & 0x7F) == method[0];
        assert (method[1] & 0x7F) == method[1];
        assert (method[2] & 0x7F) == method[2];
        assert (method[3] & 0x7F) == method[3];
        assert (method[4] & 0x7F) == method[4];
        entry.method = new String(method);
        entry.compressedSize = compressedSize;
        entry.size = size;
        entry.setTimeT(mtime);
        entry.crc = crc;
        entry.osIdentifier = osIdentifier;
        int warningCount = 0;
        ByteArrayOutputStream x01 = new ByteArrayOutputStream();
        ByteArrayOutputStream x02 = new ByteArrayOutputStream();
        short nextHeaderLength = firstExHeaderLength;
        while (nextHeaderLength > 0) {
            final byte identifier = buffer.get();
            final int length = nextHeaderLength - 3; // datalength = len - 1(id) - 2(nextlen)
            buffer.mark();
            switch (identifier) {
                case 0x00: // common
                    buffer.getShort(); // header CRC
                    break;
                case 0x01: // file name
                    x01.write(nextPath(length));
                    break;
                case 0x02: // dir name
                    x02.write(nextPath(length));
                    break;
                case 0x39: // plural disk
                    ++warningCount;
                    break;
                case 0x41: // MS Windows timestamp
                    buffer.getLong(); // ctime
                    entry.setFileTime(buffer.getLong());
                    buffer.getLong(); // atime
                    break;
                case 0x42: // MS filesize
                    entry.compressedSize = buffer.getLong();
                    entry.size = buffer.getLong();
                    break;
                case 0x54: // UNIX time_t
                    entry.setTimeT(buffer.getInt());
                    break;
                case 0x40: // MS attribute
                case 0x50: // UNIX permission
                case 0x51: // UNIX uid gid
                case 0x52: // UNIX group
                case 0x53: // UNIX user
                    break;
                case 0x3F: // comment
                default:
                    // ignore
            }
            buffer.reset();
            buffer.position(buffer.position() + length);
            nextHeaderLength = buffer.getShort();
        }
        ByteArrayOutputStream name = new ByteArrayOutputStream();
        if (x02.size() > 0) {
            name.write(x02.toByteArray());
            if (!name.toString().endsWith("/"))
                name.write('/');
        }
        if (x01.size() > 0)
            name.write(x01.toByteArray());
        assert (x01.size() == 0) == entry.isDirectory();
        entry.setName(name.toByteArray());
        assert warningCount == 0;
        return entry;
    }

    public void write(OutputStream os, LzhEntry entry) throws IOException {
        if (!entry.method.matches("-l[hz][a-z0-9]-"))
            throw new LzhException("invalid compression type: " + entry.method);
        if (entry.compressedSize > Integer.MAX_VALUE)
            throw new LzhException("too large compressed size: " + entry.compressedSize);
        if (entry.size > Integer.MAX_VALUE)
            throw new LzhException("too large size: " + entry.size);
        if (entry.getNameAsBytes().length > Short.MAX_VALUE)
            throw new LzhException("too long file name: length=" + entry.size);
        if (entry.isDirectory())
            entry.method = LzhMethod.LHD;
        buffer.clear();
        switch (entry.headerLevel) {
            case HEADER_LEVEL_0:
                writeLevel0(entry);
                break;
            case HEADER_LEVEL_1:
                writeLevel1(entry);
                break;
            case HEADER_LEVEL_2:
                writeLevel2(entry);
                break;
            default:
                throw new IllegalArgumentException(ERROR_PREFIX
                                                   + "header-level="
                                                   + entry.headerLevel);
        }
        buffer.flip();
        Channels.newChannel(os).write(buffer);
    }

    @SuppressWarnings("unused")
    private void writeLevel0(LzhEntry entry) throws IOException {
        // Header Level 0
        final byte headerLength; // length of this header
        final byte checksum; // checksum of header (SUM)
        final byte[] method; // compression method
        final int skipSize; // skip size
        final int size; // uncompressed size
        final short mtime; // last modified time (MS-DOS time)
        final short mdate; // last modified date (MS-DOS time)
        final byte attribute; // file attribute
        final byte level; // header level
        final byte nameLength; // length of path name
        final byte[] name; // path name
        final short crc; // checksum of file (CRC16)
        final short extend; // -
        // ---
        assert entry.compressedSize >= 0 && entry.compressedSize <= MAX_DATA_SIZE;
        assert entry.size >= 0 && entry.size <= MAX_DATA_SIZE;
        assert entry.attribute == FILETYPE_FILE;
        name = entry.getNameAsBytes();
        assert name.length <= 0xFF;
        method = entry.method.getBytes(ISO_8859_1);
        assert method.length == 5;
        size = (int)(entry.size & UINT_MASK);
        final int ftime = entry.getFTime();
        mtime = (short)(ftime & USHORT_MASK);
        mdate = (short)((ftime << 16) & USHORT_MASK);
        attribute = entry.attribute;
        level = HEADER_LEVEL_0;
        nameLength = (byte)(name.length & UBYTE_MASK);
        crc = entry.crc;
        assert buffer.position() == 0;
        buffer.putShort((short)0); // skip
        buffer.put(method);
        buffer.position(11); // skip
        buffer.putInt(size);
        buffer.putShort(mtime);
        buffer.putShort(mdate);
        buffer.put(attribute);
        buffer.put(level);
        buffer.put(nameLength);
        buffer.put(name);
        buffer.putShort(crc);
        final int extendedHeaderSize = 0; // not supported
        final int endp = buffer.position();
        if ((endp - 2) > 0xFF)
            throw new LzhException("invalid header length: " + (endp - 2));
        headerLength = (byte)((endp - 2) & UBYTE_MASK);
        checksum = (byte)(calculateChecksum(2, endp) & UBYTE_MASK);
        if (entry.compressedSize > 0)
            skipSize = calculateSkipSize(entry, size, extendedHeaderSize);
        else
            skipSize = 0;
        buffer.put(0, headerLength);
        buffer.put(1, checksum);
        buffer.putInt(7, skipSize);
    }

    @SuppressWarnings("unused")
    private void writeLevel1(LzhEntry entry) throws IOException {
        // Header Level 1
        final byte headerLength; // length of this header
        final byte checksum; // checksum of header (SUM)
        final byte[] method; // compression method
        final int skipSize; // skip size
        final int size; // uncompressed size
        final short mtime; // last modified time (MS-DOS time)
        final short mdate; // last modified date (MS-DOS time)
        final byte reserved; // reserved
        final byte level; // header level
        final byte nameLength; // length of path name
        final byte[] name; // path name
        final short crc; // checksum of file (CRC16)
        final byte osIdentifier; // OS identifier which compressed this
        // ---
        assert entry.compressedSize >= 0 && entry.compressedSize <= Integer.MAX_VALUE;
        assert entry.size >= 0 && entry.size <= Integer.MAX_VALUE;
        name = entry.getNameAsBytes();
        assert name.length <= 0xFF;
        method = entry.method.getBytes(ISO_8859_1);
        assert method.length == 5;
        assert entry.size >= 0 && entry.size <= MAX_DATA_SIZE;
        size = (int)(entry.size & UINT_MASK);
        final int ftime = entry.getFTime();
        mtime = (short)(ftime & USHORT_MASK);
        mdate = (short)((ftime << 16) & USHORT_MASK);
        reserved = FILETYPE_FILE;
        level = HEADER_LEVEL_1;
        nameLength = (byte)(name.length & UBYTE_MASK);
        osIdentifier = PLATFORM_JAVA; // fixed value in writing
        assert buffer.position() == 0;
        buffer.putShort((short)0); // skip
        buffer.put(method);
        buffer.putInt(11); // skip
        buffer.putInt(size);
        buffer.putShort(mtime);
        buffer.putShort(mdate);
        buffer.put(reserved);
        buffer.put(level);
        buffer.put(nameLength);
        buffer.put(name);
        buffer.putShort((short)0); // skip
        buffer.put(osIdentifier);
        final short extendedHeaderSize = 0;
        buffer.putShort(extendedHeaderSize);
        buffer.put((byte)0);
        final int endp = buffer.position();
        if ((endp - 2) > 0xFF)
            throw new LzhException("invalid header length: " + (endp - 2));
        headerLength = (byte)((endp - 2) & UBYTE_MASK);
        checksum = (byte)calculateChecksum(2, endp);
        if (entry.compressedSize > 0)
            skipSize = calculateSkipSize(entry, size, extendedHeaderSize);
        else
            skipSize = 0;
        buffer.put(0, headerLength);
        buffer.put(1, checksum);
        buffer.putInt(7, skipSize);
    }

    @SuppressWarnings("unused")
    private void writeLevel2(LzhEntry entry) throws IOException {
        // Header Level 2
        final short headerLength; // length of this header
        final byte[] method; // compression method
        final int compressedSize; // compressed size
        final int size; // uncompressed size
        final int mtime; // last modified timestamp (POSIX time)
        final byte reserved; // reserved
        final byte level; // header level
        final short crc; // checksum of file (CRC16)
        final byte osIdentifier; // OS identifier which compressed this
        final short firstExHeaderLength; // length of next (first) extend header
        // ---
        assert entry.compressedSize >= 0 && entry.compressedSize <= MAX_DATA_SIZE;
        assert entry.size >= 0 && entry.size <= MAX_DATA_SIZE;
        byte[] nameb = entry.getNameAsBytes();
        assert nameb.length <= 0xFF;
        method = entry.method.getBytes(ISO_8859_1);
        assert method.length == 5;
        final boolean isDirectory = entry.isDirectory();
        assert isDirectory == entry.method.equals(LzhMethod.LHD);
        compressedSize = isDirectory ? 0 : (int)(entry.compressedSize & UINT_MASK);
        size = isDirectory ? 0 : (int)(entry.size & UINT_MASK);
        mtime = entry.getTimeT();
        reserved = FILETYPE_FILE;
        level = HEADER_LEVEL_2;
        crc = entry.crc;
        osIdentifier = PLATFORM_JAVA; // fixed value in writing
        firstExHeaderLength = 0;
        assert buffer.position() == 0;
        buffer.putShort((short)0); // skip
        buffer.put(method);
        buffer.putInt(compressedSize);
        buffer.putInt(size);
        buffer.putInt(mtime);
        buffer.put(reserved);
        buffer.put(level);
        buffer.putShort(crc);
        buffer.put(osIdentifier);
        /* extended */
        // dividing dir and name
        int lastDelimiterOffset = -1;
        for (int i = 0; i < nameb.length; i++)
            if (nameb[i] == '/' || nameb[i] == '\\') {
                nameb[i] = PATH_DELIMITER;
                lastDelimiterOffset = i;
            }
        final int nameOffset = (lastDelimiterOffset < 0) ? 0 : lastDelimiterOffset + 1;
        // 0x01 file name
        final int nameLength = (isDirectory) ? 0 : nameb.length - nameOffset;
        buffer.putShort((short)(nameLength + 3));
        buffer.put((byte)0x01);
        if (nameLength > 0)
            buffer.put(nameb, nameOffset, nameLength);
        if (nameOffset > 0) {
            // 0x02 dir name
            byte[] dirb = Arrays.copyOf(nameb, nameb.length + 1);
            final int dirLength;
            if (isDirectory) {
                if (nameb[nameb.length - 1] != PATH_DELIMITER) {
                    dirLength = dirb.length;
                    dirb[nameb.length - 1] = PATH_DELIMITER;
                }
                else
                    dirLength = nameb.length;
            }
            else
                dirLength = nameOffset;
            buffer.putShort((short)(dirLength + 3));
            buffer.put((byte)0x02);
            buffer.put(dirb, 0, dirLength);
        }
        // 0x41 MS Windows timestamp
        buffer.putShort((short)27);
        buffer.put((byte)0x41);
        buffer.putLong(entry.getFileTime());
        buffer.putLong(entry.getFileTime());
        buffer.putLong(entry.getFileTime());
        // 0x00 common
        buffer.putShort((short)6);
        buffer.put((byte)0x00);
        final int crcIndex = buffer.position();
        buffer.putShort((short)0);
        buffer.put((byte)0x00); // ??
        // no next
        buffer.putShort((short)0);
        // header size
        short headerLength0 = 0;
        headerLength0 += buffer.position();
        if (headerLength0 % 256 == 0) {
            buffer.put((byte)0x00);
            ++headerLength0;
            assert headerLength0 >= 0;
        }
        headerLength = headerLength0;
        LzhChecksum cksum = new LzhChecksum();
        cksum.reset();
        cksum.update(buffer.array(), 0, headerLength);
        buffer.putShort(0, headerLength);
        buffer.putShort(crcIndex, cksum.getShortValue());
    }

    private int calculateChecksum(int start, int end) {
        int sum = 0;
        for (int i = start; i < end; i++)
            sum += buffer.get(i);
        return sum;
    }

    private static int calculateSkipSize(LzhEntry entry, int headerSize, int extendedHeaderSize) throws LzhException {
        assert entry.compressedSize >= 0 && entry.compressedSize <= Integer.MAX_VALUE;
        assert entry.size >= 0 && entry.size <= Integer.MAX_VALUE;
        assert headerSize >= 0 && extendedHeaderSize >= 0;
        int skipSize = 0;
        if (entry.getMethod().isCompressing()) {
            skipSize += entry.compressedSize;
            skipSize -= headerSize;
        }
        else
            skipSize += entry.size;
        skipSize += extendedHeaderSize;
        assert skipSize >= 0 : "skip size = " + skipSize;
        return skipSize;
    }

    private byte[] getBytes(int length) {
        byte[] bytes = new byte[length];
        for (int i = 0; i < bytes.length; i++)
            bytes[i] = buffer.get();
        return bytes;
    }

    private byte[] nextPath(int length) {
        byte[] bytes = new byte[length];
        buffer.get(bytes);
        for (int i = 0; i < length; i++)
            if ((bytes[i] & PATH_DELIMITER) == PATH_DELIMITER)
                bytes[i] = '/';
        return bytes;
    }

}
