package jp.sfjp.armadillo.archive.lzh;

import java.io.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.compression.lzhuf.*;
import jp.sfjp.armadillo.io.*;

public final class LzhOutputStream extends ArchiveOutputStream {

    private LzhHeader header;
    private LzhEntry ongoingEntry;
    private InspectionOutputStream ios;
    private LzhChecksum crc;
    private ByteArrayOutputStream bos;

    public LzhOutputStream(OutputStream os) {
        super(os);
        this.header = new LzhHeader();
        this.crc = new LzhChecksum();
        this.bos = new ByteArrayOutputStream();
    }

    public void putNextEntry(LzhEntry entry) throws IOException {
        if (ongoingEntry != null)
            closeEntry();
        ongoingEntry = entry;
        LzhMethod method = entry.getMethod();
        crc.reset();
        bos.reset();
        frontStream = ios = new InspectionOutputStream(openStream(bos, method), crc);
    }

    static OutputStream openStream(OutputStream os, LzhMethod method) {
        if (method.isCompressing())
            return new LzssOutputStream(new LzhHuffmanEncoder(os, method.getThreshold()),
                                        method.getDictionarySize(),
                                        method.getMatchSize(),
                                        method.getThreshold());
        else
            return os;
    }

    public void closeEntry() throws IOException {
        frontStream.flush();
        if (frontStream != out)
            frontStream.close();
        frontStream = out;
        if (ios != null) {
            ongoingEntry.compressedSize = bos.size();
            ongoingEntry.crc = getCrc();
        }
        writeHeader(ongoingEntry);
        if (ios != null)
            bos.writeTo(out);
        ongoingEntry = null;
        ios = null;
        crc.reset();
        bos.reset();
    }

    public void writeHeader(LzhEntry entry) throws IOException {
        header.write(out, entry);
    }

    public void writeHeader(LzhEntry entry, OutputStream os) throws IOException {
        header.write(os, entry);
    }

    public short getCrc() {
        return crc.getShortValue();
    }

    @Override
    public void close() throws IOException {
        try {
            flush();
            super.write(0); // end of archive
            super.flush();
        }
        finally {
            header = null;
            ongoingEntry = null;
            ios = null;
            crc = null;
            bos = null;
            super.close();
        }
    }

}
