package jp.sfjp.armadillo.archive.zip;

import java.io.*;
import java.nio.channels.*;
import java.util.*;
import java.util.zip.*;
import jp.sfjp.armadillo.archive.*;
import jp.sfjp.armadillo.archive.tar.*;
import jp.sfjp.armadillo.io.*;

public final class ZipFile extends ArchiveFile {

    private File afile;
    private ZipHeader header;
    private RandomAccessFile raf;
    private ZipEntry ongoingEntry;
    private byte[] buffer;
    private ZipEndEntry endEntry;

    public ZipFile(File afile) {
        this.afile = afile;
        this.header = new ZipHeader();
    }

    public ZipFile(File afile, boolean withOpen) throws IOException {
        this(afile);
        open();
    }

    @Override
    public void open() throws IOException {
        if (raf != null)
            throw new IOException("the file has been already opened");
        if (!afile.exists())
            afile.createNewFile();
        this.raf = new RandomAccessFile(afile, "rw");
        this.opened = true;
    }

    @Override
    public void reset() throws IOException {
        if (raf.length() == 0)
            endEntry = new ZipEndEntry();
        else if ((endEntry = readEndHeader()) == null)
            throw new IllegalStateException("failed to read END header");
        ongoingEntry = null;
        currentPosition = endEntry.offsetStartCEN;
        raf.seek(currentPosition);
    }

    @Override
    public ArchiveEntry nextEntry() throws IOException {
        ensureOpen();
        if (ongoingEntry == null)
            reset();
        else {
            long cenLength = 0L;
            cenLength += ZipHeader.LENGTH_CEN;
            cenLength += ongoingEntry.getNameAsBytes().length;
            cenLength += ongoingEntry.extlen;
            currentPosition += cenLength;
        }
        raf.seek(currentPosition);
        ongoingEntry = header.readCEN(Channels.newInputStream(raf.getChannel()));
        return (ongoingEntry == null) ? ArchiveEntry.NULL : ongoingEntry;
    }

    @Override
    public ArchiveEntry newEntry(String name) {
        return new ZipEntry(name);
    }

    @Override
    public void addEntry(ArchiveEntry entry, InputStream is, long length) throws IOException {
        reset();
        List<ZipEntry> entries = getEntries();
        ZipEntry newEntry = new ZipEntry(entry.getName());
        newEntry.copyFrom(entry);
        // overwrite new entry, CEN and END
        raf.seek(endEntry.offsetStartCEN);
        ZipOutputStream zos = new ZipOutputStream(Channels.newOutputStream(raf.getChannel()));
        zos.putNextEntry(newEntry);
        if (length > 0) {
            final long written = IOUtilities.transfer(is, zos, length);
            assert written == length;
            newEntry.setCompressedSize(written);
            newEntry.setSize(length);
        }
        zos.closeEntry();
        zos.flush();
        final long p = raf.getFilePointer();
        raf.seek(endEntry.offsetStartCEN);
        newEntry.flags &= 0xFFF7;
        zos.putNextEntry(newEntry);
        newEntry.reloff = endEntry.offsetStartCEN;
        entries.add(newEntry);
        endEntry.countCENs = (short)entries.size();
        raf.seek(p);
        writeEnd(entries);
    }

    @Override
    public void updateEntry(ArchiveEntry entry, InputStream is, long length) throws IOException {
        super.updateEntry(entry, is, length);
    }

    @Override
    public void removeEntry(ArchiveEntry entry) throws IOException {
        if (!seek(entry))
            throw new TarException("entry " + entry + " not found");
        assert ongoingEntry != null;
        ZipEntry target = this.ongoingEntry;
        ZipEndEntry endEntry = this.endEntry;
        final long offset = target.reloff;
        List<ZipEntry> entries = getEntries(); // reset offset
        raf.seek(offset);
        ZipInputStream zis = new ZipInputStream(Channels.newInputStream(raf.getChannel()));
        ZipEntry locEntry = zis.getNextEntry();
        zis.skip(locEntry.uncompsize);
        zis.closeEntry();
        final long totalLength = getOffset() - offset;
        endEntry.offsetStartCEN -= totalLength;
        truncate(offset, totalLength);
        for (ZipEntry zipEntry : entries)
            if (zipEntry.equalsName(target)) {
                entries.remove(zipEntry);
                break;
            }
        for (ZipEntry zipEntry : entries)
            if (zipEntry.reloff > offset)
                zipEntry.reloff -= totalLength;
        raf.seek(endEntry.offsetStartCEN);
        this.endEntry = endEntry;
        writeEnd(entries);
    }

    void writeEnd(List<ZipEntry> entries) throws IOException {
        ZipEndEntry endEntry = ZipEndEntry.create(getOffset(), entries);
        OutputStream os = Channels.newOutputStream(raf.getChannel());
        for (ZipEntry entry : entries)
            header.writeCEN(os, entry);
        header.writeEND(os, endEntry);
        if (raf.length() > raf.getFilePointer())
            raf.setLength(raf.getFilePointer());
    }

    public void insertEmptySpace(long offset, long blockCount) throws IOException {
        final int blockSize = 8192;
        final long insertLength = blockCount * blockSize;
        raf.seek(raf.length() + insertLength - 1);
        raf.write(0);
        final long x = raf.getFilePointer();
        long p2 = x - blockSize; // last block
        long p1 = p2 - insertLength;
        assert p1 > 0 && p2 > 0;
        byte[] buffer = this.buffer;
        for (int i = 0; i < blockCount && p1 >= offset; i++) {
            raf.seek(p1);
            raf.readFully(buffer);
            raf.seek(p2);
            raf.write(buffer);
            p1 -= blockSize;
            p2 -= blockSize;
        }
        raf.seek(currentPosition = offset);
    }

    @Override
    public long extract(OutputStream os) throws IOException {
        raf.seek(ongoingEntry.reloff);
        InputStream is = Channels.newInputStream(raf.getChannel());
        ZipEntry entry = header.read(is);
        assert entry != null;
        entry.compsize = ongoingEntry.compsize;
        entry.uncompsize = ongoingEntry.uncompsize;
        entry.flags &= (8 ^ 0xFFFF);
        ZipInputStream zis = new ZipInputStream(is);
        zis.openEntry(entry);
        return IOUtilities.transfer(zis, os, ongoingEntry.uncompsize);
    }

    int getOffset() throws IOException {
        final long offset = raf.getFilePointer();
        if (offset > Integer.MAX_VALUE)
            throw new ZipException("size not supported: " + offset);
        return (int)offset;
    }

    List<ZipEntry> getEntries() {
        List<ZipEntry> entries = new ArrayList<ZipEntry>();
        for (ArchiveEntry entry : this)
            entries.add((ZipEntry)entry);
        return entries;
    }

    ZipEndEntry readEndHeader() throws IOException {
        final int commentlen = 0;
        raf.seek(raf.length() - 22 - commentlen);
        return header.readEND(Channels.newInputStream(raf.getChannel()));
    }

    void truncate(final long offset, final long length) throws IOException {
        final int blockSize = 8192;
        long p2 = offset;
        long p1 = p2 + length;
        byte[] bytes = new byte[blockSize];
        while (true) {
            raf.seek(p1);
            final int r = raf.read(bytes);
            if (r <= 0)
                break;
            raf.seek(p2);
            raf.write(bytes, 0, r);
            p2 += r;
            p1 += r;
        }
        raf.setLength(raf.length() - length);
        reset();
    }

    @Override
    public void close() throws IOException {
        try {
            if (raf != null)
                raf.close();
        }
        finally {
            super.close();
            afile = null;
            header = null;
        }
    }

}
