package jp.sfjp.armadillo.archive.zip;

import static java.lang.String.format;
import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.util.*;
import java.util.zip.*;

/**
 * ZIP archive header.
 */
public final class ZipHeader {

    static final int SIGN_LOC = 0x04034B50;
    static final int SIGN_CEN = 0x02014B50;
    static final int SIGN_END = 0x06054B50;
    static final int SIGN_EXT = 0x08074B50;
    static final int LENGTH_LOC = 30;
    static final int LENGTH_CEN = 46;
    static final int LENGTH_END = 22;
    static final int LENGTH_EXT = 16;

    private final ByteBuffer buffer;
    private final List<ZipEntry> entries;
    private long offset;

    public ZipHeader() {
        this.buffer = ByteBuffer.allocate(LENGTH_CEN).order(ByteOrder.LITTLE_ENDIAN);
        this.entries = new ArrayList<ZipEntry>();
        this.offset = 0L;
    }

    public ZipEntry read(InputStream is) throws IOException {
        return readLOC(is);
    }

    public ZipEntry readLOC(InputStream is) throws IOException {
        // Local file header (LOC)
        final int signature; // local file header signature
        final short version; // version needed to extract
        final short flags; // general purpose bit flag
        final short method; // compression method
        final short mtime; // last mod file time
        final short mdate; // last mod file date
        final int crc; // crc-32
        final int compsize; // compressed size
        final int uncompsize; // uncompressed size
        final short namelen; // file name length
        final short extlen; // extra field length
        final byte[] nameb; // file name
        // ---
        buffer.clear();
        buffer.limit(LENGTH_LOC);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() == 0)
            return null;
        buffer.rewind();
        signature = buffer.getInt();
        if (signature == SIGN_CEN || signature == SIGN_END)
            return null;
        if (signature != SIGN_LOC)
            throw new ZipException(format("invalid LOC header: signature=0x%X", signature));
        version = buffer.getShort();
        flags = buffer.getShort();
        method = buffer.getShort();
        mtime = buffer.getShort();
        mdate = buffer.getShort();
        crc = buffer.getInt();
        compsize = buffer.getInt();
        uncompsize = buffer.getInt();
        namelen = buffer.getShort();
        extlen = buffer.getShort();
        nameb = new byte[namelen];
        if (is.read(nameb) != namelen)
            throw new ZipException("invalid LOC header (name length)");
        if (extlen > 0)
            if (is.skip(extlen) != extlen)
                throw new ZipException("invalid LOC header (extra length)");
        ZipEntry entry = new ZipEntry();
        entry.signature = signature;
        entry.version = version;
        entry.flags = flags;
        entry.method = method;
        entry.mdate = mdate;
        entry.mtime = mtime;
        entry.crc = crc;
        entry.compsize = compsize;
        entry.uncompsize = uncompsize;
        entry.setName(nameb);
        return entry;
    }

    @SuppressWarnings("unused")
    public ZipEntry readCEN(InputStream is) throws IOException {
        // Central file header (CEN)
        final int signature; // central file header signature
        final short madever; // version made by
        final short needver; // version needed to extract
        final short flags; // general purpose bit flag
        final short method; // compression method
        final short mtime; // last mod file time
        final short mdate; // last mod file date
        final int crc; // crc-32
        final int compsize; // compressed size
        final int uncompsize; // uncompressed size
        final short namelen; // file name length
        final short extlen; // extra field length
        final short fcmlen; // file comment length
        final short dnum; // disk number start
        final short infattr; // internal file attributes
        final int exfattr; // external file attributes
        final int reloff; // relative offset of local header
        final byte[] nameb; // file name
        // ---
        buffer.clear();
        buffer.limit(LENGTH_CEN);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() == 0)
            return null;
        buffer.rewind();
        signature = buffer.getInt();
        if (signature == SIGN_END)
            return null;
        else if (signature != SIGN_CEN)
            throw new ZipException(format("invalid CEN header: signature=0x%X", signature));
        madever = buffer.getShort();
        needver = buffer.getShort();
        flags = buffer.getShort();
        method = buffer.getShort();
        mtime = buffer.getShort();
        mdate = buffer.getShort();
        crc = buffer.getInt();
        compsize = buffer.getInt();
        uncompsize = buffer.getInt();
        namelen = buffer.getShort();
        extlen = buffer.getShort();
        fcmlen = buffer.getShort();
        dnum = buffer.getShort();
        infattr = buffer.getShort();
        exfattr = buffer.getInt();
        reloff = buffer.getInt();
        nameb = new byte[namelen];
        if (is.read(nameb) != namelen)
            throw new ZipException("invalid LOC header (name length)");
        if (extlen > 0)
            if (is.skip(extlen) != extlen)
                throw new ZipException("invalid LOC header (extra length)");
        ZipEntry entry = new ZipEntry();
        entry.signature = signature;
        entry.flags = flags;
        entry.method = method;
        entry.mtime = mtime;
        entry.mdate = mdate;
        entry.crc = crc;
        entry.compsize = compsize;
        entry.uncompsize = uncompsize;
        entry.extlen = extlen;
        entry.reloff = reloff;
        entry.setName(nameb);
        return entry;
    }

    public ZipEndEntry readEND(InputStream is) throws IOException {
        // End of central dir header (END)
        final int signature; // end of central dir signature
        final short disknum; // number of this disk
        final short disknumCEN; // number of the disk with the start of the central directory
        final short countDiskCENs; // total number of entries in the central directory on this disk
        final short countCENs; // total number of entries in the central directory
        final int sizeCENs; // size of the central directory
        final int offsetStartCEN; // offset of start of central directory with respect to the starting disk number
        final short commentlen; // .ZIP file comment length
        final byte[] comment; // .ZIP file comment
        // ---
        buffer.clear();
        buffer.limit(LENGTH_END);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() == 0)
            return null;
        buffer.rewind();
        signature = buffer.getInt();
        if (signature != SIGN_END)
            return null;
        disknum = buffer.getShort();
        disknumCEN = buffer.getShort();
        countDiskCENs = buffer.getShort();
        countCENs = buffer.getShort();
        sizeCENs = buffer.getInt();
        offsetStartCEN = buffer.getInt();
        commentlen = buffer.getShort();
        comment = new byte[commentlen];
        if (commentlen > 0)
            if (is.skip(commentlen) != commentlen)
                throw new ZipException("invalid END header (comment length)");
        ZipEndEntry entry = new ZipEndEntry();
        entry.signature = signature;
        entry.disknum = disknum;
        entry.disknumCEN = disknumCEN;
        entry.countDiskCENs = countDiskCENs;
        entry.countCENs = countCENs;
        entry.sizeCENs = sizeCENs;
        entry.offsetStartCEN = offsetStartCEN;
        entry.commentlen = commentlen;
        entry.comment = comment;
        return entry;
    }

    public ZipEntry readEXT(InputStream is) throws IOException {
        ZipEntry entry = new ZipEntry();
        readEXT(is, entry);
        return entry;
    }

    public boolean readEXT(InputStream is, ZipEntry entry) throws IOException {
        // Extend header (EXT)
        final int signature; // extend header signature
        final int crc; // crc-32
        final int compsize; // compressed size
        final int uncompsize; // uncompressed size
        // ---
        buffer.clear();
        buffer.limit(LENGTH_EXT);
        Channels.newChannel(is).read(buffer);
        if (buffer.position() == 0)
            return false;
        buffer.rewind();
        signature = buffer.getInt();
        if (signature != SIGN_EXT)
            throw new ZipException(format("invalid EXT header: signature=0x%X", signature));
        crc = buffer.getInt();
        compsize = buffer.getInt();
        uncompsize = buffer.getInt();
        entry.extOverwritten = true;
        entry.crc = crc;
        entry.compsize = compsize;
        entry.uncompsize = uncompsize;
        return true;
    }

    public void write(OutputStream os, ZipEntry entry) throws IOException {
        writeLOC(os, entry);
    }

    public void writeLOC(OutputStream os, ZipEntry entry) throws IOException {
        // Local file header (LOC)
        final int signature; // local file header signature
        final short version; // version needed to extract
        final short flags; // general purpose bit flag
        final short method; // compression method
        final short mtime; // last mod file time
        final short mdate; // last mod file date
        final int crc; // crc-32
        final int compsize; // compressed size
        final int uncompsize; // uncompressed size
        final short namelen; // file name length
        final short extlen; // extra field length
        final byte[] nameb; // file name
        // ---
        final boolean hasEXT = entry.hasEXT();
        nameb = entry.getNameAsBytes();
        if (nameb.length > Short.MAX_VALUE)
            throw new ZipException("too long name: length=" + nameb.length);
        entry.reloff = (int)(offset & 0xFFFFFFFFL);
        signature = SIGN_LOC;
        version = entry.version;
        flags = entry.flags;
        method = entry.method;
        mtime = entry.mtime;
        mdate = entry.mdate;
        crc = hasEXT ? 0 : entry.crc;
        compsize = hasEXT ? 0 : entry.compsize;
        uncompsize = hasEXT ? 0 : entry.uncompsize;
        namelen = (short)(nameb.length & 0xFFFF);
        extlen = entry.extlen;
        assert offset <= 0xFFFFFFFFL;
        buffer.clear();
        buffer.putInt(signature);
        buffer.putShort(version);
        buffer.putShort(flags);
        buffer.putShort(method);
        buffer.putShort(mtime);
        buffer.putShort(mdate);
        buffer.putInt(crc);
        buffer.putInt(compsize);
        buffer.putInt(uncompsize);
        buffer.putShort(namelen);
        buffer.putShort(extlen);
        assert buffer.position() == LENGTH_LOC;
        buffer.flip();
        Channels.newChannel(os).write(buffer);
        os.write(nameb);
        os.flush();
        entries.add(entry);
        assert namelen > 0;
        assert extlen >= 0;
        assert compsize >= 0;
        offset += LENGTH_LOC + namelen + extlen + compsize;
    }

    public void writeCEN(OutputStream os, ZipEntry entry) throws IOException {
        // Central file header (CEN)
        final int signature; // central file header signature
        final short madever; // version made by
        final short needver; // version needed to extract
        final short flags; // general purpose bit flag
        final short method; // compression method
        final short mtime; // last mod file time
        final short mdate; // last mod file date
        final int crc; // crc-32
        final int compsize; // compressed size
        final int uncompsize; // uncompressed size
        final short namelen; // file name length
        final short extlen; // extra field length
        final short commlen; // file comment length
        final short disknum; // disk number start
        final short inattr; // internal file attributes
        final int exattr; // external file attributes
        final int reloff; // relative offset of local header
        final byte[] nameb; // file name
        // ---
        signature = SIGN_CEN;
        madever = entry.version;
        needver = entry.version;
        flags = entry.flags;
        method = entry.method;
        mtime = entry.mtime;
        mdate = entry.mdate;
        crc = entry.crc;
        compsize = entry.compsize;
        uncompsize = entry.uncompsize;
        nameb = entry.getNameAsBytes();
        if (nameb.length > Short.MAX_VALUE)
            throw new ZipException("too long name: length=" + nameb.length);
        namelen = (short)nameb.length;
        extlen = 0; // not support
        commlen = 0; // not support
        disknum = 0; // not support
        inattr = 0; // not support
        exattr = 0; // not support
        reloff = entry.reloff;
        buffer.clear();
        buffer.putInt(signature);
        buffer.putShort(madever);
        buffer.putShort(needver);
        buffer.putShort(flags);
        buffer.putShort(method);
        buffer.putShort(mtime);
        buffer.putShort(mdate);
        buffer.putInt(crc);
        buffer.putInt(compsize);
        buffer.putInt(uncompsize);
        buffer.putShort(namelen);
        buffer.putShort(extlen);
        buffer.putShort(commlen);
        buffer.putShort(disknum);
        buffer.putShort(inattr);
        buffer.putInt(exattr);
        buffer.putInt(reloff);
        assert buffer.position() == LENGTH_CEN;
        buffer.flip();
        Channels.newChannel(os).write(buffer);
        os.write(nameb);
        os.flush();
        offset += LENGTH_CEN + namelen + extlen;
    }

    public void writeEND(OutputStream os) throws IOException {
        for (final ZipEntry entry : entries)
            writeCEN(os, entry);
        writeEND(os, ZipEndEntry.create(offset, entries));
    }

    public void writeEND(OutputStream os, ZipEndEntry entry) throws IOException {
        // End of central dir header (END)
        final int signature; // end of central dir signature
        final short disknum; // number of this disk
        final short disknumCEN; // number of the disk with the start of the central directory
        final short countDiskCENs; // total number of entries in the central directory on this disk
        final short countCENs; // total number of entries in the central directory
        final int sizeCENs; // size of the central directory
        final int offsetStartCEN; // offset of start of central directory with respect to the starting disk number
        final short commentlen; // .ZIP file comment length
        final byte[] comment; // .ZIP file comment
        // ---
        signature = entry.signature;
        disknum = entry.disknum;
        disknumCEN = entry.disknumCEN;
        countDiskCENs = entry.countDiskCENs;
        countCENs = entry.countCENs;
        sizeCENs = entry.sizeCENs;
        offsetStartCEN = entry.offsetStartCEN;
        commentlen = 0;
        comment = new byte[0];
        buffer.clear();
        buffer.putInt(signature);
        buffer.putShort(disknum);
        buffer.putShort(disknumCEN);
        buffer.putShort(countDiskCENs);
        buffer.putShort(countCENs);
        buffer.putInt(sizeCENs);
        buffer.putInt(offsetStartCEN);
        buffer.putShort(commentlen);
        buffer.put(comment);
        assert buffer.position() == LENGTH_END;
        buffer.flip();
        Channels.newChannel(os).write(buffer);
        os.flush();
        offset += LENGTH_END;
    }

    public void writeEXT(OutputStream os, ZipEntry entry) throws IOException {
        // Extend header (EXT)
        final int signature; // extend header signature
        final int crc; // crc-32
        final int compsize; // compressed size
        final int uncompsize; // uncompressed size
        // ---
        signature = SIGN_EXT;
        crc = entry.crc;
        compsize = entry.compsize;
        uncompsize = entry.uncompsize;
        buffer.clear();
        buffer.putInt(signature);
        buffer.putInt(crc);
        buffer.putInt(compsize);
        buffer.putInt(uncompsize);
        assert buffer.position() == LENGTH_EXT;
        buffer.flip();
        Channels.newChannel(os).write(buffer);
        os.flush();
        offset += LENGTH_EXT;
        assert entry.compsize >= 0;
        offset += entry.compsize;
    }

    public void clearEntries() {
        entries.clear();
        offset = 0;
    }

}
