package jp.sfjp.armadillo.compression.lzhuf;

import java.util.*;
import jp.sfjp.armadillo.archive.lzh.*;

/**
 * A huffman table for LZHUF.
 */
public final class LzhHuffmanTable {

    private static final int MAX_CODE_LENGTH = 16;

    final int[] codeTable;
    final int[] codeLengthTable;

    private LzhHuffmanTable(int[] frequencyTable) {
        int tableSize = frequencyTable.length;
        for (int i = tableSize - 1; i >= 0; i--) {
            if (frequencyTable[i] > 0)
                break;
            --tableSize;
        }
        int[] ct = new int[tableSize];
        int[] lt = new int[tableSize];
        build(frequencyTable, tableSize, ct, lt);
        this.codeTable = ct;
        this.codeLengthTable = lt;
    }

    public static LzhHuffmanTable build(int[] frequencyTable) {
        return new LzhHuffmanTable(frequencyTable);
    }

    /**
     * Builds tables.
     * @param frequencyTable
     * @param tableSize
     * @param ct code table
     * @param lt code length table
     */
    private void build(int[] frequencyTable, int tableSize, int[] ct, int[] lt) {
        // create node list
        LinkedList<Node> q = new LinkedList<Node>();
        for (int i = 0; i < tableSize; i++) {
            final int frequency = frequencyTable[i];
            if (frequency > 0)
                q.add(new Node(i, frequency));
        }
        if (q.size() < 2) {
            // queue size = ( 0, 1, 2 )
            int queueSize = q.size();
            if (queueSize == 1) {
                // queue size = ( 1 )
                lt[tableSize - 1] = 1;
            }
            else if (queueSize == 2) {
                // queue size = ( 2 )
                Node node1 = q.get(0);
                Node node2 = q.get(1);
                int[] numbers = {node1.number, node2.number};
                Arrays.sort(numbers);
                ct[numbers[0]] = 0;
                lt[numbers[0]] = 1;
                ct[numbers[1]] = 1;
                lt[numbers[1]] = 1;
            }
            return;
        }
        // create tree
        int number = tableSize;
        while (q.size() > 1) {
            Collections.sort(q);
            Node node1 = q.remove(0);
            Node node2 = q.remove(0);
            q.add(new Node(number++, node1, node2));
        }
        Node root = q.get(0);
        // create tables
        createCodeLengthTable(root, tableSize, 0, lt);
        createCodeTable(ct, lt);
    }

    /**
     * Creates code length table.
     * It traverses huffman tree recursively.
     * @param node node of huffman tree
     * @param size table size
     * @param codeLength
     * @param lt (output) code length table
     */
    private static void createCodeLengthTable(Node node, int size, int codeLength, int[] lt) {
        if (codeLength > 16)
            throw new LzhQuit("code length = " + codeLength);
        final int number = node.number;
        if (number < size)
            lt[number] = codeLength;
        else {
            createCodeLengthTable(node.left, size, codeLength + 1, lt);
            createCodeLengthTable(node.right, size, codeLength + 1, lt);
        }
    }

    /**
     * Creates code table from code length table.
     * @param lt code length table
     * @return ct code table
     */
    static int[] createCodeTable(int[] lt) {
        int[] ct = new int[lt.length];
        createCodeTable(ct, lt);
        return ct;
    }

    /**
     * Creates code table from code length table.
     * @param ct (output) code length table
     * @param lt code length table
     */
    static void createCodeTable(int[] ct, int[] lt) {
        final int workSize = MAX_CODE_LENGTH + 1;
        int[] counts = new int[workSize];
        for (int i = 0; i < lt.length; i++)
            ++counts[lt[i]];
        int[] baseCodes = new int[workSize];
        for (int i = 0; i < MAX_CODE_LENGTH; i++)
            baseCodes[i + 1] = baseCodes[i] + counts[i + 1] << 1;
        assert baseCodes[MAX_CODE_LENGTH] == 1 << workSize : baseCodes[MAX_CODE_LENGTH];
        for (int i = 0; i < ct.length; i++) {
            final int codeLength = lt[i];
            if (codeLength > 0)
                ct[i] = baseCodes[codeLength - 1]++;
        }
    }

    private static final class Node implements Comparable<Node> {

        final int number;
        final int weight;

        Node left;
        Node right;

        public Node(int number, int weight) {
            this.number = number;
            this.weight = weight;
            this.left = null;
            this.right = null;
        }

        public Node(int number, Node node1, Node node2) {
            this.number = number;
            this.weight = node1.weight + node2.weight;
            this.left = node1;
            this.right = node2;
        }

        @Override
        public int compareTo(Node anotherNode) {
            int tn;
            int an;
            if (anotherNode.weight == this.weight) {
                tn = anotherNode.number;
                an = this.number;
            }
            else {
                tn = this.weight;
                an = anotherNode.weight;
            }
            return tn < an ? -1 : (tn == an ? 0 : 1);
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + ((left == null) ? 0 : left.hashCode());
            result = prime * result + number;
            result = prime * result + ((right == null) ? 0 : right.hashCode());
            result = prime * result + weight;
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            Node other = (Node)obj;
            if (left == null) {
                if (other.left != null)
                    return false;
            }
            else if (!left.equals(other.left))
                return false;
            if (number != other.number)
                return false;
            if (right == null) {
                if (other.right != null)
                    return false;
            }
            else if (!right.equals(other.right))
                return false;
            if (weight != other.weight)
                return false;
            return true;
        }

        @Override
        public String toString() {
            return "Node:" + number + " (" + weight + ")";
        }

    }

}
