package jp.sfjp.armadillo.compression.lzhuf;

import java.io.*;

public final class LzssInputStream extends FilterInputStream {

    private final int dictionarySize;
    private final int matchSize;
    private final int threshold;
    private final int bufferLength;

    private boolean closed;
    private LzssDecoderReadable input;
    private byte[] buffer;
    private int bufferedSize = 0;
    private int index = 0;
    private int limit = 0;

    public LzssInputStream(LzssDecoderReadable input,
                           int dictionarySize,
                           int matchSize,
                           int threshold) {
        super(null);
        this.input = input;
        this.dictionarySize = dictionarySize;
        this.matchSize = matchSize;
        this.threshold = threshold;
        this.bufferLength = dictionarySize + matchSize;
        this.buffer = new byte[bufferLength];
    }

    @Override
    public int read() throws IOException {
        if (closed)
            throw new IOException("stream already closed");
        try {
            if (bufferedSize < 1)
                if (fillBuffer(1) > 0)
                    return -1;
            --bufferedSize;
            return buffer[++index] & 0xFF;
        }
        catch (RuntimeException ex) {
            throw new LzhufException("decode error", ex);
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        if (closed)
            throw new IOException("stream already closed");
        try {
            int offset = off;
            int remaining = len;
            while (remaining > 0) {
                if (remaining > bufferedSize && fillBuffer(remaining) == remaining)
                    break;
                int length = (remaining <= bufferedSize) ? remaining : bufferedSize;
                if (index < limit) {
                    System.arraycopy(buffer, index, b, offset, length);
                    index += length;
                }
                else {
                    int length1 = (index + length > buffer.length) ? buffer.length - index : length;
                    System.arraycopy(buffer, index, b, offset, length1);
                    index += length1;
                    int length2 = length - length1;
                    if (length2 > 0) {
                        System.arraycopy(buffer, 0, b, offset + length1, length2);
                        index = length2;
                    }
                }
                remaining -= length;
                bufferedSize -= length;
            }
            return (len == remaining) ? len : len - remaining;
        }
        catch (RuntimeException ex) {
            throw new LzhufException("decode error", ex);
        }
    }

    private int fillBuffer(int requiredSize) throws IOException {
        final int fill = Math.min(requiredSize, dictionarySize);
        while (bufferedSize < fill) {
            int code = input.read();
            if (code == -1) {
                break;
            }
            if (code >= 0x100) {
                int length = (code & 0xFF) + threshold;
                int offset = input.readOffset();
                assert length <= matchSize;
                int srcIndex = limit - offset - 1;
                if (srcIndex < 0)
                    srcIndex += bufferLength;
                for (int i = 0; i < length; i++) {
                    buffer[limit++] = buffer[srcIndex++];
                    if (srcIndex >= bufferLength)
                        srcIndex -= bufferLength;
                    if (limit >= bufferLength)
                        limit -= bufferLength;
                }
                bufferedSize += length;
            }
            else {
                buffer[limit++] = (byte)(code & 0xFF);
                if (limit >= bufferLength)
                    limit -= bufferLength;
                ++bufferedSize;
            }
        }
        return requiredSize - bufferedSize;
    }

    @Override
    public long skip(long n) throws IOException {
        long remaining = n;
        byte[] bytes = new byte[8192];
        while (remaining > 0) {
            long requiredLength = Math.min(bytes.length, remaining);
            assert requiredLength <= Integer.MAX_VALUE;
            remaining -= read(bytes, 0, (int)requiredLength);
        }
        return n - remaining;
    }

    @Override
    public void close() throws IOException {
        if (closed)
            throw new IOException("stream already closed");
        input.close();
    }

}
