package jp.sfjp.armadillo.io;

import java.io.*;

/**
 * A bit-by-bit readable InputStream.
 */
public final class BitInputStream extends FilterInputStream {

    private static final int INT_BITSIZE = 32;
    private static final int INPUT_BITSIZE = 8;

    private boolean closed;
    private int buffer;
    private int remaining;
    private boolean eof;

    public BitInputStream(InputStream is) {
        super(is);
        this.closed = false;
        this.buffer = 0;
        this.remaining = 0;
        this.eof = false;
    }

    private void ensureOpen() throws IOException {
        if (closed)
            throw new IOException("stream closed");
    }

    /**
     * Reads the bits that specified length.
     * Reads bits of int from the lower bit,
     * and writes it into the higher bit of this stream.
     * @param b int value
     * @param bitLength bit length to write
     * @throws IOException
     * @throws IllegalArgumentException
     */
    public int readBits(int bitLength) throws IOException {
        ensureOpen();
        if (bitLength < 1 || 32 < bitLength)
            throw new IllegalArgumentException("bit length: " + bitLength);
        if (remaining < bitLength) {
            fillBuffer(bitLength);
            if (remaining == 0)
                return -1;
            else if (remaining < bitLength)
                throw new IOException("requred=" + bitLength + ", remaining=" + remaining);
        }
        int value = buffer >>> (INT_BITSIZE - bitLength);
        remaining -= bitLength;
        buffer <<= bitLength;
        return value;
    }

    /**
     * Reads a bit.
     * @return a bit value, or <code>-1</code> if stream has already reached EOF
     * @throws IOException
     */
    public int readBit() throws IOException {
        return readBits(1);
    }

    /**
     * Prefetches bits from InputStream.
     * @param bitLength bit length to read
     * @return a value of bits, or <code>-1</code> if stream has already reached EOF
     * @throws IOException 
     */
    public int prefetchBits(int bitLength) throws IOException {
        ensureOpen();
        if (bitLength > INT_BITSIZE)
            throw new IllegalArgumentException("overflow: " + bitLength);
        if (remaining < bitLength)
            fillBuffer(bitLength);
        return buffer >>> (INT_BITSIZE - bitLength);
    }

    /**
     * Prefetches bits from InputStream.
     * @return an octet, or <code>-1</code> if stream has already reached EOF
     * @throws IOException
     */
    public int prefetch() throws IOException {
        return prefetchBits(INPUT_BITSIZE);
    }

    /**
     * Fills the buffer.
     * When the buffer has already filled enough, it does nothing.
     * @param requiredSize a bit size to need, not a fill size
     * @throws IOException
     */
    private void fillBuffer(int requiredSize) throws IOException {
        assert requiredSize >= 1 && requiredSize <= INT_BITSIZE;
        while (!eof && remaining < requiredSize) {
            int b = super.read();
            if (b == -1) {
                eof = true;
                return;
            }
            b <<= (INT_BITSIZE - INPUT_BITSIZE - remaining);
            assert (buffer | b) == buffer + b;
            buffer |= b;
            remaining += INPUT_BITSIZE;
        }
    }

    /**
     * Clears buffer.
     */
    public void clearBuffer() {
        this.buffer = 0;
        this.remaining = 0;
    }

    /**
     * Gets the value of buffer.
     * @return
     */
    public int getBuffer() {
        return buffer;
    }

    /**
     * Gets the number of available bits.
     * @return
     */
    public int getRemaining() {
        return remaining;
    }

    /**
     * Tests whether this stream has already reached EOF or not.
     * @return <code>true</code> if this stream has already reached EOF, otherwise <code>false</code>
     */
    public boolean isEOF() {
        return eof;
    }

    @Override
    public int read() throws IOException {
        return readBits(8);
    }

    @Override
    public void close() throws IOException {
        ensureOpen();
        closed = true;
        super.close();
    }

}
