package jp.sfjp.armadillo.io;

public final class ByteQueue {

    private static final int DEFAULT_CAPACITY = 8192;

    private int p;
    private int q; // len
    private byte[] buffer;

    public ByteQueue() {
        this(DEFAULT_CAPACITY);
    }

    public ByteQueue(int initialiCapacity) {
        this.p = 0;
        this.q = 0;
        this.buffer = new byte[initialiCapacity];
    }

    public int size() {
        return this.q;
    }

    public void enqueue(byte[] bytes) {
        enqueue(bytes, 0, bytes.length);
    }

    public void enqueue(byte[] bytes, int offset, int length) {
        final int requiredLength = buffer.length - q;
        if (length > requiredLength)
            if (!ensureBuffer(length))
                throw new IllegalStateException("failed to ensure buffer");
        System.arraycopy(bytes, offset, buffer, p + q, length);
        q += length;
    }

    public byte[] dequeue(int length) {
        byte[] bytes = new byte[length];
        dequeue(bytes, 0, length);
        return bytes;
    }

    public void dequeue(byte[] bytes) {
        dequeue(bytes, 0, bytes.length);
    }

    public void dequeue(byte[] bytes, int offset, int length) {
        if (q < length)
            throw new IllegalStateException(String.format("required=%d, qsize=%d", length, q));
        trimHead();
        System.arraycopy(buffer, p, bytes, 0, length);
        p += length;
        q -= length;
    }

    public boolean ensureBuffer(int length) {
        final int oldLength = buffer.length;
        final int newSize;
        if (length > DEFAULT_CAPACITY)
            newSize = oldLength + (length / DEFAULT_CAPACITY + 1) * DEFAULT_CAPACITY;
        else
            newSize = oldLength + 8192;
        byte[] bytes = new byte[newSize];
        System.arraycopy(buffer, 0, bytes, 0, oldLength);
        buffer = bytes;
        return buffer.length >= length + q;
    }

    public void trimHead() {
        if (p == 0)
            return;
        final int size = q - p;
        if (size < DEFAULT_CAPACITY / 2 && buffer.length > DEFAULT_CAPACITY * 4) {
            byte[] bytes = new byte[DEFAULT_CAPACITY];
            System.arraycopy(buffer, p, bytes, 0, size);
            buffer = bytes;
        }
        else {
            byte[] bytes = new byte[size];
            System.arraycopy(buffer, p, bytes, 0, size);
            System.arraycopy(bytes, 0, buffer, 0, size);
        }
        p = 0;
        q = size;
    }

}
