package jp.sfjp.armadillo.io;

import java.io.*;

public final class RewindableInputStream extends PushbackInputStream {

    private byte[] buffer;
    private int limit;

    public RewindableInputStream(InputStream is, int size) {
        super(is, size);
        this.buffer = new byte[size];
        this.limit = 0;
    }

    public int rewind(int length) throws IOException {
        if (limit < length)
            throw new IOException("length=" + length + ", but limit=" + limit);
        int offset = limit - length;
        unread(buffer, offset, length);
        limit = offset;
        return length;
    }

    private void add(byte[] bytes, int offset, int length) {
        if (length > buffer.length) {
            limit = length;
            buffer = new byte[length];
            System.arraycopy(bytes, 0, buffer, 0, length);
        }
        else {
            int rest = limit + length - buffer.length;
            if (rest > 0) {
                limit -= rest;
                System.arraycopy(buffer, rest, buffer, 0, limit);
            }
            System.arraycopy(bytes, offset, buffer, limit, length);
            limit += length;
        }
    }

    @Override
    public int read() throws IOException {
        int read = super.read();
        if (read != -1)
            add(new byte[]{(byte)(read & 0xFF)}, 0, 1);
        return read;
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        int readLength = super.read(b, off, len);
        assert readLength != 0 : "Read Zero";
        if (readLength > 0)
            add(b, off, readLength);
        return readLength;
    }

    @Override
    public long skip(long n) throws IOException {
        long skipped = 0;
        long length = n;
        if (length >= buffer.length) {
            if (length > buffer.length) {
                length -= buffer.length;
                skipped += super.skip(length);
            }
            length = buffer.length;
        }
        assert length <= Integer.MAX_VALUE;
        int rest = (int)length;
        skipped += read(new byte[rest]);
        return skipped;
    }

    @Override
    public int available() throws IOException {
        return buffer.length;
    }

}
