#ifndef __COMMON_H__
#define __COMMON_H__

#if defined(__BORLANDC__) || defined(_MSC_VER)
#include <windows.h>	// WINVER is defined
#endif
#include <stdio.h>
#include <stdarg.h>
#include <tchar.h>
#include <math.h>

#include <complex>
#include <vector>
#include <list>
#include <map>
#include <set>
#include <memory>
#include <algorithm>

extern "C" {
#include "SFMT.h"
}

#define NUMBEROF(x) (sizeof(x) / sizeof(x[0]))
#define foreach(T, i, c) for (T::iterator i = (c).begin(); i != (c).end(); i++)
#define foreach_const(T, i, c) for (T::const_iterator i = (c).begin(); i != (c).end(); i++)
#define foreach_reverse(T, i, c) for (T::reverse_iterator i = (c).rbegin(); i != (c).rend(); i++)
#define foreach_const_reverse(T, i, c) for (T::const_reverse_iterator i = (c).rbegin(); i != (c).rend(); i++)

#ifdef DEBUG_ASCRIPT
#define DBG(x) x
#define DBGPARSER(x) x
#define ASSUME(env, x) if (!(x)) { env.Error(__FILE__, __LINE__, #x); }
#else
#define DBG(x)
#define DBGPARSER(x)
#define ASSUME(env, x) if (!(x)) { env.Error(__FILE__, __LINE__, #x); }
#endif

#define ERROREND(env, str) (env).Error(__FILE__, __LINE__, (str));

#define AScript_Symbol(name) (SymbolPool::GetInstance()->_pSymbol_##name)

#define AScript_DeclareSymbol(name) \
const Symbol *_pSymbol_##name

#define AScript_RealizeSymbol(name) \
_pSymbol_##name = Symbol::Add(#name);

#define AScript_RealizeSymbolEx(name, str) \
_pSymbol_##name = Symbol::Add(str);

typedef _TCHAR TCHAR;

namespace AScript {

extern const int MAX_STACK_LEVEL;

//-----------------------------------------------------------------------------
// Simple type declarations
//-----------------------------------------------------------------------------
class Symbol;
class SymbolSet;
class Value;
class ValueList;
class ValueMap;
class Function;
class Environment;

typedef double Number;
typedef std::complex<Number> Complex;
typedef std::basic_string<TCHAR> String;

enum ErrorType {
	ERR_None,
	ERR_SyntaxError,
	ERR_ArithmeticError,
	ERR_TypeError,
	ERR_ZeroDivisionError,
	ERR_ValueError,
	ERR_SystemError,
	ERR_IOError,
	ERR_IndexError,
	ERR_KeyError,
	ERR_ImportError,
	ERR_AttributeError,
};

enum FunctionType {
	FUNCTYPE_Function,
	FUNCTYPE_Class,
	FUNCTYPE_Instance,
	FUNCTYPE_Block,
};

enum OccurPattern {
	OCCUR_Invalid,
	OCCUR_Zero,			// (none)
	OCCUR_Once,			// 1
	OCCUR_ZeroOrOnce,	// ?
	OCCUR_ZeroOrMore,	// *
	OCCUR_OnceOrMore,	// +
};

enum ValueType {
	// this order affects on the function of Value::Compare()
	VTYPE_Invalid,
	VTYPE_Symbol,
	VTYPE_Boolean,
	VTYPE_Number,
	VTYPE_Complex,
	VTYPE_String,
	VTYPE_Module,
	VTYPE_Class,
	VTYPE_Object,
	VTYPE_Function,		// specific type of Object
	VTYPE_List,			// specific type of Object
	VTYPE_Dict,			// specific type of Object
	VTYPE_File,			// specific type of Object
	VTYPE_FileStat,		// specific type of Object
	VTYPE_DateTime,		// specific type of Object
	VTYPE_Expr,			// specific type of Object
	VTYPE_Environment,	// specific type of Object
	VTYPE_Error,		// specific type of Object
	VTYPE_Struct,		// specific type of Object
	VTYPE_Quote,		// this type is used in declaration
	VTYPE_AnyType,		// this type is used in declaration
};

enum SignalType {
	SIGTYPE_None,
	SIGTYPE_Error,
	SIGTYPE_Break,
	SIGTYPE_Continue,
	SIGTYPE_Return,
};

enum EnvType {
	ENVTYPE_Invalid,
	ENVTYPE_Root,
	ENVTYPE_Local,
	ENVTYPE_Block,
	ENVTYPE_Module,
	ENVTYPE_Class,
	ENVTYPE_Instance,
	ENVTYPE_Method,
	ENVTYPE_Lister,
	ENVTYPE_Outer,	// this type is not set as type of each frame
};

enum ResultMode {
	RSLTMODE_Normal,
	RSLTMODE_List,	// :list
	RSLTMODE_XList,	// :xlist
	RSLTMODE_Set,	// :set
	RSLTMODE_XSet,	// :xset
	RSLTMODE_Void,	// :void
};

enum MapFlag {
	MAP_Off,
	MAP_On,
};

const Symbol *GetValueTypeSymbol(ValueType valType);
const Symbol *GetOccurPatternSymbol(OccurPattern occurPattern);
const TCHAR *GetSignalTypeName(SignalType sigType);
const TCHAR *GetEnvTypeName(EnvType envType);
const TCHAR *GetErrorTypeName(ErrorType errType);

//void InitModule(Environment &env);

//-----------------------------------------------------------------------------
// String operations
//-----------------------------------------------------------------------------
int GetEscaped(int ch);
const TCHAR *FindString(const TCHAR *str, const TCHAR *sub, bool ignoreCaseFlag);

String EscapeString(const TCHAR *str);
String Capitalize(const TCHAR *str);
String Lower(const TCHAR *str);
String Upper(const TCHAR *str);
String Strip(const TCHAR *str, const SymbolSet &attrs);
String Left(const TCHAR *str, size_t len);
String Right(const TCHAR *str, size_t len);
String Middle(const TCHAR *str, int start, int len);
String Join(const ValueList &valList, const TCHAR *str);
String Replace(const TCHAR *str, const TCHAR *sub, const TCHAR *replace,
									int nMaxReplace, const SymbolSet &attrs);

inline bool IsSJISFirst(int ch) {
	return 0x81 <= ch && ch <= 0x9f || 0xe0 <= ch && ch <= 0xef;
}

inline bool IsSJISSecond(int ch) {
	return 0x40 <= ch && ch <= 0x7e || 0x80 <= ch && ch <= 0xfc;
}

inline bool IsDigit(int ch) {
	return _T('0') <= ch && ch <= _T('9');
}

inline bool IsHexDigit(int ch) {
	return
		_T('0') <= ch && ch <= _T('9') ||
		_T('A') <= ch && ch <= _T('F') ||
		_T('a') <= ch && ch <= _T('f');
}

inline int ConvHexDigit(int ch) {
	return
		(_T('0') <= ch && ch <= _T('9'))? ch - _T('0') :
		(_T('A') <= ch && ch <= _T('F'))? ch - _T('A') + 10 :
		(_T('a') <= ch && ch <= _T('f'))? ch - _T('a') + 10 : 0;
}

inline bool IsOctDigit(int ch) {
	return _T('0') <= ch && ch <= _T('7');
}

inline int ConvOctDigit(int ch) {
	return (_T('0') <= ch && ch <= _T('7'))? ch - _T('0') : 0;
}

inline bool IsWhite(int ch) {
	return ch == _T(' ') || ch == _T('\t');
}

inline bool IsSpace(int ch) {
	return ch == _T(' ') || ch == _T('\t') || ch == _T('\r') || ch == _T('\n');
}

inline bool IsSymbolFirstChar(int ch) {
	return
		(_T('a') <= ch && ch <= _T('z')) ||
		(_T('A') <= ch && ch <= _T('Z')) ||
		ch == _T('_') || ch == _T('$') || ch == _T('@') || IsSJISFirst(ch);
}

inline bool IsSymbolChar(int ch) {
	return
		(_T('a') <= ch && ch <= _T('z')) ||
		(_T('A') <= ch && ch <= _T('Z')) ||
		ch == _T('_') || ch == _T('$') || (_T('0') <= ch && ch <= _T('9'));
}

inline TCHAR ToUpper(TCHAR ch) {
	return (_T('a') <= ch && ch <= _T('z'))? ch - _T('a') + _T('A') : ch;
}

inline TCHAR ToLower(TCHAR ch) {
	return (_T('A') <= ch && ch <= _T('Z'))? ch - _T('A') + _T('a') : ch;
}

inline int CompareChar(TCHAR ch1, TCHAR ch2, bool ignoreCaseFlag) {
	return ignoreCaseFlag?
		static_cast<int>(ToUpper(ch1)) - static_cast<int>(ToUpper(ch2)) :
		static_cast<int>(ch1) - static_cast<int>(ch2);
}

#if defined(WINVER)
inline bool IsPathSeparator(TCHAR ch) { return ch == _T(';'); }
#else
inline bool IsPathSeparator(TCHAR ch) { return ch == _T(';') || ch == _T(':'); }
#endif

//-----------------------------------------------------------------------------
// StringList
//-----------------------------------------------------------------------------
class StringList : public std::vector<String> {
public:
	static const StringList Null;
	inline StringList() {}
private:
	inline StringList(const StringList &stringList) {}
	inline void operator=(const StringList &stringList) {}
};

//-----------------------------------------------------------------------------
// RandomNumberGenerator
//-----------------------------------------------------------------------------
class RandomNumberGenerator {
public:
	inline int operator()(int n) {
		return static_cast<int>(::genrand_real2() * n);
	}
};

//-----------------------------------------------------------------------------
// Signal
//-----------------------------------------------------------------------------
class Signal {
private:
	struct Message {
		SignalType sigType;
		ErrorType errType;
		ErrorType errTypeSaved;
		String str;
		String strSaved;
		std::auto_ptr<Value> pValue;
		Message();
	};
private:
	Message *_pMsg;
	int _stackLevel;
public:
	Signal();
	Signal(const Signal &sig);
	inline bool IsSignalled() const	{ return _pMsg->sigType != SIGTYPE_None; }
	inline bool IsError() const		{ return _pMsg->sigType == SIGTYPE_Error; }
	inline bool IsBreak() const		{ return _pMsg->sigType == SIGTYPE_Break; }
	inline bool IsContinue() const	{ return _pMsg->sigType == SIGTYPE_Continue; }
	inline bool IsReturn() const	{ return _pMsg->sigType == SIGTYPE_Return; }
	inline const TCHAR *GetSignalName() const { return GetSignalTypeName(_pMsg->sigType); }
	inline const TCHAR *GetErrorName() const { return GetErrorTypeName(_pMsg->errType); }
	inline const TCHAR *GetSavedString() const { return _pMsg->strSaved.c_str(); }
	inline Value &GetValue() const { return *_pMsg->pValue; }
	inline void ClearSignal() {
		_pMsg->sigType = SIGTYPE_None, _pMsg->errType = ERR_None;
	}
	inline void SetSignal(SignalType sigType, const Value &value) {
		_pMsg->sigType = sigType, *_pMsg->pValue = value;
	}
	inline void SaveError() {
		_pMsg->errTypeSaved = _pMsg->errType, _pMsg->strSaved = _pMsg->str;
	}
	String GetErrString() const;
	inline GetSavedErrorType() const { return _pMsg->errTypeSaved; }
	void SetError(ErrorType errType, const TCHAR *format, ...);
	void SetErrorV(ErrorType errType, const TCHAR *format, va_list list);
};

//-----------------------------------------------------------------------------
// Symbol
//-----------------------------------------------------------------------------
class Symbol {
public:
	typedef unsigned int UniqNumber;
	struct KeyCompare_UniqNumber {
		inline bool operator()(const Symbol *pSymbol1, const Symbol *pSymbol2) const {
			return pSymbol1->GetUniqNum() < pSymbol2->GetUniqNum();
		}
	};
	struct KeyCompare_Name {
		inline bool operator()(const Symbol *pSymbol1, const Symbol *pSymbol2) const {
			return ::_tcscmp(pSymbol1->GetName(), pSymbol2->GetName()) < 0;
		}
	};
private:
	UniqNumber _uniqNum;
	TCHAR *_name;
public:
	Symbol(UniqNumber uniqNum, const TCHAR *name);
	~Symbol();
	bool IsFlowControlSymbol() const;
	inline static void Delete(Symbol *pSymbol) { delete pSymbol; }
	inline UniqNumber GetUniqNum() const { return _uniqNum; }
	inline const TCHAR *GetName() const { return _name; }
	inline bool IsIdentical(const Symbol *pSymbol) const {
		return GetUniqNum() == pSymbol->GetUniqNum();
	}
	inline bool IsNone() const { return _name[0] = _T('\0'); }
	static const Symbol *Add(const TCHAR *name);
};

//-----------------------------------------------------------------------------
// SymbolSet
//-----------------------------------------------------------------------------
class SymbolSet : public std::set<const Symbol *, Symbol::KeyCompare_UniqNumber> {
public:
	static const SymbolSet Null;
public:
	inline SymbolSet() {}
	SymbolSet(const SymbolSet &symbolSet);
	void operator=(const SymbolSet &symbolSet);
	inline bool IsSet(const Symbol *pSymbol) const {
		return find(pSymbol) != const_cast<SymbolSet *>(this)->end();
	}
	inline void Insert(const Symbol *pSymbol) {
		insert(pSymbol);
	}
};

//-----------------------------------------------------------------------------
// SymbolPool
//-----------------------------------------------------------------------------
class SymbolPool {
public:
	AScript_DeclareSymbol(Str_Empty);
	AScript_DeclareSymbol(Char_Plus);
	AScript_DeclareSymbol(Char_Multiply);
	AScript_DeclareSymbol(Char_Question);
	AScript_DeclareSymbol(Char_Modulo);
	AScript_DeclareSymbol(Char_And);
	AScript_DeclareSymbol(unknown);
	AScript_DeclareSymbol(number);
	AScript_DeclareSymbol(boolean);
	AScript_DeclareSymbol(symbol);
	AScript_DeclareSymbol(string);
	AScript_DeclareSymbol(complex);
	AScript_DeclareSymbol(Module);
	AScript_DeclareSymbol(Class);
	AScript_DeclareSymbol(Object);
	AScript_DeclareSymbol(Function);
	AScript_DeclareSymbol(List);
	AScript_DeclareSymbol(Dict);
	AScript_DeclareSymbol(File);
	AScript_DeclareSymbol(FileStat);
	AScript_DeclareSymbol(DateTime);
	AScript_DeclareSymbol(Expr);
	AScript_DeclareSymbol(Environment);
	AScript_DeclareSymbol(Error);
	AScript_DeclareSymbol(Struct);
	AScript_DeclareSymbol(e);
	AScript_DeclareSymbol(pi);
	AScript_DeclareSymbol(nil);
	AScript_DeclareSymbol(zero);
	AScript_DeclareSymbol(raise);
	AScript_DeclareSymbol(true_);
	AScript_DeclareSymbol(false_);
	AScript_DeclareSymbol(rem);			// dummy for MS-DOS batch
	AScript_DeclareSymbol(j);
	AScript_DeclareSymbol(if_);
	AScript_DeclareSymbol(elsif);
	AScript_DeclareSymbol(else_);
	AScript_DeclareSymbol(repeat);
	AScript_DeclareSymbol(while_);
	AScript_DeclareSymbol(for_);
	AScript_DeclareSymbol(break_);
	AScript_DeclareSymbol(continue_);
	AScript_DeclareSymbol(except_);
	AScript_DeclareSymbol(__init__);
	AScript_DeclareSymbol(__del__);
	AScript_DeclareSymbol(__str__);
	AScript_DeclareSymbol(super);
	AScript_DeclareSymbol(self);
	AScript_DeclareSymbol(static_);
	AScript_DeclareSymbol(const_);
	AScript_DeclareSymbol(_anonymous_);
	AScript_DeclareSymbol(public_);
	AScript_DeclareSymbol(private_);
	AScript_DeclareSymbol(protected_);
	AScript_DeclareSymbol(dynamic_scope);
	AScript_DeclareSymbol(inside_scope);
	AScript_DeclareSymbol(map);
	AScript_DeclareSymbol(nomap);
	AScript_DeclareSymbol(strict);
	AScript_DeclareSymbol(loose);
	AScript_DeclareSymbol(block);
	AScript_DeclareSymbol(list);
	AScript_DeclareSymbol(xlist);
	AScript_DeclareSymbol(set);
	AScript_DeclareSymbol(xset);
	AScript_DeclareSymbol(rev);
	AScript_DeclareSymbol(and);
	AScript_DeclareSymbol(or);
	AScript_DeclareSymbol(xor);
	AScript_DeclareSymbol(void_);
	AScript_DeclareSymbol(index);
	AScript_DeclareSymbol(icase);
	AScript_DeclareSymbol(last_index);
	AScript_DeclareSymbol(indices);
	AScript_DeclareSymbol(ascend);
	AScript_DeclareSymbol(descend);
	AScript_DeclareSymbol(stable);
	AScript_DeclareSymbol(up);
	AScript_DeclareSymbol(down);
	AScript_DeclareSymbol(left);
	AScript_DeclareSymbol(right);
	AScript_DeclareSymbol(prev);
	AScript_DeclareSymbol(next);
	AScript_DeclareSymbol(name);
	AScript_DeclareSymbol(parent);
	AScript_DeclareSymbol(parents);
	AScript_DeclareSymbol(sibling);
	AScript_DeclareSymbol(siblings);
	AScript_DeclareSymbol(child);
	AScript_DeclareSymbol(children);
	AScript_DeclareSymbol(path);
	AScript_DeclareSymbol(x);
	AScript_DeclareSymbol(y);
	AScript_DeclareSymbol(z);
private:
	typedef std::set<Symbol *, Symbol::KeyCompare_Name> Content;
private:
	static SymbolPool *_pInst;
	Symbol::UniqNumber _uniqNum;
	Content _content;
private:
	inline SymbolPool() {}
public:
	~SymbolPool();
	static void Initialize();
	inline static void SetInstance(SymbolPool *pInst) { _pInst = pInst; }
	inline static SymbolPool *GetInstance() { return _pInst; }
	const Symbol *Add(const TCHAR *name);
private:
	void _Initialize();
private:
	inline SymbolPool(const SymbolPool &symbolPool) {}
	inline void operator=(const SymbolPool &symbolPool) {}
};

//-----------------------------------------------------------------------------
// Miscellaneous
//-----------------------------------------------------------------------------
void SplitPathList(Environment &env, const TCHAR *str, ValueList &valList);
Value FindString(Environment &env, Signal sig,
		const TCHAR *str, const TCHAR *sub, int start, const SymbolSet &attrs);

}

#endif
