#ifndef __CODEC_H__
#define __CODEC_H__

#include "Common.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Codec
//-----------------------------------------------------------------------------
class Codec {
protected:
	int _idxBuff;
	char _buffOut[8];
public:
	inline Codec() : _idxBuff(0) {}
	bool FollowChar(char &chConv);
	void Convert(String &dst, const char *src);
	virtual bool FeedChar(char ch, char &chConv) = 0;
protected:
	inline void StoreChar(char ch) { _buffOut[_idxBuff++] = ch; }
};

//-----------------------------------------------------------------------------
// CodecFactory
//-----------------------------------------------------------------------------
class CodecFactory {
public:
	typedef std::vector<CodecFactory *> List;
private:
	static List _list;
	String _name;
public:
	CodecFactory(const char *name);
	inline const char *GetName() { return _name.c_str(); }
	virtual Codec *CreateEncoder() = 0;
	virtual Codec *CreateDecoder() = 0;
	static CodecFactory *Lookup(const char *name);
};

//-----------------------------------------------------------------------------
// Codec_None
//-----------------------------------------------------------------------------
class Codec_None : public Codec {
public:
	Codec_None();
	virtual bool FeedChar(char ch, char &chConv);
};

//-----------------------------------------------------------------------------
// Codec_Encoder
//-----------------------------------------------------------------------------
class Codec_Encoder : public Codec {
protected:
	int _cntChars;
	unsigned long _codeUTF32;
public:
	Codec_Encoder();
	inline unsigned long GetUTF32() const { return _codeUTF32; }
	virtual bool FeedChar(char ch, char &chConv);
	virtual char FeedUTF32(unsigned long codeUTF32) = 0;
};

//-----------------------------------------------------------------------------
// Codec_Decoder
//-----------------------------------------------------------------------------
class Codec_Decoder : public Codec {
public:
	char FeedUTF32(unsigned long codeUTF32);
};

//-----------------------------------------------------------------------------
// Codec_Encoder_UTF16LE
//-----------------------------------------------------------------------------
class Codec_Encoder_UTF16LE : public Codec_Encoder {
public:
	Codec_Encoder_UTF16LE();
	virtual char FeedUTF32(unsigned long codeUTF32);
};

//-----------------------------------------------------------------------------
// Codec_Decoder_UTF16LE
//-----------------------------------------------------------------------------
class Codec_Decoder_UTF16LE : public Codec_Decoder {
private:
	bool _firstByteFlag;
	unsigned long _codeUTF32;
public:
	Codec_Decoder_UTF16LE();
	virtual bool FeedChar(char ch, char &chConv);
};

//-----------------------------------------------------------------------------
// Codec_Encoder_CP932
//-----------------------------------------------------------------------------
class Codec_Encoder_CP932 : public Codec_Encoder {
public:
	typedef std::map<unsigned short, unsigned short> Map;
private:
	static Map *_pMap;
public:
	Codec_Encoder_CP932();
	virtual char FeedUTF32(unsigned long codeUTF32);
	static void PrepareMap(Map *pMap);
};

//-----------------------------------------------------------------------------
// Codec_Decoder_CP932
//-----------------------------------------------------------------------------
class Codec_Decoder_CP932 : public Codec_Decoder {
private:
	unsigned short _codeCP932;
public:
	Codec_Decoder_CP932();
	virtual bool FeedChar(char ch, char &chConv);
};

}

//-----------------------------------------------------------------------------
// macros
//-----------------------------------------------------------------------------
#define AScript_RegisterCodecFactory(symbol, dispname, encoderClass, decoderClass) \
class CodecFactory_##symbol : public CodecFactory {										\
public:																					\
	inline CodecFactory_##symbol(const char *name = (dispname)) : CodecFactory(name) {}	\
	virtual Codec *CreateEncoder();														\
	virtual Codec *CreateDecoder();														\
};																						\
CodecFactory_##symbol _codecFactory_##symbol;											\
Codec *CodecFactory_##symbol::CreateEncoder() { return new encoderClass(); }			\
Codec *CodecFactory_##symbol::CreateDecoder() { return new decoderClass(); }

#endif
