#ifndef __ITERATOR_H__
#define __ITERATOR_H__

#include "Environment.h"
#include "Signal.h"
#include "Value.h"
#include "OAL.h"

namespace AScript {

class Expr_Caller;
class Context;

//-----------------------------------------------------------------------------
// Iterator
//-----------------------------------------------------------------------------
class Iterator {
private:
	int _cntRef;
	bool _infiniteFlag;
public:
	inline Iterator(bool infiniteFlag) : _cntRef(1), _infiniteFlag(infiniteFlag) {}
	inline Iterator *IncRef() { _cntRef++; return this; }
	inline int DecRef() { _cntRef--; return _cntRef; }
	inline int GetRefCnt() const { return _cntRef; }
	inline static void Delete(Iterator *pIterator) {
		if (pIterator != NULL && pIterator->DecRef() <= 0) delete pIterator;
	}
	inline bool IsInfinite() const { return _infiniteFlag; }
	virtual bool Next(Signal sig, Value &value) = 0;
	virtual Value Eval(Environment &env, Signal sig, Context &context);
	virtual Value Eval(Environment &envBlock, Signal sig, Context &context,
									const Function *pFuncBlock);
	Value MinMax(Environment &env, Signal sig,
									bool maxFlag, const SymbolSet &attrs);
protected:
	virtual ~Iterator();
private:
	inline Iterator(const Iterator &iterator) {}
	inline void operator=(const Iterator &iterator) {}
};

//-----------------------------------------------------------------------------
// IteratorOwner
//-----------------------------------------------------------------------------
class IteratorOwner : public std::vector<Iterator *> {
public:
	virtual ~IteratorOwner();
	bool Next(Signal sig, ValueList &valList);
	bool IsInfinite() const;
};

//-----------------------------------------------------------------------------
// Iterator_Constant
//-----------------------------------------------------------------------------
class Iterator_Constant : public Iterator {
private:
	Value _value;
public:
	inline Iterator_Constant(const Value &value) : Iterator(true), _value(value) {}
	~Iterator_Constant();
	virtual bool Next(Signal sig, Value &value);
};

//-----------------------------------------------------------------------------
// Iterator_Mapping
//-----------------------------------------------------------------------------
class Iterator_Mapping : public Iterator {
private:
	Environment _env;
	Function *_pFunc;
	Value _valueSelf;
	IteratorOwner _iterOwner;
public:
	Iterator_Mapping(Environment &env, Signal sig,
									Function *pFunc, Context &context);
	~Iterator_Mapping();
	virtual bool Next(Signal sig, Value &value);
};

//-----------------------------------------------------------------------------
// Iterator_MemberMapping
//-----------------------------------------------------------------------------
class Iterator_MemberMapping : public Iterator {
private:
	Iterator *_pIterator;
	Expr *_pExpr;
public:
	inline Iterator_MemberMapping(Iterator *pIterator, Expr *pExpr) :
		Iterator(pIterator->IsInfinite()), _pIterator(pIterator), _pExpr(pExpr) {}
	~Iterator_MemberMapping();
	virtual bool Next(Signal sig, Value &value);
};

//-----------------------------------------------------------------------------
// Iterator_MethodMapping
//-----------------------------------------------------------------------------
class Iterator_MethodMapping : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	Expr_Caller *_pExprCaller;
public:
	inline Iterator_MethodMapping(Environment &env, Iterator *pIterator, Expr_Caller *pExprCaller) :
		Iterator(pIterator->IsInfinite()), _env(env), _pIterator(pIterator), _pExprCaller(pExprCaller) {}
	~Iterator_MethodMapping();
	virtual bool Next(Signal sig, Value &value);
};

//-----------------------------------------------------------------------------
// Iterator_Fork
//-----------------------------------------------------------------------------
class Iterator_Fork : public Iterator, public OAL::Thread {
private:
	Environment _env;
	Function *_pFunc;
	IteratorOwner _iterOwner;
	ValueList _valListA;
	ValueList _valListB;
	ValueList *_pValListToWrite;
	ValueList *_pValListToRead;
	ValueList::iterator _pValueRead;
	OAL::Semaphore _semaphore;
	bool _doneFlag;
	struct {
		bool blockedFlag;
		OAL::Event event;
	} _readBlock;
	struct {
		bool blockedFlag;
		OAL::Event event;
	} _writeBlock;
	public:
	Iterator_Fork(Environment &env, Signal sig,
									Function *pFunc, Context &context);
	~Iterator_Fork();
	virtual bool Next(Signal sig, Value &value);
	virtual void Run();
	void SwapList();
	void ForkProcess();
};

//-----------------------------------------------------------------------------
// Iterator_Delay
//-----------------------------------------------------------------------------
class Iterator_Delay : public Iterator {
private:
	Iterator *_pIterator;
	Number _delay;
public:
	inline Iterator_Delay(Iterator *pIterator, Number delay) : 
		Iterator(pIterator->IsInfinite()), _pIterator(pIterator), _delay(delay) {}
	~Iterator_Delay();
	virtual bool Next(Signal sig, Value &value);
};

//-----------------------------------------------------------------------------
// Iterator_Skip
//-----------------------------------------------------------------------------
class Iterator_Skip : public Iterator {
private:
	Iterator *_pIterator;
	int _nSkip;
public:
	inline Iterator_Skip(Iterator *pIterator, int nSkip) :
		Iterator(pIterator->IsInfinite()), _pIterator(pIterator), _nSkip(nSkip) {}
	~Iterator_Skip();
	virtual bool Next(Signal sig, Value &value);
};

//-----------------------------------------------------------------------------
// Iterator_SkipInvalid
//-----------------------------------------------------------------------------
class Iterator_SkipInvalid : public Iterator {
private:
	Iterator *_pIterator;
public:
	inline Iterator_SkipInvalid(Iterator *pIterator) :
				Iterator(pIterator->IsInfinite()), _pIterator(pIterator) {}
	~Iterator_SkipInvalid();
	virtual bool Next(Signal sig, Value &value);
};

//-----------------------------------------------------------------------------
// Iterator_FilterByFunction
//-----------------------------------------------------------------------------
class Iterator_FilterByFunction : public Iterator {
private:
	Environment _env;
	Iterator *_pIterator;
	Object_Function *_pObjFunc;
public:
	inline Iterator_FilterByFunction(Environment &env,
							Iterator *pIterator, Object_Function *pObjFunc) :
			Iterator(pIterator->IsInfinite()), _env(env),
			_pIterator(pIterator), _pObjFunc(pObjFunc) {}
	~Iterator_FilterByFunction();
	virtual bool Next(Signal sig, Value &value);
};

//-----------------------------------------------------------------------------
// Iterator_FilterByIterator
//-----------------------------------------------------------------------------
class Iterator_FilterByIterator : public Iterator {
private:
	Iterator *_pIterator;
	Iterator *_pIteratorCriteria;
public:
	inline Iterator_FilterByIterator(Iterator *pIterator, Iterator *pIteratorCriteria) :
			Iterator(pIterator->IsInfinite()),
			_pIterator(pIterator), _pIteratorCriteria(pIteratorCriteria) {}
	~Iterator_FilterByIterator();
	virtual bool Next(Signal sig, Value &value);
};

//-----------------------------------------------------------------------------
// Iterator_Align
//-----------------------------------------------------------------------------
class Iterator_Align : public Iterator {
private:
	Iterator *_pIterator;
	int _cnt;
	Value _valueFill;
public:
	inline Iterator_Align(Iterator *pIterator, int cnt, const Value &valueFill) :
		Iterator(false), _pIterator(pIterator), _cnt(cnt), _valueFill(valueFill) {}
	virtual ~Iterator_Align();
	virtual bool Next(Signal sig, Value &value);
};

}

#endif
