// OAL: OS Abstraction Layer
#include "OAL.h"

namespace AScript {
namespace OAL {

#if defined(HAVE_WINDOWS_H)

//=============================================================================
// Windows API
//=============================================================================
void rename(const char *src, const char *dst)
{
	::MoveFileEx(src, dst, MOVEFILE_REPLACE_EXISTING);
}

void remove(const char *pathName)
{
	::DeleteFile(pathName);
}

void Sleep(Number delay)	// unit: sec
{
	::Sleep(static_cast<long>(delay * 1000));	// unit: msec
}

//-----------------------------------------------------------------------------
// Thread
//-----------------------------------------------------------------------------
static DWORD WINAPI ThreadProc(LPVOID lpParameter)
{
	reinterpret_cast<Thread *>(lpParameter)->Run();
	return 0;
}

void Thread::Start()
{
	DWORD threadId;
	::CreateThread(NULL, 0, ThreadProc, this, 0, &threadId);
}

//-----------------------------------------------------------------------------
// Semaphore
//-----------------------------------------------------------------------------
Semaphore::Semaphore()
{
	_hMutex = ::CreateMutex(NULL, FALSE, NULL);
}

Semaphore::~Semaphore()
{
	::DeleteObject(_hMutex);
}

void Semaphore::Wait()
{
	::WaitForSingleObject(_hMutex, INFINITE);
}

void Semaphore::Release()
{
	::ReleaseMutex(_hMutex);
}

//-----------------------------------------------------------------------------
// Event
//-----------------------------------------------------------------------------
Event::Event()
{
	_hEvent = ::CreateEvent(NULL, FALSE, FALSE, NULL);
}

Event::~Event()
{
	::DeleteObject(_hEvent);
}

void Event::Wait()
{
	::WaitForSingleObject(_hEvent, INFINITE);
}

void Event::Notify()
{
	::SetEvent(_hEvent);
}

#else
//=============================================================================
// POSIX
//=============================================================================
void rename(const char *src, const char *dst)
{
	::rename(src, dst);
}

void remove(const char *pathName)
{
	::unlink(pathName);
}

void Sleep(Number delay)	// unit: sec
{
	::usleep(static_cast<unsigned long>(delay * 1000000));	// unit: usec
}

//-----------------------------------------------------------------------------
// Thread
//-----------------------------------------------------------------------------
void Thread::Start()
{
}

//-----------------------------------------------------------------------------
// Semaphore
//-----------------------------------------------------------------------------
#if defined(HAVE_SEMAPHORE_H)
// should use mutex instead?
Semaphore::Semaphore()
{
	::sem_init(&_sem, 0, 1);
}

Semaphore::~Semaphore()
{
	::sem_destroy(&_sem);
}

void Semaphore::Wait()
{
	::sem_wait(&_sem);
}

void Semaphore::Release()
{
	::sem_post(&_sem);
}
#else
#endif

//-----------------------------------------------------------------------------
// Event
//-----------------------------------------------------------------------------
#if defined(HAVE_SEMAPHORE_H)
Event::Event()
{
}

Event::~Event()
{
}

void Event::Wait()
{
}

void Event::Notify()
{
}
#else
#endif

#endif

}
}
