//
// Object_Function
//

#include "Object_Function.h"
#include "Expr.h"
#include "Module.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Object_Function
//-----------------------------------------------------------------------------
bool Object_Function::IsFunction() const { return true; }

Object_Function::~Object_Function()
{
	Function::Delete(_pFunc);
}

Value Object_Function::Call(Environment &env, Signal sig, ContextExpr &contextExpr)
{
	return _pFunc->EvalExpr(env, sig, contextExpr);
}

Value Object_Function::Eval(Environment &env, Signal sig, Context &context) const
{
	return _pFunc->Eval(env, sig, context);
}

Object *Object_Function::Clone() const
{
	return new Object_Function(*this);
}

String Object_Function::ToString(Signal sig, bool exprFlag)
{
	return _pFunc->ToString(sig);
}

//-----------------------------------------------------------------------------
// Object_WrappedMethod
//-----------------------------------------------------------------------------
bool Object_WrappedMethod::IsWrappedMethod() const { return true; }

Object_WrappedMethod::~Object_WrappedMethod()
{
}

Value Object_WrappedMethod::Call(Environment &env, Signal sig, ContextExpr &contextExpr)
{
	contextExpr.SetSelf(_valueSelf);
	return _pFunc->EvalExpr(env, sig, contextExpr);
}

Value Object_WrappedMethod::Eval(Environment &env, Signal sig, Context &context) const
{
	context.SetSelf(_valueSelf);
	return _pFunc->Eval(env, sig, context);
}

Object *Object_WrappedMethod::Clone() const
{
	return new Object_WrappedMethod(*this);
}

String Object_WrappedMethod::ToString(Signal sig, bool exprFlag)
{
	String str;
	ObjectBase *pObj = _valueSelf.ExtractObject(sig);
	if (sig.IsSignalled()) return str;
	if (pObj->IsModule()) {
		str += dynamic_cast<Module *>(pObj)->GetName();
		str += ".";
	} else if (pObj->IsClass()) {
		str += dynamic_cast<Class *>(pObj)->GetName();
		str += ".";
	} else if (pObj->IsObject()) {
		str += dynamic_cast<Object *>(pObj)->GetClass()->GetName();
		str += "#";
	}
	str += _pFunc->ToString(sig);
	return str;
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Function
//-----------------------------------------------------------------------------
// func = Function(`args*) {block}
Object_Function::Constructor::Constructor(Environment &env, const char *name) :
							ConstructorBase(name, env.GetClass_Function())
{
	DeclareArg(env, "args", VTYPE_Quote, false, OCCUR_ZeroOrMore);
	DeclareBlock(OCCUR_Once);
}

Value Object_Function::Constructor::DoEval(Environment &env, Signal sig, Context &context) const
{
	const Expr *pExprBlock = context.GetBlock();
	ExprList exprArgs;
	foreach_const (ValueList, pValue, context.GetList(0)) {
		exprArgs.push_back(const_cast<Expr *>(pValue->GetExpr()));
	}
	if (exprArgs.empty()) {
		ExprVisitorEx visitor(exprArgs);
		pExprBlock->Accept(visitor);
	}
	FunctionCustom *pFunc = new FunctionCustom(env, AScript_Symbol(_anonymous_),
							pExprBlock->IncRef(), NULL, FUNCTYPE_Function);
	ContextExpr contextExpr(context.GetAttrs(), SymbolSet::Null,
									NULL, NULL, Value::Null, exprArgs);
	if (!pFunc->CustomDeclare(env, sig, SymbolSet::Null, contextExpr)) {
		Function::Delete(pFunc);
		return Value::Null;
	}
	Value result;
	result.InitAsFunction(env, pFunc);
	return result;
}

bool Object_Function::ExprVisitorEx::Visit(const Expr *pExpr)
{
	if (pExpr->IsCaller()) {
		// avoid searching in a simple lambda inside
		const Expr *pExprCar =
					dynamic_cast<const Expr_Caller *>(pExpr)->GetCar();
		if (pExprCar->IsSymbol()) {
			const Symbol *pSymbol =
					dynamic_cast<const Expr_Symbol *>(pExprCar)->GetSymbol();
			if (::strcmp(pSymbol->GetName(), "@") == 0) return false;
		}
		
	} else if (pExpr->IsSymbol()) {
		const Symbol *pSymbol =
						dynamic_cast<const Expr_Symbol *>(pExpr)->GetSymbol();
		if (pSymbol->GetName()[0] == '$' &&
								_symbolSet.find(pSymbol) == _symbolSet.end()) {
			_exprList.push_back(const_cast<Expr *>(pExpr));
			_symbolSet.insert(pSymbol);
		}
	}
	return true;
}

// Function#name()
AScript_DeclareMethod(Function, name)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Function, name)
{
	Object_Function *pSelf = Object_Function::GetSelfObj(context);
	return Value(env, pSelf->GetFunction()->GetName());
}

// Function#expr()
AScript_DeclareMethod(Function, expr)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Function, expr)
{
	Object *pObj = context.GetSelfObj();
	Value value;
	const Function *pFunc = dynamic_cast<Object_Function *>(pObj)->GetFunction();
	if (!pFunc->IsCustom()) {
		sig.SetError(ERR_ValueError, "function '%s' is a built-in function", pFunc->GetName());
		return Value::Null;
	}
	const FunctionCustom *pFuncCustom = dynamic_cast<const FunctionCustom *>(pFunc);
	value.InitAsExpr(env, pFuncCustom->GetExprBody()->IncRef());
	return value;
}

// Assignment
Class_Function::Class_Function(Class *pClassSuper, const Symbol *pSymbol) :
												Class(pClassSuper, pSymbol)
{
	// don't assign functions here, because the function objects themselves
	// shall be constructed here!!!!!!! instead, they must be assigned in
	// Prepare() funtion below.
}

void Class_Function::Prepare()
{
	AScript_AssignMethod(Function, expr);
	AScript_AssignMethod(Function, name);
}

Object *Class_Function::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	ERROREND(env, "this function must not be called");
	return NULL;
}

}
