//
// Object_Iterator
//

#include "Object_Iterator.h"
#include "Object_Function.h"
#include "Expr.h"

namespace AScript {

//-----------------------------------------------------------------------------
// Object_Iterator
//-----------------------------------------------------------------------------
Object_Iterator::Object_Iterator(const Object_Iterator &obj) :
							Object(obj), _pIterator(obj._pIterator->IncRef())
{
}

Object_Iterator::~Object_Iterator()
{
	Iterator::Delete(_pIterator);
}

Object *Object_Iterator::Clone() const
{
	return new Object_Iterator(*this);
}

Iterator *Object_Iterator::CreateIterator(Signal sig)
{
	return _pIterator->IncRef();
}

String Object_Iterator::ToString(Signal sig, bool exprFlag)
{
	String str;
	str += "<Iterator";
	str += ">";
	return str;
}

//-----------------------------------------------------------------------------
// AScript interfaces for Object_Iterator
//-----------------------------------------------------------------------------
// Iterator#is_infinite()
AScript_DeclareMethod(Iterator, is_infinite)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Iterator, is_infinite)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	return Value(pSelf->GetIterator()->IsInfinite());
}

// Iterator#next()
AScript_DeclareMethod(Iterator, next)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Iterator, next)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Value value;
	if (pSelf->GetIterator()->Next(sig, value)) return value;
	return Value::Null;
}

// Iterator#len()
AScript_DeclareMethod(Iterator, len)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Iterator, len)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Value value;
	int cnt;
	for (cnt = 0; pSelf->GetIterator()->Next(sig, value); cnt++) ;
	return Value(static_cast<Number>(cnt));
}

// Iterator#min():[index,last_index,indices]
AScript_DeclareMethod(Iterator, min)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(index));
	DeclareAttr(AScript_Symbol(last_index));
	DeclareAttr(AScript_Symbol(indices));
}

AScript_ImplementMethod(Iterator, min)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	return pSelf->GetIterator()->MinMax(env, sig, false, context.GetAttrs());
}

// Iterator#max():[index,last_index,indices]
AScript_DeclareMethod(Iterator, max)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareAttr(AScript_Symbol(index));
	DeclareAttr(AScript_Symbol(last_index));
	DeclareAttr(AScript_Symbol(indices));
}

AScript_ImplementMethod(Iterator, max)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	return pSelf->GetIterator()->MinMax(env, sig, true, context.GetAttrs());
}

// Iterator#filter(criteria:Function) {block?}
AScript_DeclareMethod(Iterator, filter)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "criteria", VTYPE_Function);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, filter)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = new Iterator_FilterByFunction(env,
			pSelf->GetIterator()->IncRef(),
			dynamic_cast<Object_Function *>(context.GetFunctionObj(0)->IncRef()));
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#eval()
AScript_DeclareMethod(Iterator, eval)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
}

AScript_ImplementMethod(Iterator, eval)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = pSelf->GetIterator();
	return pIterator->Eval(env, sig, context);
}

// Iterator#each() {block}
AScript_DeclareMethod(Iterator, each)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_Once);
}

AScript_ImplementMethod(Iterator, each)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = pSelf->GetIterator();
	Environment envBlock(&env, ENVTYPE_Block);
	const Function *pFuncBlock = GetBlockFunction(envBlock, sig, context);
	if (pFuncBlock == NULL) return Value::Null;
	return pIterator->Eval(env, sig, context, pFuncBlock);
}

// List#join(sep?:string)
AScript_DeclareMethod(Iterator, join)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "sep", VTYPE_String, false, OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, join)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	const char *sep = context.IsString(0)? context.GetString(0) : "";
	Iterator *pIterator = pSelf->GetIterator();
	String rtn;
	Value value;
	if (pIterator->Next(sig, value)) {
		rtn += value.ToString(sig, false);
		if (sig.IsSignalled()) return Value::Null;
		while (pIterator->Next(sig, value)) {
			rtn += sep;
			rtn += value.ToString(sig, false);
			if (sig.IsSignalled()) return Value::Null;
		}
	}
	if (sig.IsSignalled()) return Value::Null;
	return Value(env, rtn.c_str());
}

// Iterator#delay(delay:number) {block?}
AScript_DeclareMethod(Iterator, delay)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "delay", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, delay)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator =
		new Iterator_Delay(pSelf->GetIterator()->IncRef(), context.GetNumber(0));
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#skip(n:number) {block?}
AScript_DeclareMethod(Iterator, skip)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, skip)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator =
		new Iterator_Skip(pSelf->GetIterator()->IncRef(), static_cast<int>(context.GetNumber(0)));
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#skip_nil() {block?}
AScript_DeclareMethod(Iterator, skip_nil)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, skip_nil)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = new Iterator_SkipInvalid(pSelf->GetIterator()->IncRef());
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#align(n:number, value?) {block?}
AScript_DeclareMethod(Iterator, align)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareArg(env, "value", VTYPE_AnyType, false, OCCUR_ZeroOrOnce);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, align)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = new Iterator_Align(pSelf->GetIterator()->IncRef(),
			context.IsNumber(0)? static_cast<int>(context.GetNumber(0)) : -1,
			context.GetValue(1));
	return ReturnIterator(env, sig, context, pIterator);
}

// Iterator#offset(n:number) {block?}
AScript_DeclareMethod(Iterator, offset)
{
	SetMode(RSLTMODE_Normal, MAP_Off, FLAT_Off);
	DeclareArg(env, "n", VTYPE_Number);
	DeclareBlock(OCCUR_ZeroOrOnce);
}

AScript_ImplementMethod(Iterator, offset)
{
	Object_Iterator *pSelf = Object_Iterator::GetSelfObj(context);
	Iterator *pIterator = pSelf->GetIterator()->IncRef();
	for (int cnt = static_cast<int>(context.GetNumber(0)); cnt > 0; cnt--) {
		Value value;
		pIterator->Next(sig, value);
	}
	return ReturnIterator(env, sig, context, pIterator);
}

// assignment
Class_Iterator::Class_Iterator(Class *pClassSuper, const Symbol *pSymbol) :
												Class(pClassSuper, pSymbol)
{
	AScript_AssignMethod(Iterator, is_infinite);
	AScript_AssignMethod(Iterator, next);
	AScript_AssignMethod(Iterator, len);
	AScript_AssignMethod(Iterator, min);
	AScript_AssignMethod(Iterator, max);
	AScript_AssignMethod(Iterator, filter);
	AScript_AssignMethod(Iterator, eval);
	AScript_AssignMethod(Iterator, each);
	AScript_AssignMethod(Iterator, join);
	AScript_AssignMethod(Iterator, delay);
	AScript_AssignMethod(Iterator, skip);
	AScript_AssignMethod(Iterator, skip_nil);
	AScript_AssignMethod(Iterator, align);
	AScript_AssignMethod(Iterator, offset);
}

Object *Class_Iterator::CreateDescendant(Environment &env, Signal sig, Class *pClass)
{
	ERROREND(env, "this function must not be called");
	return NULL;
}

}
