#ifndef __SIGNAL_H__
#define __SIGNAL_H__

#include "Common.h"

namespace AScript {

class Expr;
class ExprList;
class Value;

//-----------------------------------------------------------------------------
// ErrorType
//-----------------------------------------------------------------------------
enum ErrorType {
	ERR_None,
	ERR_SyntaxError,
	ERR_ArithmeticError,
	ERR_TypeError,
	ERR_ZeroDivisionError,
	ERR_ValueError,
	ERR_SystemError,
	ERR_IOError,
	ERR_IndexError,
	ERR_KeyError,
	ERR_ImportError,
	ERR_AttributeError,
	ERR_StopIteration,
	ERR_RuntimeError,
	ERR_NameError,
	ERR_NotImplementedError,
	ERR_IteratorError,
	ERR_CodecError,
	ERR_CommandError,
};

const char *GetErrorTypeName(ErrorType errType);

//-----------------------------------------------------------------------------
// SignalType
//-----------------------------------------------------------------------------
enum SignalType {
	SIGTYPE_None,
	SIGTYPE_Error,
	SIGTYPE_Break,
	SIGTYPE_Continue,
	SIGTYPE_Return,
	SIGTYPE_DetectEncoding,
};

const char *GetSignalTypeName(SignalType sigType);

//-----------------------------------------------------------------------------
// Signal
//-----------------------------------------------------------------------------
class Signal {
private:
	struct Message {
		SignalType sigType;
		ErrorType errType;
		ErrorType errTypeSaved;
		String str;
		String strSaved;
		String fileName;
		String fileNameSaved;
		std::auto_ptr<Value> pValue;
		std::auto_ptr<ExprList> pExprCauseList;
		Message();
	};
private:
	Message *_pMsg;
	int _stackLevel;
public:
	Signal();
	Signal(const Signal &sig);
	inline bool IsSignalled() const	{ return _pMsg->sigType != SIGTYPE_None; }
	inline bool IsError() const		{ return _pMsg->sigType == SIGTYPE_Error; }
	inline bool IsBreak() const		{ return _pMsg->sigType == SIGTYPE_Break; }
	inline bool IsContinue() const	{ return _pMsg->sigType == SIGTYPE_Continue; }
	inline bool IsReturn() const	{ return _pMsg->sigType == SIGTYPE_Return; }
	inline SignalType GetSignalType() const { return _pMsg->sigType; }
	inline const char *GetSignalName() const { return GetSignalTypeName(_pMsg->sigType); }
	inline ErrorType GetErrorType() const { return _pMsg->errType; }
	inline const char *GetErrorName() const { return GetErrorTypeName(_pMsg->errType); }
	inline const char *GetSavedString() const { return _pMsg->strSaved.c_str(); }
	inline Value &GetValue() const { return *_pMsg->pValue; }
	void ClearSignal();
	void SetSignal(SignalType sigType, const Value &value);
	inline void SaveError() {
		_pMsg->errTypeSaved = _pMsg->errType, _pMsg->strSaved = _pMsg->str;
		_pMsg->fileNameSaved = _pMsg->fileName;
	}
	inline void RestoreError() {
		_pMsg->errType = _pMsg->errTypeSaved, _pMsg->str = _pMsg->strSaved;
		_pMsg->fileName = _pMsg->fileNameSaved;
	}
	void AddExprCause(const Expr *pExpr);
	inline const ExprList &GetExprCauseList() const {
		return *_pMsg->pExprCauseList;
	}
	void SetFileName(const char *fileName) { _pMsg->fileName = fileName; }
	const char *GetFileName() const { return _pMsg->fileName.c_str(); }
	String GetErrString() const;
	inline ErrorType GetSavedErrorType() const { return _pMsg->errTypeSaved; }
	void SetError(ErrorType errType, const char *format, ...);
	void SetErrorV(ErrorType errType,
			const char *format, va_list list, const char *textPre = "");
};

}

#endif
